/*
 * $Header: /home/cvs/jakarta-commons/collections/src/java/org/apache/commons/collections/FastTreeMap.java,v 1.5 2002/02/10 08:07:42 jstrachan Exp $
 * $Revision: 1.5 $
 * $Date: 2002/02/10 08:07:42 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */


package org.apache.commons.collections;


import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Collection;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;


/**
 * <p>A customized implementation of <code>java.util.TreeMap</code> designed
 * to operate in a multithreaded environment where the large majority of
 * method calls are read-only, instead of structural changes.  
 *
 * <p>A customized implementation of <code>java.util.ArrayList</code> designed
 * to operate in a multithreaded environment where the large majority of
 * method calls are read-only, instead of structural changes.
 *
 *  <P>This class can operate in two modes, fast mode and slow mode.  In
 *  slow mode, all method invocations are synchronized.  This is appropriate
 *  for initially populating the collection.
 *
 *  <P>Previous releases of this class provided unsychronized read access
 *  in fast mode.  Write operations would clone an internal map, update
 *  the clone, then reassign the internal field to the clone.  Read operations
 *  would simply obtain the most recent reference and use that.
 *
 *  <P>However, that technique does not work on some platforms, particularly 
 *  multiprocessor architectures.  Even on a single processor, however, an
 *  optimizing compiler or a pipeline could alter the order of operations, so
 *  that the reference would be reassigned to the clone before the clone was
 *  fully updated. 
 *
 *  <P>This release doesn't guarantee unsynchronized read access, but offers
 *  the next best thing:  The critical section of a read operation is the shortest
 *  imaginable.  A read operation is synchronized only long enough to fetch the
 *  reference to the internal map.  (Although on some platforms, you may
 *  be able to safely avoid the synchronization on reads.  See {@link SafeReferenceFactory}
 *  for more information.)
 *
 *  <P>Write operations are still fully synchronized, and still clone the 
 *  internal map.  Only one write can occur at a time; multiple reads can
 *  be in progress at the same time.
 *
 *  <P>Sublists produced by this class are also thread-safe, and also provide
 *  fast read access but slow write access.
 *
 *  <P>Note that the iterators produced by this class are <I>not</I> fail-fast:
 *  Read operations on an iterator <I>never</I> fail.  A {@link ConcurrentModificationException}
 *  will only be raised if the original list is modified after the iterator is 
 *  created, and the iterator then attempts to modify the list.
 *
 * <p><strong>NOTE</strong>: If you are creating and accessing a
 * <code>TreeMap</code> only within a single thread, you should use
 * <code>java.util.TreeMap</code> directly (with no synchronization), for
 * maximum performance.</p>
 *
 * @author Craig R. McClanahan, Paul Jack
 * @version $Revision: 1.5 $ $Date: 2002/02/10 08:07:42 $
 */

public class FastTreeMap extends TreeMap {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a an empty map.
     */
    public FastTreeMap() {

        this(new TreeMap());

    }


    /**
     * Construct an empty map with the specified comparator.
     *
     * @param comparator The comparator to use for ordering tree elements
     */
    public FastTreeMap(Comparator comparator) {

        this(new TreeMap(comparator));

    }


    /**
     * Construct a new map with the same mappings as the specified map,
     * sorted according to the keys's natural order
     *
     * @param map The map whose mappings are to be copied
     */
    public FastTreeMap(Map map) {

        this(new TreeMap(map));

    }


    /**
     * Construct a new map with the same mappings as the specified map,
     * sorted according to the same ordering
     *
     * @param map The map whose mappings are to be copied
     */
    public FastTreeMap(SortedMap map) {

        this(new TreeMap(map));

    }


    private FastTreeMap(TreeMap map) {
        super();
        this.map = map;
        this.mapRef = SafeReference.create(map);
    }

    // ----------------------------------------------------- Instance Variables


    /**
     * The underlying map we are managing.
     *
     *  @deprecated  use the getMap() and setMap(TreeMap) methods instead.
     */
    private transient TreeMap map = null;


    transient SafeReference mapRef;

    // ------------------------------------------------------------- Properties


    /**
     * Are we operating in "fast" mode?
     */
    protected boolean fast = false;

    public boolean getFast() {
        return (this.fast);
    }

    public void setFast(boolean fast) {
        this.fast = fast;
    }


    protected TreeMap getMap() {
        return (TreeMap)mapRef.get();
    }


    protected void setMap(TreeMap map) {
        mapRef.set(map);
    }


    // --------------------------------------------------------- Public Methods


    /**
     * Remove all mappings from this map.
     */
    public void clear() {

        if (fast) {
            synchronized (this) {
                TreeMap temp = (TreeMap) getMap().clone();
                temp.clear();
                setMap(temp);
            }
        } else {
            synchronized (map) {
                map.clear();
            }
        }

    }


    /**
     * Return a shallow copy of this <code>FastTreeMap</code> instance.
     * The keys and values themselves are not copied.
     */
    public Object clone() {

        FastTreeMap results = null;
        if (fast) {
            results = new FastTreeMap(map);
        } else {
            synchronized (map) {
                results = new FastTreeMap(map);
            }
        }
        results.setFast(getFast());
        return (results);

    }


    /**
     * Return the comparator used to order this map, or <code>null</code>
     * if this map uses its keys' natural order.
     */
    public Comparator comparator() {

        if (fast) {
            return (getMap().comparator());
        } else {
            synchronized (map) {
                return (map.comparator());
            }
        }

    }


    /**
     * Return <code>true</code> if this map contains a mapping for the
     * specified key.
     *
     * @param key Key to be searched for
     */
    public boolean containsKey(Object key) {

        if (fast) {
            return (getMap().containsKey(key));
        } else {
            synchronized (map) {
                return (map.containsKey(key));
            }
        }

    }


    /**
     * Return <code>true</code> if this map contains one or more keys mapping
     * to the specified value.
     *
     * @param value Value to be searched for
     */
    public boolean containsValue(Object value) {

        if (fast) {
            return (getMap().containsValue(value));
        } else {
            synchronized (map) {
                return (map.containsValue(value));
            }
        }

    }


    /**
     * Return a collection view of the mappings contained in this map.  Each
     * element in the returned collection is a <code>Map.Entry</code>.
     */
    public Set entrySet() {

        if (fast) {
            return new FastMapEntrySet(new FastTreeMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.entrySet());
            }
        }

    }


    /**
     * Compare the specified object with this list for equality.  This
     * implementation uses exactly the code that is used to define the
     * list equals function in the documentation for the
     * <code>Map.equals</code> method.
     *
     * @param o Object to be compared to this list
     */
    public boolean equals(Object o) {

        // Simple tests that require no synchronization
        if (o == this)
            return (true);
        else if (!(o instanceof Map))
            return (false);
        Map mo = (Map) o;

        // Compare the two maps for equality
        if (fast) {
            if (mo.size() != map.size())
                return (false);
            java.util.Iterator i = map.entrySet().iterator();
            while (i.hasNext()) {
                Map.Entry e = (Map.Entry) i.next();
                Object key = e.getKey();
                Object value = e.getValue();
                if (value == null) {
                    if (!(mo.get(key) == null && mo.containsKey(key)))
                        return (false);
                } else {
                    if (!value.equals(mo.get(key)))
                        return (false);
                }
            }
            return (true);
        } else {
            synchronized (map) {
                if (mo.size() != map.size())
                    return (false);
                java.util.Iterator i = map.entrySet().iterator();
                while (i.hasNext()) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (value == null) {
                        if (!(mo.get(key) == null && mo.containsKey(key)))
                            return (false);
                    } else {
                        if (!value.equals(mo.get(key)))
                            return (false);
                    }
                }
                return (true);
            }
        }

    }


    /**
     * Return the first (lowest) key currently in this sorted map.
     */
    public Object firstKey() {

        if (fast) {
            return (getMap().firstKey());
        } else {
            synchronized (map) {
                return (map.firstKey());
            }
        }

    }


    /**
     * Return the value to which this map maps the specified key.  Returns
     * <code>null</code> if the map contains no mapping for this key, or if
     * there is a mapping with a value of <code>null</code>.  Use the
     * <code>containsKey()</code> method to disambiguate these cases.
     *
     * @param key Key whose value is to be returned
     */
    public Object get(Object key) {

        if (fast) {
            return (getMap().get(key));
        } else {
            synchronized (map) {
                return (map.get(key));
            }
        }

    }


    /**
     * Return the hash code value for this map.  This implementation uses
     * exactly the code that is used to define the list hash function in the
     * documentation for the <code>Map.hashCode</code> method.
     */
    public int hashCode() {

        if (fast) {
            int h = 0;
            java.util.Iterator i = map.entrySet().iterator();
            while (i.hasNext())
                h += i.next().hashCode();
            return (h);
        } else {
            synchronized (map) {
                int h = 0;
                java.util.Iterator i = map.entrySet().iterator();
                while (i.hasNext())
                    h += i.next().hashCode();
                return (h);
            }
        }

    }


    /**
     * Return a view of the portion of this map whose keys are strictly
     * less than the specified key.
     *
     * @param key Key higher than any in the returned map
     */
    public SortedMap headMap(Object key) {

        if (fast) {
            return new SubFastTreeMap(this, EXTREMITY, key);
        } else {
            synchronized (map) {
                return (map.headMap(key));
            }
        }

    }


    /**
     * Test if this list has no elements.
     */
    public boolean isEmpty() {

        if (fast) {
            return (getMap().isEmpty());
        } else {
            synchronized (map) {
                return (map.isEmpty());
            }
        }

    }


    /**
     * Return a set view of the keys contained in this map.
     */
    public Set keySet() {

        if (fast) {
            return new FastMapKeySet(new FastTreeMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.keySet());
            }
        }

    }


    /**
     * Return the last (highest) key currently in this sorted map.
     */
    public Object lastKey() {

        if (fast) {
            return (getMap().lastKey());
        } else {
            synchronized (map) {
                return (map.lastKey());
            }
        }

    }


    /**
     * Associate the specified value with the specified key in this map.
     * If the map previously contained a mapping for this key, the old
     * value is replaced and returned.
     *
     * @param key The key with which the value is to be associated
     * @param value The value to be associated with this key
     */
    public Object put(Object key, Object value) {

        if (fast) {
            synchronized (this) {
                TreeMap temp = (TreeMap) getMap().clone();
                Object result = temp.put(key, value);
                setMap(temp);
                return (result);
            }
        } else {
            synchronized (map) {
                return (getMap().put(key, value));
            }
        }

    }


    /**
     * Copy all of the mappings from the specified map to this one, replacing
     * any mappings with the same keys.
     *
     * @param in Map whose mappings are to be copied
     */
    public void putAll(Map in) {

        if (fast) {
            synchronized (this) {
                TreeMap temp = (TreeMap) getMap().clone();
                temp.putAll(in);
                setMap(temp);
            }
        } else {
            synchronized (map) {
                map.putAll(in);
            }
        }

    }


    /**
     * Remove any mapping for this key, and return any previously
     * mapped value.
     *
     * @param key Key whose mapping is to be removed
     */
    public Object remove(Object key) {

        if (fast) {
            synchronized (this) {
                TreeMap temp = (TreeMap) getMap().clone();
                Object result = temp.remove(key);
                setMap(temp);
                return (result);
            }
        } else {
            synchronized (map) {
                return (map.remove(key));
            }
        }

    }


    /**
     * Return the number of key-value mappings in this map.
     */
    public int size() {

        if (fast) {
            return (getMap().size());
        } else {
            synchronized (map) {
                return (map.size());
            }
        }

    }


    /**
     * Return a view of the portion of this map whose keys are in the
     * range fromKey (inclusive) to toKey (exclusive).
     *
     * @param fromKey Lower limit of keys for the returned map
     * @param toKey Upper limit of keys for the returned map
     */
    public SortedMap subMap(Object fromKey, Object toKey) {

        if (fast) {
            TreeMap map = getMap();
            return new SubFastTreeMap(this, fromKey, toKey);
        } else {
            synchronized (map) {
                return (map.subMap(fromKey, toKey));
            }
        }

    }


    /**
     * Return a view of the portion of this map whose keys are greater than
     * or equal to the specified key.
     *
     * @param key Key less than or equal to any in the returned map
     */
    public SortedMap tailMap(Object key) {

        if (fast) {
            return new SubFastTreeMap(this, key, EXTREMITY);
        } else {
            synchronized (map) {
                return (map.tailMap(key));
            }
        }

    }


    /**
     * Return a collection view of the values contained in this map.
     */
    public Collection values() {

        if (fast) {
            return new FastMapValues(new FastTreeMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.values());
            }
        }

    }


    public String toString() {
        return getMap().toString();
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        TreeMap temp = (TreeMap)getMap();
        out.writeObject(temp);
    }


    private void readObject(ObjectInputStream inp) throws IOException, ClassNotFoundException {
        inp.defaultReadObject();
	TreeMap map = (TreeMap)inp.readObject();
        this.mapRef = SafeReference.create(map);
	this.map = map;
    }



    final static Object EXTREMITY = new Object();

}


class SubFastTreeMap implements SortedMap
{

	final Object first;
	final Object last;
	final FastTreeMap fast;


	public SubFastTreeMap(FastTreeMap fast, Object first, Object last)
	{
		this.fast = fast;
		this.first = first;
		this.last = last;
	}


	SortedMap getSubMap()
	{
		return getSubMap(fast.getMap());
	}


        SortedMap getSubMap(SortedMap map)
	{
		if (first == FastTreeMap.EXTREMITY) {
			return map.headMap(last);
		}
		if (last == FastTreeMap.EXTREMITY) {
			return map.tailMap(first);
		}
		return map.subMap(first, last);
	}


	public int size()
	{
		return getSubMap().size();
	}


	public boolean isEmpty()
	{
		return getSubMap().isEmpty();
	}


	public boolean containsKey(Object key)
	{
		return getSubMap().containsKey(key);
	}


	public boolean containsValue(Object value)
	{
		return getSubMap().containsValue(value);
	}


	public Object get(Object key)
	{
		return getSubMap().get(key);
	}


	public Object put(Object key, Object value)
	{
		synchronized (fast)
		{
			TreeMap clone = (TreeMap)fast.getMap().clone();
			SortedMap sub = getSubMap(clone);
			Object result = sub.put(key, value);
			fast.setMap(clone);
			return result;
		}
	}


	public Object remove(Object key)
	{
		synchronized (fast)
		{
			TreeMap clone = (TreeMap)fast.getMap().clone();
			SortedMap sub = getSubMap(clone);
			Object result = sub.remove(key);
			fast.setMap(clone);
			return result;
		}
	}


	public void putAll(Map m)
	{
		synchronized (fast)
		{
			TreeMap clone = (TreeMap)fast.getMap().clone();
			SortedMap sub = getSubMap(clone);
			sub.putAll(m);
			fast.setMap(clone);
		}
	}


	public void clear()
	{
		synchronized (fast)
		{
			TreeMap clone = (TreeMap)fast.getMap().clone();
			SortedMap sub = getSubMap(clone);
			sub.clear();
			fast.setMap(clone);
		}
	}


	public Set keySet()
	{
		return new FastMapKeySet(new SubFastTreeMapInnerCollectionState(this));
	}


	public Collection values()
	{
		return new FastMapValues(new SubFastTreeMapInnerCollectionState(this));
	}


	public Set entrySet()
	{
		return new FastMapEntrySet(new SubFastTreeMapInnerCollectionState(this));
	}


	public boolean equals(Object value)
	{
		return getSubMap().equals(value);
	}


	public int hashCode()
	{
		return getSubMap().hashCode();
	}


	public Comparator comparator()
	{
		return getSubMap().comparator();
	}


	public SortedMap subMap(Object first, Object last)
	{
		return new SubFastTreeMap(fast, first, last);
	}


	public SortedMap headMap(Object last)
	{
		return new SubFastTreeMap(fast, FastTreeMap.EXTREMITY, last);
	}


	public SortedMap tailMap(Object first)
	{
		return new SubFastTreeMap(fast, first, FastTreeMap.EXTREMITY);
	}


	public Object firstKey()
	{
		return getSubMap().firstKey();
	}


	public Object lastKey()
	{
		return getSubMap().lastKey();
	}


	public String toString()
	{
		return getSubMap().toString();
	}

}


class FastTreeMapInnerCollectionState extends FastMapInnerCollectionState
{

	public FastTreeMapInnerCollectionState(FastTreeMap map)
	{
		super(map, map);
	}


	public Map getInternalMap()
	{
		return ((FastTreeMap)fast).getMap();
	}


	public void setInternalMap(Map map)
	{
		((FastTreeMap)fast).setMap((TreeMap)map);
	}


	public Map getInternalClone()
	{
		TreeMap map = (TreeMap)getInternalMap();
		return (Map)map.clone();
	}

}



class SubFastTreeMapInnerCollectionState extends FastMapInnerCollectionState
{

	public SubFastTreeMapInnerCollectionState(SubFastTreeMap sub)
	{
		super(sub.fast, sub);
	}


	public Map getInternalMap()
	{
		SubFastTreeMap sub = (SubFastTreeMap)fast;
		return sub.fast.getMap();
	}

	public Map getSubMap(Map outer)
	{
		SubFastTreeMap sub = (SubFastTreeMap)fast;
		return sub.getSubMap((SortedMap)outer);
	}


	public void setInternalMap(Map m)
	{
		SubFastTreeMap sub = (SubFastTreeMap)fast;
		sub.fast.setMap((TreeMap)m);
	}


	public Map getInternalClone()
	{
		TreeMap map = (TreeMap)getInternalMap();
		return (Map)map.clone();
	}

}
