package org.apache.commons.collections;


import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;


/**
 *  Thread-safe hard reference to an object.
 *
 *  <P>I know that <I>sounds</I> simple, but in fact the 
 *  JVM's memory/thread model can allow odd things to 
 *  happen.  In particular, an object reference can
 *  be assigned to a field before the object has been
 *  fully constructed.  In the absence of synchronization,
 *  this means that a thread can potentially invoke a 
 *  method on an object that hasn't been constructed.
 *
 *  <P>The <Code>set</Code> method of this class 
 *  guarantees that an object will be fully constructed
 *  before it is stored; the <Code>get</Code> method
 *  will always return a fully constructed object.
 *  The <Code>get</Code> method also always returns the
 *  most recent operand of <Code>set</Code>.
 *
 *  <P>There is a system factory that produces concrete
 *  instances of this class.  By default, the system 
 *  factory synchronizes access to the referent.  See
 *  the {@link SafeReferenceFactory} interface for information
 *  on how to change the system factory.
 *
 *  @author Paul Jack
 */
public abstract class SafeReference
{

	/**
	 *  Specifies which factory to use to create references.
	 */
	final static SafeReferenceFactory factory;


	/**
	 *  Default constructor.  Please do not provide a public 
	 *  constructor for your subclass, as it is preferrable
	 *  for users to use the system factory.
	 */
	protected SafeReference()
	{
	}


	/**
	 *  Returns the referent of this reference.
	 *
	 *  <P>Subclasses must provide the referent provided
	 *  to the most recent <Code>set</Code> invocation.
	 *
	 *  @return the referent of this reference
	 */
	abstract Object get();


	/**
	 *  Sets the referent of this reference.
	 *
	 *  <P>Subclasses must ensure that an optimizing, inlining
	 *  compiler or sophisticated processor won't inline this 
	 *  method and potentially assign a less than fully formed
	 *  object to the reference.
	 *
	 *  @param object the new referent for this reference
	 */
	abstract void set(Object object);


	/**
	 *  Determines which factory to use to create references.
	 */
	static 
	{
		String s = System.getProperty("org.apache.SafeReferenceFactoryClass");
		SafeReferenceFactory f;
		if (s == null)
		{
			f = new SynchronizedSafeReferenceFactory();
		}
		else 
		{
			try
			{
				f = (SafeReferenceFactory)Class.forName(s).newInstance();
			}
			catch (Exception e)
			{
				f = new SynchronizedSafeReferenceFactory();
			}
		}
		factory = f;
	}


	/**
	 *  Provides a safe reference to the given object.<P>
	 *
	 *  <P>The approach used depends on the <Code>org.apache.SafeReferenceFactoryClass</Code>
	 *  system property as defined in {@link SafeReferenceFactory}.
	 *
	 *  @param object  the object to refer to
	 *  @return  a safe reference to that object
	 */
	public static SafeReference create(Object object)
	{
		return factory.createSafeReference(object);
	}


}


class SynchronizedSafeReferenceFactory implements SafeReferenceFactory
{


	public SafeReference createSafeReference(final Object object)
	{
		return new SynchronizedSafeReference(object);
	}

}


class SynchronizedSafeReference extends SafeReference
{

	private Object referent;


	public SynchronizedSafeReference(Object object)
	{
		this.referent = object;
	}


	public synchronized Object get()
	{
		return referent;
	}


	public synchronized void set(Object object)
	{
		this.referent = object;
	}

}
