package org.apache.commons.collections;


/**
 *  Factory interface for creating <Code>SafeReference</Code> objects.<P>
 *
 *  <P>The factory used by the "fast" collection classes
 *  is determined by setting the <Code>org.apache.SafeReferenceFactoryClass</Code>
 *  system property.  Set it the fully-qualified class name of your factory.
 *  The factory will be instantiated with <Code>Class.forName(name).newInstance()</Code>.
 *  If the system property is not set, a default factory that produces synchronized
 *  safe references will be used.
 *
 *  <P>The only cross-platform way to ensure a safe reference is to use
 *  explicit synchronization in the <Code>get</Code> and <Code>set</Code>
 *  methods, so if you have any doubts, you should use the default factory.  
 *  However, it may be possible for you to create a platform-
 *  specific <Code>SafeReference</Code> implementation that doesn't require
 *  the overhead of synchronization, either by exploiting some aspect of the
 *  native compiler, providing your own native method, or relying on stricter
 *  semantics of the <Code>volatile</Code> keyword.
 *
 *  <P>The Java Memory Model is currently being revised; in some future version of 
 *  Java, the following implementation will work:
 *
 *  <Pre>
 *  class VolatileSafeReference extends SafeReference {
 *
 *      volatile Object referent;
 *
 *      public Object get() {
 *          return referent;
 *      }
 *
 *      public void set(Object referent) {
 *          this.referent = referent;
 *      }
 *  }
 *  </Pre>
 *
 *  <P>Some current JVMs allegedly use this stricter version of 
 *  <Code>volatile</Code>, but I haven't bothered to confirm that,
 *  so no factory is provided for the above class.
 *
 *  <P>For background material on why the fast collection classes require this
 *  kind of "safe" reference, please see <A Href="http://www.cs.umd.edu/~pugh/java/memoryModel/DoubleCheckedLocking.html">
 *  Bill Pugh</A> or <A Href="http://www.javaworld.com/javaworld/jw-05-2001/jw-0525-double.html">JavaWorld</A>.
 *
 *  <P>If you have any doubts, use the default factory and forget about
 *  this interface and its system property.
 *
 *  @author Paul Jack
 */
public interface SafeReferenceFactory
{

	/**
	 *  Creates a new safe reference to the given object.
	 *
	 *  @param referent  the object to refer to
	 */
	SafeReference createSafeReference(Object referent);

}
