#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include <geos_c.h>
#include "shapefil.h"


#define isShapePolygon(t)	((t)==SHPT_POLYGON||(t)==SHPT_POLYGONZ||(t)==SHPT_POLYGONM)
#define isShapeArc(t)		((t)==SHPT_ARC||(t)==SHPT_ARCZ||(t)==SHPT_ARCM)

/*
 *  returns an array with the coordinates sequences of a Shapefile object
 *  (one sequence per part)
 */

GPtrArray *SHPGetGEOSCoordSeq (const SHPObject* object)
{
    GPtrArray         *seqArray;
    GEOSCoordSequence *sequence;
    int                start,end,i,j;

    assert(object);

    seqArray = g_ptr_array_sized_new(object->nParts);

    for (i=0;i<object->nParts;i++)
    {
        start = object->panPartStart[i];
        if (i != object->nParts-1) end = object->panPartStart[i+1];
        else                       end = object->nVertices;

        assert(sequence = GEOSCoordSeq_create(end-start,2));
        for (j=start;j<end;j++)
        {
            GEOSCoordSeq_setX(sequence,j-start,object->padfX[j]);
            GEOSCoordSeq_setY(sequence,j-start,object->padfY[j]);
        }
        g_ptr_array_add(seqArray,sequence);
    }
    return seqArray;
}

/*
 *  return the GEOS geometry of a Shapefile polygon
 *
 *  Assertion : the first part is the outer ring. Following parts are inner
 *  rings. ESRI SHapefile specifications indicate that rings can be done in any order.
 */

GEOSGeometry *SHPGetGEOSGeomFromPolygon (SHPObject *object)
{
    GPtrArray    *sequences	= NULL;
    GEOSGeometry *geosPolygon,*outerRing,**innerRings = NULL;
    int           i;

    assert(isShapePolygon(object->nSHPType));
    assert(sequences = SHPGetGEOSCoordSeq(object));
    assert(outerRing = GEOSGeom_createLinearRing(g_ptr_array_index(sequences,0)));

    if (sequences->len != 1)
    {
        assert(innerRings = calloc(sequences->len-1,sizeof(GEOSGeometry *)));
        for (i=1;i<sequences->len;i++)
            assert(innerRings[i-1] = GEOSGeom_createLinearRing(g_ptr_array_index(sequences,i)));
    }

    assert(geosPolygon = GEOSGeom_createPolygon(outerRing,innerRings,sequences->len-1));
    if (innerRings) free(innerRings);
    g_ptr_array_free(sequences,TRUE);
    return geosPolygon;
}

/*
 *  return the GEOS geometry of a Shapefile arc (polylines)
 *
 *  If there's only one part, a lineString (or linearRing) geometry is returned.
 *  If not, a multiLineString collection is returned.
 */

GEOSGeometry *SHPGetGEOSGeomFromArc (SHPObject *object)
{
    GEOSGeometry *geometry;
    GPtrArray    *sequences;
    int           i;
    

    assert(isShapeArc(object->nSHPType));
    assert(sequences = SHPGetGEOSCoordSeq (object));

    if (sequences->len == 1)
    {
        GEOSCoordSequence   *theSequence = g_ptr_array_index(sequences,0);
        double               firstX,lastX,firstY,lastY;

        GEOSCoordSeq_getX(theSequence,0,&firstX);
        GEOSCoordSeq_getY(theSequence,0,&firstY);
        GEOSCoordSeq_getX(theSequence,object->nVertices-1,&lastX); 
        GEOSCoordSeq_getY(theSequence,object->nVertices-1,&lastY);

        if (firstX == lastX && firstY == lastY)
            assert(geometry = GEOSGeom_createLinearRing(theSequence));
        else
            assert(geometry = GEOSGeom_createLineString(theSequence));
    }
    else
    {
        GEOSGeometry **lineStrings;
        assert(lineStrings = calloc(sequences->len,sizeof(GEOSGeometry *)));
        for (i=0;i<sequences->len;i++)
            assert(lineStrings[i] = GEOSGeom_createLineString(g_ptr_array_index(sequences,i)));
        assert(geometry = GEOSGeom_createCollection(GEOS_MULTILINESTRING,lineStrings,sequences->len));
        free(*lineStrings);
    }

    g_ptr_array_free(sequences,TRUE);
    return geometry;
}


int main (int argc,char *argv[])
{
    SHPHandle        shape;
    SHPObject       *object;
    GEOSGeometry    *geom;
    int              i,j,nEntities,shapeType;
    double           minBound[4],maxBound[4];
    GPtrArray       *geometries;
    GEOSWKTWriter   *wktwriter;
    GEOSGeometry    *(*GeosGeomFunction)(SHPObject *);
    
    // opens the shapefile and gets its characteristics
    assert(shape = SHPOpen(argv[1],"rb"));
    SHPGetInfo(shape,&nEntities,&shapeType,minBound,maxBound );

    // initializes (with allocation) the GEOSGeometry array
    geometries = g_ptr_array_sized_new(nEntities);

    // sets the GeosGeomFunction to call (to get GEOSGeometry from a shape object)
    // according to the shapefile type (works only with SHPT_POLYGON[ZM] and SHPT_ARC[ZM] types)
    if      (isShapePolygon(shapeType)) GeosGeomFunction = SHPGetGEOSGeomFromPolygon;
    else if (isShapeArc(shapeType))     GeosGeomFunction = SHPGetGEOSGeomFromArc;
    else assert(0);

    // loads the objects, builds the GEOSGeometry
    for (i=0;i<nEntities;i++)
    {
        assert(object = SHPReadObject(shape,i));
        assert(geom  = GeosGeomFunction(object));
        g_ptr_array_add(geometries,geom);
        SHPDestroyObject(object);
    }
    SHPClose(shape);


    assert(wktwriter = GEOSWKTWriter_create());


    for (i=0;i<geometries->len-1;i++)
    {
        GEOSGeometry *geom1 = g_ptr_array_index(geometries,i);
        for (j=i+1;j<geometries->len;j++)
        {
            GEOSGeometry *geom2 = g_ptr_array_index(geometries,j);
            if (GEOSDisjoint(geom1,geom2) == 0)
            {
                char *buffer;
                GEOSGeometry *res = GEOSIntersection(geom1,geom2);
                assert(res);
                buffer = GEOSWKTWriter_write(wktwriter,res);
                fprintf(stdout,"%s\n",buffer);
                free(buffer);
                GEOSGeom_destroy(res);
            }
        }
    }

    GEOSWKTWriter_destroy(wktwriter);
    for (i=0;i<geometries->len;i++)
        GEOSGeom_destroy(g_ptr_array_index(geometries,i));
    g_ptr_array_free(geometries,TRUE);

    return 0;
}

