
#if !defined(_I2ODEF_H_)

#define	_I2ODEF_H

#if !defined(_I2OTYPES_)

#define	_I2OTYPES_

/*
 * Pragma macros. These are to assure appropriate alignment between
 * host/IOP as defined by the I2O Specification. Each one of the shared
 * header files includes these macros.
 */

#define PRAGMA_ALIGN_PUSH \
#pragma align 4
#define PRAGMA_ALIGN_POP \
#pragma align 0
#define PRAGMA_PACK_PUSH \
#pragma pack 1
#define PRAGMA_PACK_POP \
#pragma pack 0

/* Setup the basics */

typedef    char   S8;
typedef    short  S16;

typedef    unsigned char  U8;
typedef    unsigned short U16;

//typedef    unsigned int   U32;
//typedef    int            S32;
typedef	   unsigned long U32;
typedef	   long		 S32;

/* Bitfields */

typedef    U32  BF;


/* VOID */

#ifndef __VOID
typedef    void  VOID;
#define __VOID
#endif


/* Boolean */

#ifndef __BOOL
#define __BOOL

#ifdef FALSE
#undef FALSE
#undef TRUE
#endif

typedef enum
        {
        FALSE = 0,
	TRUE  = 1
	} BOOL;
#endif

/**************************************************************************/

/* 64 bit defines */

typedef struct _S64 {
   U32                         LowPart;
   S32                         HighPart;
} S64;

typedef struct _U64 {
   U32                         LowPart;
   U32                         HighPart;
} U64;

/* Pointer to Basics */

typedef    VOID                *PVOID;
typedef    S8                  *PS8;
typedef    S16                 *PS16;
typedef    S32                 *PS32;
typedef    S64                 *PS64;

/* Pointer to Unsigned Basics */

typedef    U8                  *PU8;
typedef    U16                 *PU16;
typedef    U32                 *PU32;
typedef    U64                 *PU64;

/* misc */

typedef S32             I2O_ARG;
typedef U32             I2O_COUNT;
typedef U32             I2O_USECS;
typedef U32             I2O_ADDR32;
typedef U32             I2O_SIZE;

#endif	/* _I2OTYPES_ */

/**************************************************************************/

/* I2O BSA Block Read Message Control Flags */

typedef U16     I2O_BSA_READ_FLAGS;
#define I2O_BSA_RD_FLAG_DONT_RETRY      0x0001
#define I2O_BSA_RD_FLAG_SOLO            0x0002
#define I2O_BSA_RD_FLAG_CACHE_READ      0x0004
#define I2O_BSA_RD_FLAG_READ_PREFETCH   0x0008
#define I2O_BSA_RD_FLAG_CACHE_DATA      0x0010

/* I2O BSA Block Write Message Control Flags */

typedef U16     I2O_BSA_WRITE_FLAGS;
#define I2O_BSA_WR_FLAG_DONT_RETRY      0x0001
#define I2O_BSA_WR_FLAG_SOLO            0x0002
#define I2O_BSA_WR_FLAG_DONT_CACHE      0x0004
#define I2O_BSA_WR_FLAG_WRITE_THRU      0x0008
#define I2O_BSA_WR_FLAG_WRITE_TO        0x0010

/****************************************************************************/

typedef    U32        I2O_INITIATOR_CONTEXT;
typedef    U32        I2O_TRANSACTION_CONTEXT;

typedef    U32        I2O_PARAMETER_TID;

/****************************************************************************/
/* Message Frame defines and structures  */

/*   Defines for the Version_Status field. */

#define    I2O_VERSION_10                              0x00
#define    I2O_VERSION_11                              0x01

#define    I2O_VERSION_OFFSET_NUMBER_MASK              0x07
#define    I2O_VERSION_OFFSET_SGL_TRL_OFFSET_MASK      0xF0

/*   Defines for the Message Flags Field. */
/*   Please Note the the FAIL bit is only set in the Transport Fail Message. */
#define    I2O_MESSAGE_FLAGS_STATIC                    0x01
#define    I2O_MESSAGE_FLAGS_64BIT_CONTEXT             0x02
#define    I2O_MESSAGE_FLAGS_MULTIPLE                  0x10
#define    I2O_MESSAGE_FLAGS_FAIL                      0x20
#define    I2O_MESSAGE_FLAGS_LAST                      0x40
#define    I2O_MESSAGE_FLAGS_REPLY                     0x80

/* Defines for Request Status Codes:  Table 3-1 Reply Status Codes.  */

#define    I2O_REPLY_STATUS_SUCCESS                    0x00
#define    I2O_REPLY_STATUS_ABORT_DIRTY                0x01
#define    I2O_REPLY_STATUS_ABORT_NO_DATA_TRANSFER     0x02
#define    I2O_REPLY_STATUS_ABORT_PARTIAL_TRANSFER     0x03
#define    I2O_REPLY_STATUS_ERROR_DIRTY                0x04
#define    I2O_REPLY_STATUS_ERROR_NO_DATA_TRANSFER     0x05
#define    I2O_REPLY_STATUS_ERROR_PARTIAL_TRANSFER     0x06
#define    I2O_REPLY_STATUS_PROCESS_ABORT_DIRTY        0x08
#define    I2O_REPLY_STATUS_PROCESS_ABORT_NO_DATA_TRANSFER   0x09
#define    I2O_REPLY_STATUS_PROCESS_ABORT_PARTIAL_TRANSFER   0x0A
#define    I2O_REPLY_STATUS_TRANSACTION_ERROR          0x0B
#define    I2O_REPLY_STATUS_PROGRESS_REPORT            0x80

/* DetailedStatusCode defines for ALL messages: Table 3-2 Detailed Status Codes.  */

#define    I2O_DETAIL_STATUS_SUCCESS                        0x0000
#define    I2O_DETAIL_STATUS_BAD_KEY                        0x0002
#define    I2O_DETAIL_STATUS_TCL_ERROR                      0x0003
#define    I2O_DETAIL_STATUS_REPLY_BUFFER_FULL              0x0004
#define    I2O_DETAIL_STATUS_NO_SUCH_PAGE                   0x0005
#define    I2O_DETAIL_STATUS_INSUFFICIENT_RESOURCE_SOFT     0x0006
#define    I2O_DETAIL_STATUS_INSUFFICIENT_RESOURCE_HARD     0x0007
#define    I2O_DETAIL_STATUS_CHAIN_BUFFER_TOO_LARGE         0x0009
#define    I2O_DETAIL_STATUS_UNSUPPORTED_FUNCTION           0x000A
#define    I2O_DETAIL_STATUS_DEVICE_LOCKED                  0x000B
#define    I2O_DETAIL_STATUS_DEVICE_RESET                   0x000C
#define    I2O_DETAIL_STATUS_INAPPROPRIATE_FUNCTION         0x000D
#define    I2O_DETAIL_STATUS_INVALID_INITIATOR_ADDRESS      0x000E
#define    I2O_DETAIL_STATUS_INVALID_MESSAGE_FLAGS          0x000F
#define    I2O_DETAIL_STATUS_INVALID_OFFSET                 0x0010
#define    I2O_DETAIL_STATUS_INVALID_PARAMETER              0x0011
#define    I2O_DETAIL_STATUS_INVALID_REQUEST                0x0012
#define    I2O_DETAIL_STATUS_INVALID_TARGET_ADDRESS         0x0013
#define    I2O_DETAIL_STATUS_MESSAGE_TOO_LARGE              0x0014
#define    I2O_DETAIL_STATUS_MESSAGE_TOO_SMALL              0x0015
#define    I2O_DETAIL_STATUS_MISSING_PARAMETER              0x0016
#define    I2O_DETAIL_STATUS_TIMEOUT                        0x0017
#define    I2O_DETAIL_STATUS_UNKNOWN_ERROR                  0x0018
#define    I2O_DETAIL_STATUS_UNKNOWN_FUNCTION               0x0019
#define    I2O_DETAIL_STATUS_UNSUPPORTED_VERSION            0x001A
#define    I2O_DEATIL_STATUS_DEVICE_BUSY                    0x001B
#define    I2O_DETAIL_STATUS_DEVICE_NOT_AVAILABLE           0x001C

/* Common I2O Field sizes  */

#define    I2O_TID_SZ                                  12
#define    I2O_FUNCTION_SZ                             8
#define    I2O_UNIT_ID_SZ                              16
#define    I2O_SEGMENT_NUMBER_SZ                       12

#define    I2O_IOP_ID_SZ                               12
#define    I2O_GROUP_ID_SZ                             16
#define    I2O_IOP_STATE_SZ                            8
#define    I2O_MESSENGER_TYPE_SZ                       8

#define    I2O_CLASS_ID_SZ                             12
#define    I2O_CLASS_ORGANIZATION_ID_SZ                16

#define    I2O_4BIT_VERSION_SZ                         4
#define    I2O_8BIT_FLAGS_SZ                           8
#define    I2O_COMMON_LENGTH_FIELD_SZ                  16


#define    I2O_DEVID_DESCRIPTION_SZ                    16
#define    I2O_DEVID_VENDOR_INFO_SZ                    16
#define    I2O_DEVID_PRODUCT_INFO_SZ                   16
#define    I2O_DEVID_REV_LEVEL_SZ                      8
#define    I2O_MODULE_NAME_SZ                          24

#define    I2O_BIOS_INFO_SZ                            8

#define    I2O_RESERVED_4BITS                          4
#define    I2O_RESERVED_8BITS                          8
#define    I2O_RESERVED_12BITS                         12
#define    I2O_RESERVED_16BITS                         16
#define    I2O_RESERVED_20BITS                         20
#define    I2O_RESERVED_24BITS                         24
#define    I2O_RESERVED_28BITS                         28

/****************************************************************************/

/* Common functions accross all classes. */

#define    I2O_PRIVATE_MESSAGE                         0xFF

/****************************************************************************/
/* Class ID and Code Assignments */


#define    I2O_CLASS_VERSION_10                        0x00
#define    I2O_CLASS_VERSION_11                        0x01

/*    Class Code Names:  Table 6-1 Class Code Assignments. */
#define    I2O_CLASS_EXECUTIVE                         0x000
#define    I2O_CLASS_DDM                               0x001
#define    I2O_CLASS_RANDOM_BLOCK_STORAGE              0x010
#define    I2O_CLASS_SEQUENTIAL_STORAGE                0x011
#define    I2O_CLASS_LAN                               0x020
#define    I2O_CLASS_WAN                               0x030
#define    I2O_CLASS_FIBRE_CHANNEL_PORT                0x040
#define    I2O_CLASS_FIBRE_CHANNEL_PERIPHERAL          0x041
#define    I2O_CLASS_SCSI_PERIPHERAL                   0x051
#define    I2O_CLASS_ATE_PORT                          0x060
#define    I2O_CLASS_ATE_PERIPHERAL                    0x061
#define    I2O_CLASS_FLOPPY_CONTROLLER                 0x070
#define    I2O_CLASS_FLOPPY_DEVICE                     0x071
#define    I2O_CLASS_BUS_ADAPTER_PORT                  0x080
/* Class Codes 0x090 - 0x09f are reserved for Peer-to-Peer classes */
#define    I2O_CLASS_MATCH_ANYCLASS                    0xffffffff

#define    I2O_SUBCLASS_i960                           0x001
#define    I2O_SUBCLASS_HDM                            0x020
#define    I2O_SUBCLASS_ISM                            0x021


/*********************************************************************/

#define    I2O_MAX_SERIAL_NUMBER_SZ                    256

/****************************************************************************/

/* I2O Message Frame common for all messages  */

typedef struct _I2O_MESSAGE_FRAME {
   U8                          VersionOffset;
   U8                          MsgFlags;
   U16                         MessageSize;
   BF                          TargetAddress:I2O_TID_SZ;
   BF                          InitiatorAddress:I2O_TID_SZ;
   BF                          Function:I2O_FUNCTION_SZ;
   I2O_INITIATOR_CONTEXT       InitiatorContext;
} I2O_MESSAGE_FRAME, *PI2O_MESSAGE_FRAME;


/****************************************************************************/

/* I2O Successful Single Transaction Reply Message Frame structure. */

typedef struct _I2O_SINGLE_REPLY_MESSAGE_FRAME {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
   U16                         DetailedStatusCode;
   U8                          reserved;
   U8                          ReqStatus;
/*                             ReplyPayload        */
} I2O_SINGLE_REPLY_MESSAGE_FRAME, *PI2O_SINGLE_REPLY_MESSAGE_FRAME;


/****************************************************************************/

/* I2O Private Message Frame structure. */
typedef struct _I2O_PRIVATE_MESSAGE_FRAME {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
   U16                         XFunctionCode;
   U16                         OrganizationID;
/*                             PrivatePayload[]        */
} I2O_PRIVATE_MESSAGE_FRAME, *PI2O_PRIVATE_MESSAGE_FRAME;

/****************************************************************************/

/*
    Random Block Storage Class specific functions
    
    Although the names are block storage class specific, the values 
    assigned are common with other classes when applicable.
*/

#define     I2O_BSA_BLOCK_READ              0x30
#define     I2O_BSA_BLOCK_REASSIGN          0x71
#define     I2O_BSA_BLOCK_WRITE             0x31
#define     I2O_BSA_BLOCK_WRITE_VERIFY      0x33
#define     I2O_BSA_CACHE_FLUSH             0x37
#define     I2O_BSA_DEVICE_RESET            0x27
#define     I2O_BSA_MEDIA_EJECT             0x43
#define     I2O_BSA_MEDIA_FORMAT            0x45
#define     I2O_BSA_MEDIA_LOCK              0x49
#define     I2O_BSA_MEDIA_MOUNT             0x41
#define     I2O_BSA_MEDIA_UNLOCK            0x4B
#define     I2O_BSA_MEDIA_VERIFY            0x35
#define     I2O_BSA_POWER_MANAGEMENT        0x70
#define     I2O_BSA_STATUS_CHECK            0x25

/****************************************************************************/

/* Memory Addressing structures and defines. */

/* SglFlags defines. */

#define    I2O_SGL_FLAGS_LAST_ELEMENT                  0x80
#define    I2O_SGL_FLAGS_END_OF_BUFFER                 0x40

#define    I2O_SGL_FLAGS_IGNORE_ELEMENT                0x00
#define    I2O_SGL_FLAGS_TRANSPORT_ELEMENT             0x04
#define    I2O_SGL_FLAGS_BIT_BUCKET_ELEMENT            0x08
#define    I2O_SGL_FLAGS_IMMEDIATE_DATA_ELEMENT        0x0C
#define    I2O_SGL_FLAGS_SIMPLE_ADDRESS_ELEMENT        0x10
#define    I2O_SGL_FLAGS_PAGE_LIST_ADDRESS_ELEMENT     0x20
#define    I2O_SGL_FLAGS_CHAIN_POINTER_ELEMENT         0x30
#define    I2O_SGL_FLAGS_LONG_TRANSACTION_ELEMENT      0x40
#define    I2O_SGL_FLAGS_SHORT_TRANSACTION_ELEMENT     0x70
#define    I2O_SGL_FLAGS_SGL_ATTRIBUTES_ELEMENT        0x7C

#define    I2O_SGL_FLAGS_BC0                           0x01
#define    I2O_SGL_FLAGS_BC1                           0x02
#define    I2O_SGL_FLAGS_DIR                           0x04
#define    I2O_SGL_FLAGS_LOCAL_ADDRESS                 0x08

#define    I2O_SGL_FLAGS_CONTEXT_COUNT_MASK            0x03
#define    I2O_SGL_FLAGS_ADDRESS_MODE_MASK             0x3C
#define    I2O_SGL_FLAGS_NO_CONTEXT                    0x00

/* 32 Bit Context Field defines */

#define    I2O_SGL_FLAGS_CONTEXT32_NULL                0x00
#define    I2O_SGL_FLAGS_CONTEXT32_U32                 0x01
#define    I2O_SGL_FLAGS_CONTEXT32_U64                 0x02
#define    I2O_SGL_FLAGS_CONTEXT32_U96                 0x03

#define    I2O_SGL_FLAGS_CONTEXT32_NULL_SZ             0x00
#define    I2O_SGL_FLAGS_CONTEXT32_U32_SZ              0x04
#define    I2O_SGL_FLAGS_CONTEXT32_U64_SZ              0x08
#define    I2O_SGL_FLAGS_CONTEXT32_U96_SZ              0x0C

/* 64 Bit Context Field defines */

#define    I2O_SGL_FLAGS_CONTEXT64_NULL                0x00
#define    I2O_SGL_FLAGS_CONTEXT64_U64                 0x01
#define    I2O_SGL_FLAGS_CONTEXT64_U128                0x02
#define    I2O_SGL_FLAGS_CONTEXT64_U192                0x03

#define    I2O_SGL_FLAGS_CONTEXT64_NULL_SZ             0x00
#define    I2O_SGL_FLAGS_CONTEXT64_U64_SZ              0x08
#define    I2O_SGL_FLAGS_CONTEXT64_U128_SZ             0x10
#define    I2O_SGL_FLAGS_CONTEXT64_U192_SZ             0x18

/* SGL Attribute Element defines */

#define    I2O_SGL_ATTRIBUTE_FLAGS_BIT_BUCKET_HINT     0x0400
#define    I2O_SGL_ATTRIBUTE_FLAGS_IMMEDIATE_DATA_HINT 0x0200
#define    I2O_SGL_ATTRIBUTE_FLAGS_LOCAL_ADDRESS_HINT  0x0100
#define    I2O_SGL_ATTRIBUTE_FLAGS_32BIT_TRANSACTION   0x0000
#define    I2O_SGL_ATTRIBUTE_FLAGS_64BIT_TRANSACTION   0x0004
#define    I2O_SGL_ATTRIBUTE_FLAGS_32BIT_LOCAL_ADDRESS 0x0000

/* SG Size defines */

#define    I2O_SG_COUNT_SZ                             24
#define    I2O_SG_FLAGS_SZ                             8

/* Standard Flags and Count fields for SG Elements */

typedef struct _I2O_FLAGS_COUNT {
   BF                          Count:I2O_SG_COUNT_SZ;
   BF                          Flags:I2O_SG_FLAGS_SZ;
} I2O_FLAGS_COUNT, *PI2O_FLAGS_COUNT;

/* Bit Bucket Element */
typedef struct _I2O_SGE_BIT_BUCKET_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         BufferContext;
} I2O_SGE_BIT_BUCKET_ELEMENT, *PI2O_SGE_BIT_BUCKET_ELEMENT;

/* Chain Addressing Scatter-Gather Element */

typedef struct _I2O_SGE_CHAIN_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         PhysicalAddress;
} I2O_SGE_CHAIN_ELEMENT, *PI2O_SGE_CHAIN_ELEMENT;

/* Chain Addressing with Context Scatter-Gather Element */

typedef struct _I2O_SGE_CHAIN_CONTEXT_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         Context[1];
   U32                         PhysicalAddress;
} I2O_SGE_CHAIN_CONTEXT_ELEMENT, *PI2O_SGE_CHAIN_CONTEXT_ELEMENT;

/* Ignore Scatter-Gather Element */

typedef struct _I2O_SGE_IGNORE_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
} I2O_SGE_IGNORE_ELEMENT, *PI2O_SGE_IGNORE_ELEMENT;

/* Immediate Data Element */

typedef struct _I2O_SGE_IMMEDIATE_DATA_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
} I2O_SGE_IMMEDIATE_DATA_ELEMENT, *PI2O_SGE_IMMEDIATE_DATA_ELEMENT;

/* Immediate Data with Context Element */

typedef struct _I2O_SGE_IMMEDIATE_DATA_CONTEXT_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         BufferContext;
} I2O_SGE_IMMEDIATE_DATA_CONTEXT_ELEMENT, *PI2O_SGE_IMMEDIATE_DATA_CONTEXT_ELEMENT;

/* Long Transaction Parameters Element */

typedef struct _I2O_SGE_LONG_TRANSACTION_ELEMENT {
   BF                          LongElementLength:I2O_SG_COUNT_SZ;
   BF                          Flags:I2O_SG_FLAGS_SZ;
   U32                         BufferContext;
} I2O_SGE_LONG_TRANSACTION_ELEMENT, *PI2O_SGE_LONG_TRANSACTION_ELEMENT;

/* Page List Scatter-Gather Element */

typedef struct _I2O_SGE_PAGE_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         PhysicalAddress[1];
} I2O_SGE_PAGE_ELEMENT , *PI2O_SGE_PAGE_ELEMENT ;

/* Page List with Context Scatter-Gather Element */

typedef struct _I2O_SGE_PAGE_CONTEXT_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         BufferContext[1];
   U32                         PhysicalAddress[1];
} I2O_SGE_PAGE_CONTEXT_ELEMENT, *PI2O_SGE_PAGE_CONTEXT_ELEMENT;

/* SGL Attribute Element */

typedef struct _I2O_SGE_SGL_ATTRIBUTES_ELEMENT {
   U16                         SglAttributeFlags;
   U8                          ElementLength;
   U8                          Flags;
   U32                         PageFrameSize;
} I2O_SGE_SGL_ATTRIBUTES_ELEMENT, *PI2O_SGE_SGL_ATTRIBUTES_ELEMENT;

/* Short Transaction Parameters Element */

typedef struct _I2O_SGE_SHORT_TRANSACTION_ELEMENT {
   U16                         ClassFields;
   U8                          ElementLength;
   U8                          Flags;
   U32                         BufferContext;
} I2O_SGE_SHORT_TRANSACTION_ELEMENT, *PI2O_SGE_SHORT_TRANSACTION_ELEMENT;

/* Simple Addressing Scatter-Gather Element */

typedef struct _I2O_SGE_SIMPLE_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         PhysicalAddress;
} I2O_SGE_SIMPLE_ELEMENT, *PI2O_SGE_SIMPLE_ELEMENT;

/* Simple Addressing with Context Scatter-Gather Element */

typedef struct _I2O_SGE_SIMPLE_CONTEXT_ELEMENT {
   I2O_FLAGS_COUNT             FlagsCount;
   U32                         BufferContext[1];
   U32                         PhysicalAddress;
} I2O_SGE_SIMPLE_CONTEXT_ELEMENT, *PI2O_SGE_SIMPLE_CONTEXT_ELEMENT;

/* Transport Detail Element */

typedef struct _I2O_SGE_TRANSPORT_ELEMENT {
   BF                          LongElementLength:I2O_SG_COUNT_SZ;
   BF                          Flags:I2O_SG_FLAGS_SZ;
} I2O_SGE_TRANSPORT_ELEMENT, *PI2O_SGE_TRANSPORT_ELEMENT;


typedef struct _I2O_SG_ELEMENT {
   union {
       /* Bit Bucket Element */
       I2O_SGE_BIT_BUCKET_ELEMENT          BitBucket;

       /* Chain Addressing Element */
       I2O_SGE_CHAIN_ELEMENT               Chain;

       /* Chain Addressing with Context Element */
       I2O_SGE_CHAIN_CONTEXT_ELEMENT       ChainContext;

       /* Ignore Scatter-Gather Element */
       I2O_SGE_IGNORE_ELEMENT              Ignore;

       /* Immediate Data Element */
       I2O_SGE_IMMEDIATE_DATA_ELEMENT      ImmediateData;

       /* Immediate Data with Context Element */
       I2O_SGE_IMMEDIATE_DATA_CONTEXT_ELEMENT  ImmediateDataContext;

       /* Long Transaction Parameters Element */
       I2O_SGE_LONG_TRANSACTION_ELEMENT    LongTransaction;

       /* Page List Element */
       I2O_SGE_PAGE_ELEMENT                Page;

       /* Page List with Context Element */
       I2O_SGE_PAGE_CONTEXT_ELEMENT        PageContext;

       /* SGL Attribute Element */
       I2O_SGE_SGL_ATTRIBUTES_ELEMENT      SGLAttribute;

       /* Short Transaction Parameters Element */
       I2O_SGE_SHORT_TRANSACTION_ELEMENT   ShortTransaction;

       /* Simple Addressing Element */
       I2O_SGE_SIMPLE_ELEMENT              Simple[1];

       /* Simple Addressing with Context Element */
       I2O_SGE_SIMPLE_CONTEXT_ELEMENT      SimpleContext[1];

       /* Transport Detail Element */
       I2O_SGE_TRANSPORT_ELEMENT           Transport;
   }u;
} I2O_SG_ELEMENT, *PI2O_SG_ELEMENT;

/* I2O BSA Block Read Message Frame */
typedef struct _I2O_BSA_READ_MESSAGE {
    I2O_MESSAGE_FRAME       StdMessageFrame;
    I2O_TRANSACTION_CONTEXT TransactionContext;
    I2O_BSA_READ_FLAGS      ControlFlags;
    U8                      TimeMultiplier;
    U8                      FetchAhead;
    U32                     TransferByteCount;
    U64                     LogicalByteAddress;
    I2O_SG_ELEMENT          SGL;
} I2O_BSA_READ_MESSAGE, *PI2O_BSA_READ_MESSAGE;


/***********************************************************************/

/* Class ID Block */

typedef struct _I2O_CLASS_ID {
   BF                          Class:I2O_CLASS_ID_SZ;
   BF                          Version:I2O_4BIT_VERSION_SZ;
   BF                          OrganizationID:I2O_CLASS_ORGANIZATION_ID_SZ;
} I2O_CLASS_ID, *PI2O_CLASS_ID;


/****************************************************************************/
/* Logical Configuration Table  */
/****************************************************************************/

/* I2O Logical Configuration Table structures. */

#define    I2O_IDENTITY_TAG_SZ                         8

/* I2O Logical Configuration Table Device Flags */

#define    I2O_LCT_DEVICE_FLAGS_CONF_DIALOG_REQUEST            0x01
#define    I2O_LCT_DEVICE_FLAGS_MORE_THAN_1_USER               0x02
#define    I2O_LCT_DEVICE_FLAGS_PEER_SERVICE_DISABLED          0x10
#define    I2O_LCT_DEVICE_FLAGS_MANAGEMENT_SERVICE_DISABLED    0x20

/* LCT Entry Block */

typedef struct _I2O_LCT_ENTRY {
   BF                          TableEntrySize:I2O_COMMON_LENGTH_FIELD_SZ;
   BF                          LocalTID:I2O_TID_SZ;
   BF                          reserved:I2O_4BIT_VERSION_SZ;
   U32                         ChangeIndicator;
   U32                         DeviceFlags;
   I2O_CLASS_ID                ClassID;
   U32                         SubClassInfo;
   BF                          UserTID:I2O_TID_SZ;
   BF                          ParentTID:I2O_TID_SZ;
   BF                          BiosInfo:I2O_BIOS_INFO_SZ;
   U8                          IdentityTag[I2O_IDENTITY_TAG_SZ];
   U32                         EventCapabilities;
} I2O_LCT_ENTRY, *PI2O_LCT_ENTRY;

/* I2O Logical Configuration Table structure. */
typedef struct _I2O_LCT {
   BF                          TableSize:I2O_COMMON_LENGTH_FIELD_SZ;
   BF                          BootDeviceTID:I2O_TID_SZ;
   BF                          LctVer:I2O_4BIT_VERSION_SZ;
   U32                         IopFlags;
   U32                         CurrentChangeIndicator;
   I2O_LCT_ENTRY               LCTEntry[1];
} I2O_LCT, *PI2O_LCT;


/****************************************************************************/

/*  I2O Executive Function Codes.  */

#define    I2O_EXEC_ADAPTER_ASSIGN                     0xB3
#define    I2O_EXEC_ADAPTER_READ                       0xB2
#define    I2O_EXEC_ADAPTER_RELEASE                    0xB5
#define    I2O_EXEC_BIOS_INFO_SET                      0xA5
#define    I2O_EXEC_BOOT_DEVICE_SET                    0xA7
#define    I2O_EXEC_CONFIG_VALIDATE                    0xBB
#define    I2O_EXEC_CONN_SETUP                         0xCA
#define    I2O_EXEC_DDM_DESTROY                        0xB1
#define    I2O_EXEC_DDM_ENABLE                         0xD5
#define    I2O_EXEC_DDM_QUIESCE                        0xC7
#define    I2O_EXEC_DDM_RESET                          0xD9
#define    I2O_EXEC_DDM_SUSPEND                        0xAF
#define    I2O_EXEC_DEVICE_ASSIGN                      0xB7
#define    I2O_EXEC_DEVICE_RELEASE                     0xB9
#define    I2O_EXEC_HRT_GET                            0xA8
#define    I2O_EXEC_IOP_CLEAR                          0xBE
#define    I2O_EXEC_IOP_CONNECT                        0xC9
#define    I2O_EXEC_IOP_RESET                          0xBD
#define    I2O_EXEC_LCT_NOTIFY                         0xA2
#define    I2O_EXEC_OUTBOUND_INIT                      0xA1
#define    I2O_EXEC_PATH_ENABLE                        0xD3
#define    I2O_EXEC_PATH_QUIESCE                       0xC5
#define    I2O_EXEC_PATH_RESET                         0xD7
#define    I2O_EXEC_STATIC_MF_CREATE                   0xDD
#define    I2O_EXEC_STATIC_MF_RELEASE                  0xDF
#define    I2O_EXEC_STATUS_GET                         0xA0
#define    I2O_EXEC_SW_DOWNLOAD                        0xA9
#define    I2O_EXEC_SW_UPLOAD                          0xAB
#define    I2O_EXEC_SW_REMOVE                          0xAD
#define    I2O_EXEC_SYS_ENABLE                         0xD1
#define    I2O_EXEC_SYS_MODIFY                         0xC1
#define    I2O_EXEC_SYS_QUIESCE                        0xC3
#define    I2O_EXEC_SYS_TAB_SET                        0xA3


/* I2O Get Status State values */

#define    I2O_IOP_STATE_INITIALIZING                  0x01
#define    I2O_IOP_STATE_RESET                         0x02
#define    I2O_IOP_STATE_HOLD                          0x04
#define    I2O_IOP_STATE_READY                         0x05
#define    I2O_IOP_STATE_OPERATIONAL                   0x08
#define    I2O_IOP_STATE_FAILED                        0x10
#define    I2O_IOP_STATE_FAULTED                       0x11


#define    I2O_EXEC_STATUS_GET_RESERVED_SZ             16

/* ExecStatusGet Function Message Frame structure. */

typedef struct _I2O_EXEC_STATUS_GET_MESSAGE {
   U8                          VersionOffset;
   U8                          MsgFlags;
   U16                         MessageSize;
   BF                          TargetAddress:I2O_TID_SZ;
   BF                          InitiatorAddress:I2O_TID_SZ;
   BF                          Function:I2O_FUNCTION_SZ;
   U8                          Reserved[I2O_EXEC_STATUS_GET_RESERVED_SZ];
   U32                         ReplyBufferAddressLow;
   U32                         ReplyBufferAddressHigh;
   U32                         ReplyBufferLength;
} I2O_EXEC_STATUS_GET_MESSAGE, *PI2O_EXEC_STATUS_GET_MESSAGE;


#define    I2O_IOP_STATUS_PROD_ID_STR_SZ               24
#define    I2O_EXEC_STATUS_GET_REPLY_RESERVED_SZ       6

/* ExecStatusGet reply Structure */

#define    I2O_IOP_CAP_CONTEXT_32_ONLY                 0x00000000
#define    I2O_IOP_CAP_CONTEXT_64_ONLY                 0x00000001
#define    I2O_IOP_CAP_CONTEXT_32_64_NOT_CURRENTLY     0x00000002
#define    I2O_IOP_CAP_CONTEXT_32_64_CURRENTLY         0x00000003
#define    I2O_IOP_CAP_CURRENT_CONTEXT_NOT_CONFIG      0x00000000
#define    I2O_IOP_CAP_CURRENT_CONTEXT_32_ONLY         0x00000004
#define    I2O_IOP_CAP_CURRENT_CONTEXT_64_ONLY         0x00000008
#define    I2O_IOP_CAP_CURRENT_CONTEXT_32_64           0x0000000C
#define    I2O_IOP_CAP_INBOUND_PEER_SUPPORT            0x00000010
#define    I2O_IOP_CAP_OUTBOUND_PEER_SUPPORT           0x00000020
#define    I2O_IOP_CAP_PEER_TO_PEER_SUPPORT            0x00000040

typedef struct _I2O_EXEC_STATUS_GET_REPLY {
   U16                         OrganizationID;
   U16                         reserved;
   BF                          IOP_ID:I2O_IOP_ID_SZ;
   BF                          reserved1:I2O_RESERVED_4BITS;
   BF                          HostUnitID:I2O_UNIT_ID_SZ;
   BF                          SegmentNumber:I2O_SEGMENT_NUMBER_SZ;
   BF                          I2oVersion:I2O_4BIT_VERSION_SZ;
   BF                          IopState:I2O_IOP_STATE_SZ;
   BF                          MessengerType:I2O_MESSENGER_TYPE_SZ;
   U16                         InboundMFrameSize;
   U8                          InitCode;
   U8                          reserved2;
   U32                         MaxInboundMFrames;
   U32                         CurrentInboundMFrames;
   U32                         MaxOutboundMFrames;
   U8                          ProductIDString[I2O_IOP_STATUS_PROD_ID_STR_SZ];
   U32                         ExpectedLCTSize;
   U32                         IopCapabilities;
   U32                         DesiredPrivateMemSize;
   U32                         CurrentPrivateMemSize;
   U32                         CurrentPrivateMemBase;
   U32                         DesiredPrivateIOSize;
   U32                         CurrentPrivateIOSize;
   U32                         CurrentPrivateIOBase;
   U8                          reserved3[3];
   U8                          SyncByte;
} I2O_EXEC_STATUS_GET_REPLY, *PI2O_EXEC_STATUS_GET_REPLY;


/***************************************************************************/

/* ExecSysTabSet (System Table) Function Message Frame structure. */

#define I2O_EXEC_SYS_TAB_IOP_ID_LOCAL_IOP           0x000
#define I2O_EXEC_SYS_TAB_IOP_ID_LOCAL_HOST          0x001
#define I2O_EXEC_SYS_TAB_IOP_ID_UNKNOWN_IOP         0xFFF
#define I2O_EXEC_SYS_TAB_HOST_UNIT_ID_LOCAL_UNIT    0x0000
#define I2O_EXEC_SYS_TAB_HOST_UNIT_ID_UNKNOWN_UNIT  0xffff
#define I2O_EXEC_SYS_TAB_SEG_NUMBER_LOCAL_SEGMENT   0x000
#define I2O_EXEC_SYS_TAB_SEG_NUMBER_UNKNOWN_SEGMENT 0xfff

typedef struct _I2O_EXEC_SYS_TAB_SET_MESSAGE {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
   BF                          IOP_ID:I2O_IOP_ID_SZ;
   BF                          reserved:I2O_RESERVED_4BITS;
   BF                          HostUnitID:I2O_UNIT_ID_SZ;
   BF                          SegmentNumber:I2O_SEGMENT_NUMBER_SZ;
   BF                          reserved2:I2O_RESERVED_20BITS;
   I2O_SG_ELEMENT              SGL;
} I2O_EXEC_SYS_TAB_SET_MESSAGE, *PI2O_EXEC_SYS_TAB_SET_MESSAGE;


/****************************************************************************/

/* Operation Function Numbers */

#define   I2O_PARAMS_OPERATION_FIELD_GET               0x0001
#define   I2O_PARAMS_OPERATION_LIST_GET                0x0002
#define   I2O_PARAMS_OPERATION_MORE_GET                0x0003
#define   I2O_PARAMS_OPERATION_SIZE_GET                0x0004
#define   I2O_PARAMS_OPERATION_TABLE_GET               0x0005
#define   I2O_PARAMS_OPERATION_FIELD_SET               0x0006
#define   I2O_PARAMS_OPERATION_LIST_SET                0x0007
#define   I2O_PARAMS_OPERATION_ROW_ADD                 0x0008
#define   I2O_PARAMS_OPERATION_ROW_DELETE              0x0009
#define   I2O_PARAMS_OPERATION_TABLE_CLEAR             0x000A

/* Operations List Header */

typedef struct _I2O_PARAM_OPERATIONS_LIST_HEADER {
   U16                         OperationCount;
   U16                         Reserved;
} I2O_PARAM_OPERATIONS_LIST_HEADER, *PI2O_PARAM_OPERATIONS_LIST_HEADER;

/* Results List Header */

typedef struct _I2O_PARAM_RESULTS_LIST_HEADER {
   U16                         ResultCount;
   U16                         Reserved;
} I2O_PARAM_RESULTS_LIST_HEADER, *PI2O_PARAM_RESULTS_LIST_HEADER;

/* Read Operation Result Block Template Structure */

typedef struct _I2O_PARAM_READ_OPERATION_RESULT {
   U16                         BlockSize;
   U8                          BlockStatus;
   U8                          ErrorInfoSize;
   /*                          Operations Results          */
   /*                          Pad (if any)                */
   /*                          ErrorInformation (if any)   */
} I2O_PARAM_READ_OPERATION_RESULT, *PI2O_PARAM_READ_OPERATION_RESULT;

/* Operation Template for Specific Fields */

typedef struct _I2O_PARAM_OPERATION_SPECIFIC_TEMPLATE {
   U16                         Operation;
   U16                         GroupNumber;
   U16                         FieldCount;
   U16                         FieldIdx[1];
   /*                          Pad (if any)                    */
} I2O_PARAM_OPERATION_SPECIFIC_TEMPLATE, *PI2O_PARAM_OPERATION_SPECIFIC_TEMPLATE;

/* Operation Template for All Fields */

typedef struct _I2O_PARAM_OPERATION_ALL_TEMPLATE {
   U16                         Operation;
   U16                         GroupNumber;
   U16                         FieldCount;
   /*                          Pad (if any)                    */
} I2O_PARAM_OPERATION_ALL_TEMPLATE, *PI2O_PARAM_OPERATION_ALL_TEMPLATE;

/****************************************************************************/

/* Utility Message class functions. */

#define    I2O_UTIL_NOP                                0x00
#define    I2O_UTIL_ABORT                              0x01
#define    I2O_UTIL_CLAIM                              0x09
#define    I2O_UTIL_CLAIM_RELEASE                      0x0B
#define    I2O_UTIL_CONFIG_DIALOG                      0x10
#define    I2O_UTIL_DEVICE_RESERVE                     0x0D
#define    I2O_UTIL_DEVICE_RELEASE                     0x0F
#define    I2O_UTIL_EVENT_ACKNOWLEDGE                  0x14
#define    I2O_UTIL_EVENT_REGISTER                     0x13
#define    I2O_UTIL_LOCK                               0x17
#define    I2O_UTIL_LOCK_RELEASE                       0x19
#define    I2O_UTIL_PARAMS_GET                         0x06
#define    I2O_UTIL_PARAMS_SET                         0x05
#define    I2O_UTIL_REPLY_FAULT_NOTIFY                 0x15

/****************************************************************************/

/* UtilNOP Function Message Frame structure. */

typedef struct _I2O_UTIL_NOP_MESSAGE {
    I2O_MESSAGE_FRAME          StdMessageFrame;
} I2O_UTIL_NOP_MESSAGE, *PI2O_UTIL_NOP_MESSAGE;


/*************************************************************************/

/* UtilParamsGet Message Frame structure. */

typedef struct _I2O_UTIL_PARAMS_GET_MESSAGE {
    I2O_MESSAGE_FRAME          StdMessageFrame;
    I2O_TRANSACTION_CONTEXT    TransactionContext;
    U32                        OperationFlags;
    I2O_SG_ELEMENT             SGL;
} I2O_UTIL_PARAMS_GET_MESSAGE, *PI2O_UTIL_PARAMS_GET_MESSAGE;


/****************************************************************************/
/* GROUP Parameter Groups */
/****************************************************************************/

/* GROUP Configuration and Operating Structures and Defines */

/* Groups Numbers */

#define    I2O_UTIL_PARAMS_DESCRIPTOR_GROUP_NO          0xF000
#define    I2O_UTIL_PHYSICAL_DEVICE_TABLE_GROUP_NO      0xF001
#define    I2O_UTIL_CLAIMED_TABLE_GROUP_NO              0xF002
#define    I2O_UTIL_USER_TABLE_GROUP_NO                 0xF003
#define    I2O_UTIL_PRIVATE_MESSAGE_EXTENSIONS_GROUP_NO 0xF005
#define    I2O_UTIL_AUTHORIZED_USER_TABLE_GROUP_NO      0xF006
#define    I2O_UTIL_DEVICE_IDENTITY_GROUP_NO            0xF100
#define    I2O_UTIL_DDM_IDENTITY_GROUP_NO               0xF101
#define    I2O_UTIL_USER_INFORMATION_GROUP_NO           0xF102
#define    I2O_UTIL_SGL_OPERATING_LIMITS_GROUP_NO       0xF103
#define    I2O_UTIL_SENSORS_GROUP_NO                    0xF200

/* UTIL Group F000h - GROUP DESCRIPTORS Parameter Group */

#define    I2O_UTIL_GROUP_PROPERTIES_GROUP_TABLE       0x01
#define    I2O_UTIL_GROUP_PROPERTIES_ROW_ADDITION      0x02
#define    I2O_UTIL_GROUP_PROPERTIES_ROW_DELETION      0x04
#define    I2O_UTIL_GROUP_PROPERTIES_CLEAR_OPERATION   0x08

/* UTIL Group F100h - Device Identity Parameter Group */

typedef struct _I2O_UTIL_DEVICE_IDENTITY_SCALAR {
   U32                         ClassID;
   U16                         OwnerTID;
   U16                         ParentTID;
   U8                          VendorInfo[I2O_DEVID_VENDOR_INFO_SZ];
   U8                          ProductInfo[I2O_DEVID_PRODUCT_INFO_SZ];
   U8                          Description[I2O_DEVID_DESCRIPTION_SZ];
   U8                          ProductRevLevel[I2O_DEVID_REV_LEVEL_SZ];
   U8                          SNFormat;
   U8                          SerialNumber[I2O_MAX_SERIAL_NUMBER_SZ];
} I2O_UTIL_DEVICE_IDENTITY_SCALAR, *PI2O_UTIL_DEVICE_IDENTITY_SCALAR;

/* UTIL Group F101h - DDM Identity Parameter Group */

typedef struct _I2O_UTIL_DDM_IDENTITY_SCALAR {
   U16                         DdmTID;
   U8                          ModuleName[I2O_MODULE_NAME_SZ];
   U8                          ModuleRevLevel[I2O_DEVID_REV_LEVEL_SZ];
   U8                          SNFormat;
   U8                          SerialNumber[I2O_MAX_SERIAL_NUMBER_SZ];
} I2O_UTIL_DDM_IDENTITY_SCALAR, *PI2O_UTIL_DDM_IDENTITY_SCALAR;

/****************************************************************************/

/* Block Storage Parameter Groups */

#define     I2O_BSA_DEVICE_INFO_GROUP_NO                0x0000
#define     I2O_BSA_OPERATIONAL_CONTROL_GROUP_NO        0x0001
#define     I2O_BSA_POWER_CONTROL_GROUP_NO              0x0002
#define     I2O_BSA_CACHE_CONTROL_GROUP_NO              0x0003
#define     I2O_BSA_MEDIA_INFO_GROUP_NO                 0x0004
#define     I2O_BSA_ERROR_LOG_GROUP_NO                  0x0005

/***************************************************************************/

/* I2O Block Storage Reply Message Frame Template */

typedef struct _I2O_BSA_REPLY_MESSAGE_FRAME {
    I2O_MESSAGE_FRAME           StdMessageFrame;
    I2O_TRANSACTION_CONTEXT     TransactionContext;
    U16                         DetailedStatusCode;
    U8                          RetryCount;
    U8                          ReqStatus;
/*                              ReplyPayload        */
} I2O_BSA_REPLY_MESSAGE_FRAME, *PI2O_BSA_REPLY_MESSAGE_FRAME;

/**************************************************************************/

/* Block Storage Group 0000h - Device Information Parameter Group */

typedef struct _I2O_BSA_DEVICE_INFO_SCALAR {
    U8          DeviceType;
    U8          NumberOfPaths;
    U16         PowerState;     
    U32         BlockSize;
    U64         DeviceCapacity;
    U32         DeviceCapabilitySupport;
    U32         DeviceState;
} I2O_BSA_DEVICE_INFO_SCALAR, *PI2O_BSA_DEVICE_INFO_SCALAR;


/****************************************************************************/


/* ExecOutboundInit Function Message Frame structure. */

typedef struct _I2O_EXEC_OUTBOUND_INIT_MESSAGE {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
   U32                         HostPageFrameSize;
   U8                          InitCode;
   U8                          reserved;
   U16                         OutboundMFrameSize;
   I2O_SG_ELEMENT              SGL;
} I2O_EXEC_OUTBOUND_INIT_MESSAGE, *PI2O_EXEC_OUTBOUND_INIT_MESSAGE;


#define    I2O_EXEC_OUTBOUND_INIT_IN_PROGRESS          0x01
#define    I2O_EXEC_OUTBOUND_INIT_REJECTED             0x02
#define    I2O_EXEC_OUTBOUND_INIT_FAILED               0x03
#define    I2O_EXEC_OUTBOUND_INIT_COMPLETE             0x04

#define    I2O_EXEC_OUTBOUND_INIT_RESERVED_SZ          3


typedef struct _I2O_EXEC_OUTBOUND_INIT_STATUS {
   U8                          InitStatus;
   U8                          reserved[I2O_EXEC_OUTBOUND_INIT_RESERVED_SZ];
} I2O_EXEC_OUTBOUND_INIT_STATUS, *PI2O_EXEC_OUTBOUND_INIT_STATUS;


typedef struct _I2O_EXEC_OUTBOUND_INIT_RECLAIM_LIST {
   U32                         MFACount;
   U32                         MFAReleaseCount;
   U32                         MFAAddress[1];
} I2O_EXEC_OUTBOUND_INIT_RECLAIM_LIST, *PI2O_EXEC_OUTBOUND_INIT_RECLAIM_LIST;

/****************************************************************************/


/* ExecSysEnable Function Message Frame structure. */

typedef struct _I2O_EXEC_SYS_ENABLE_MESSAGE {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
} I2O_EXEC_SYS_ENABLE_MESSAGE, *PI2O_EXEC_SYS_ENABLE_MESSAGE;


/****************************************************************************/

/* ExecLCTNotify Function Message Frame structure. */

typedef struct _I2O_EXEC_LCT_NOTIFY_MESSAGE {
   I2O_MESSAGE_FRAME           StdMessageFrame;
   I2O_TRANSACTION_CONTEXT     TransactionContext;
   U32                         ClassIdentifier;
   U32                         LastReportedChangeIndicator;
   I2O_SG_ELEMENT              SGL;
} I2O_EXEC_LCT_NOTIFY_MESSAGE, *PI2O_EXEC_LCT_NOTIFY_MESSAGE;

/****************************************************************************/

/* ExecSysTabSet (System Table) Header Reply structure. */

#define    I2O_SET_SYSTAB_RESERVED_SZ                  8

typedef struct _I2O_SET_SYSTAB_HEADER {
   U8                          NumberEntries;
   U8                          SysTabVersion;
   U16                         reserved;
   U32                         CurrentChangeIndicator;
   U8                          reserved1[I2O_SET_SYSTAB_RESERVED_SZ];
/*    I2O_SYSTAB_ENTRY    SysTabEntry[1]; */
} I2O_SET_SYSTAB_HEADER, *PI2O_SET_SYSTAB_HEADER;


#define    I2O_RESOURCE_MANAGER_VERSION        0

typedef struct _MESSENGER_INFO {
   U32                         InboundMessagePortAddressLow;
   U32                         InboundMessagePortAddressHigh;
   } I2O_MESSENGER_INFO, *PI2O_MESSENGER_INFO;

/* ExecSysTabSet IOP Descriptor Entry structure. */

typedef struct _I2O_IOP_ENTRY {
   U16                         OrganizationID;
   U16                         reserved;
   BF                          IOP_ID:I2O_IOP_ID_SZ;
   BF                          reserved1:I2O_RESERVED_20BITS;
   BF                          SegmentNumber:I2O_SEGMENT_NUMBER_SZ;
   BF                          I2oVersion:I2O_4BIT_VERSION_SZ;
   BF                          IopState:I2O_IOP_STATE_SZ;
   BF                          MessengerType:I2O_MESSENGER_TYPE_SZ;
   U16                         InboundMessageFrameSize;
   U16                         reserved2;
   U32                         LastChanged;
   U32                         IopCapabilities;
   I2O_MESSENGER_INFO          MessengerInfo;              
} I2O_IOP_ENTRY, *PI2O_IOP_ENTRY;


/****************************************************************************/


#define    I2O_EXEC_IOP_RESET_RESERVED_SZ              16

#define    I2O_EXEC_IOP_RESET_IN_PROGRESS              0x01
#define    I2O_EXEC_IOP_RESET_REJECTED                 0x02

#define    I2O_EXEC_IOP_RESET_STATUS_RESERVED_SZ       3

typedef struct _I2O_EXEC_IOP_RESET_STATUS {
   U8                          ResetStatus;
   U8                          reserved[I2O_EXEC_IOP_RESET_STATUS_RESERVED_SZ];
} I2O_EXEC_IOP_RESET_STATUS, *PI2O_EXEC_IOP_RESET_STATUS;


/* ExecIopReset Function Message Frame structure. */

typedef struct _I2O_EXEC_IOP_RESET_MESSAGE {
   U8                          VersionOffset;
   U8                          MsgFlags;
   U16                         MessageSize;
   BF                          TargetAddress:I2O_TID_SZ;
   BF                          InitiatorAddress:I2O_TID_SZ;
   BF                          Function:I2O_FUNCTION_SZ;
   U8                          Reserved[I2O_EXEC_IOP_RESET_RESERVED_SZ];
   U32                         StatusWordLowAddress;
   U32                         StatusWordHighAddress;
} I2O_EXEC_IOP_RESET_MESSAGE, *PI2O_EXEC_IOP_RESET_MESSAGE;


/****************************************************************************/
/* EXEC Group 0001h - IOP Message Interface Parameter Group */

/* InitCode defines */
#define    I2O_MESSAGE_IF_INIT_CODE_NO_OWNER           0x00
#define    I2O_MESSAGE_IF_INIT_CODE_BIOS               0x10
#define    I2O_MESSAGE_IF_INIT_CODE_OEM_BIOS_EXTENSION 0x20
#define    I2O_MESSAGE_IF_INIT_CODE_ROM_BIOS_EXTENSION 0x30
#define    I2O_MESSAGE_IF_INIT_CODE_OS                 0x80

/****************************************************************************/

#endif	/* _I2ODEF_H_ */
