
#include "pcopy-opts.h"

#define HDR_LINES 2
#define ENTRY_LINE  (HDR_LINES + 2 * fseg->idx)

static void
short_read(file_seg_t * fseg, ssize_t rdlen, ssize_t rdct)
{
    static char const fmt[] =
        "th %2d read %d bytes not %d at 0x%08lX\n";
    char msgbf[sizeof(fmt) + 64];
    sprintf(msgbf, fmt,
            fseg->idx, (int)rdct, (int)rdlen, fseg->start);
    fserr(PCOPY_EXIT_FS_ERR_IN, msgbf, fseg->fname);
    /* NOTREACHED */
}

static void *
do_copy(void * arg)
{
    static char const st_fmt[]  = "th %2d reading 0x%08lX";
    static char const prg_fmt[] =  "      read to 0x%08lX togo 0x%08lX";
    file_seg_t * fseg = arg;
    size_t const ttl_size = fseg->end - fseg->start;
    size_t const readlim  = (ttl_size > 16 * 1024) ? (16 * 1024) : ttl_size;

    FILE * cur_fp = fopen(fseg->fname, "r");

    if (cur_fp == NULL)
        fserr(PCOPY_EXIT_FS_ERR_IN, "fopen 'r'", fseg->fname);

    pthread_mutex_lock(&tty_mutex);
    if (fseg->idx == 0) {
        move(0,0);
        printw("copying %s", fseg->fname);
        move(1,0);
        printw("copy to %s", fseg->seg_name);
    }

    move(ENTRY_LINE, 0);
    printw(st_fmt, fseg->idx, fseg->start);
    refresh();
    pthread_mutex_unlock(&tty_mutex);

    {
        int fdin  = fileno(cur_fp);
        int fdout = open(fseg->seg_name, O_RDWR | O_CREAT, 0600);

        if (fdout < 0)
            fserr(PCOPY_EXIT_FS_ERR_OUT, "open (w)", fseg->seg_name);

        if (fseg->start > 0) {
            if (lseek(fdin,  fseg->start, SEEK_SET) != fseg->start)
                fserr(PCOPY_EXIT_FS_ERR_IN, "seek (in)", fseg->fname);

            if (lseek(fdout, fseg->start, SEEK_SET) != fseg->start)
                fserr(PCOPY_EXIT_FS_ERR_OUT, "seek (out)", fseg->seg_name);
        }

        while (fseg->start < fseg->end) {
            ssize_t rdlen = fseg->end - fseg->start;
            if (rdlen > readlim)
                rdlen = readlim;
            ssize_t rdct = read(fdin, fseg->buffer, rdlen);
            if (rdct <= 0)
                short_read(fseg, rdlen, rdct);

            rdlen = write(fdout, fseg->buffer, rdct);
            if (rdlen != rdct)
                fserr(PCOPY_EXIT_FS_ERR_OUT, "write", fseg->seg_name);

            fseg->start += rdlen;

            pthread_mutex_lock(&tty_mutex);
            move(ENTRY_LINE + 1, 0);
            printw(prg_fmt, fseg->start, fseg->end - fseg->start);
            refresh();
            pthread_mutex_unlock(&tty_mutex);
        }
    }

    pthread_mutex_lock(&tty_mutex);
    move(ENTRY_LINE + 1, 0);
    clrtoeol();
    move(ENTRY_LINE, 0);
    clrtoeol();
    printw("th %2d DONE - copied 0x%08X bytes", fseg->idx, ttl_size);
    refresh();
    pthread_mutex_unlock(&tty_mutex);
    free(arg);
    pthread_exit(0);
}

static inline char const *
basenm(char const * f)
{
    char const * res = strrchr(f, '/');
    return res ? (res + 1) : f;
}

file_seg_t *
new_file_seg(char const * fnm, size_t start, size_t seg_len)
{
    static char const too_many[] =
        "The %s destination is a regular file, yet there are multiple inputs";
    static char const bad_dest[] =
        "The %s destination is neither a file nor a directory";
    static char const no_dir[] =
        "the directory portion of the destination is not a directory:\n\t%s";

    static bool been_here = false;

    char const * bfile = basenm(fnm);
    file_seg_t * res;
    size_t len = sizeof(*res) + seg_len;
    char const * dname = OPT_ARG(DESTINATION);

    enum {
        NMIS_UNK,
        NMIS_F_ONLY,
        NMIS_D_ONLY,
        NMIS_F_N_D
    } nm_type = NMIS_UNK;

    if (! HAVE_OPT(DESTINATION)) {
        len += strlen(bfile);
        nm_type = NMIS_F_ONLY;

    } else {
        struct stat sb;
        if (stat(dname, &sb) == 0) {
            if (S_ISDIR(sb.st_mode)) {
                len += strlen(dname) + strlen(bfile) + 1;
                nm_type = NMIS_F_N_D;

            } else if (S_ISREG(sb.st_mode)) {
                if (been_here)
                    die(PCOPY_EXIT_FAILURE, too_many, dname);
                been_here = true;
                unlink(dname);
                len += strlen(dname);
                nm_type = NMIS_D_ONLY;

            } else {
                die(PCOPY_EXIT_FS_ERR_OUT, bad_dest, dname);
                /* NOTREACHED */
            }

        } else {
            char * pe = strrchr(dname, '/');
            if (pe != NULL) {
                size_t l = pe - dname;
                char * d = malloc(l + 1);

                memcpy(d, dname, l);
                d[l] = NUL;
                if ((stat(d, &sb) != 0) || ! S_ISDIR(sb.st_mode))
                    die(PCOPY_EXIT_FS_ERR_OUT, no_dir, d);
                free(d);
            }
            len += strlen(dname);
            nm_type = NMIS_F_ONLY;
        }
    }

    res = malloc(len);
    if (res == NULL) {
        errno = ENOMEM;
        fserr(PCOPY_EXIT_NO_MEM, "alloc copy space", fnm);
    }

    memset(res, NUL, len);

    switch (nm_type) {
    case NMIS_F_ONLY:
        len = strlen(bfile);
        memcpy(res->seg_name, bfile, ++len);
        break;

    case NMIS_D_ONLY:
        len = strlen(dname);
        memcpy(res->seg_name, dname, ++len);
        break;

    case NMIS_F_N_D:
        len = sprintf(res->seg_name, "%s/%s", dname, bfile) + 1;
        
    default:
        break;
    }

    {
        uintptr_t addr = (uintptr_t)res->seg_name + len + 1 + sizeof(char *);
        addr = addr & ~(sizeof(char *) - 1);
        res->buffer = (void *)addr;
    }

    res->start  = start;
    res->end    = start + seg_len;
    res->fname  = fnm;
    return res;
}

void
pcopy(char const * fn)
{
    size_t base_size;
    size_t seg_size;
    size_t modulus;
    int    ix = 0;

    {
        struct stat sb;
        if (stat(fn, &sb) < 0)
            fserr(PCOPY_EXIT_FAILURE, "cannot re-stat", fn);
        if (! S_ISREG(sb.st_mode)) {
            errno = EINVAL;
            fserr(PCOPY_EXIT_FS_ERR_IN, "regular file check", fn);
        }
        base_size = sb.st_size;
        modulus   = sb.st_blksize;
    }

    {
        size_t sz = base_size / OPT_VALUE_THREAD_CT;

        /*
         * Make sure that thread-ct * size >= base_size
         */
        if ((sz * OPT_VALUE_THREAD_CT) != base_size)
            sz += modulus;

        /*
         * Now make sure sz is an even multiple of modulus
         */
        seg_size = modulus * (sz / modulus);
    }

    memset(pth_list, 0, OPT_VALUE_THREAD_CT * sizeof(*pth_list));

    for (modulus = 0; modulus < base_size; ix++) {
        file_seg_t * fseg = new_file_seg(fn, modulus, seg_size);

        modulus    += seg_size;
        fseg->end   = modulus;
        if (fseg->end > base_size)
            fseg->end = base_size;
        fseg->idx   = ix;
        pthread_create(pth_list + ix, NULL, do_copy, (void *)fseg);
    }

    for (int j = 0; j < ix; j++)
        pthread_join(pth_list[j], NULL);
}
