"""
Defines the `GridWorld` class.
"""

# import the appropriate classes
from geometry cimport Box, Point
from gravity cimport setInducedGF_box_nomask_nonrandom, GravityField
from numpy cimport ndarray as ar
import numpy as np
cimport cython

ctypedef unsigned long int size_t

cdef extern from "math.h":
    float ceil(float)

############################################################
# Some low-level inline functions to make sure the indexing is
# consistent

cdef inline int _idx_to_xi(int idx, int nx, int ny, int nz):
    return idx % (ny*nz)

cdef inline int _idx_to_yi(int idx, int nx, int ny, int nz):
    return (idx / nx) % nz

cdef inline int _idx_to_zi(int idx, int nx, int ny, int nz):
    return idx / (nx*ny)
    
@cython.boundscheck(True)
def _edges_from_center(a_o, tuple bounds = None):

    cdef ar[float] a = np.asarray(a_o, dtype= np.float32)
    cdef ar[float] r = np.empty( a.size + 1, dtype=np.float32)
    
    if r.size > 2:
        r[1:-1] = (a[:-1] + a[1:]) / 2

    if bounds is None:
        r[0]  = a[0]  - (r[1]  - a[0])
        r[-1] = a[-1] + (a[-1] - r[-2])
    else:
        r[0] = bounds[0]
        r[-1] = bounds[1]

######################################################################
# Now the gridworld class

cdef class GridWorld:
    """
    Defines a world that has a grid structure.  
    """
    
    # Class type declarations:
    cdef:
        ar x_edges, y_edges, z_edges

    def __init__(self, x_centers, y_centers, z_centers, 
                 tuple x_bounds = None, tuple y_bounds = None, tuple z_bounds = None):
        """
        Borkborkbork
        """

        self.x_edges = _edges_from_center(x_centers, x_bounds)
        self.y_edges = _edges_from_center(y_centers, y_bounds)
        self.z_edges = _edges_from_center(z_centers, z_bounds)
        
    
    @cython.boundscheck(False)
    def generateCSCoefficients(self, cs_class, mset, size_t samples_per_box):
        """
        Returns a CS problem of type `cs_class` with the coefficients
        from 
        """

        # Type the data
        cdef ar[float, ndim=3] mdata = mset.data
        cdef ar[float, ndim=2] mpts = mset.measure_pts

        # Init the output array
        cdef ar[float, mode="c", ndim=3] mT = \
            np.empty( (n_boxes, n_measurements), dtype=np.float32)

        # Get the temporary objects we need
        cdef Box curbox = Box()
        cdef Point measure_pt = Point()
        cdef GravityField gf = GravityField()

        # Get the gridsize:
        cdef size_t grid_size = <size_t>ceil(( <float> samples_per_box)**(1.0/3))

        # get the arrays for setting the box locations, along with the points
        cdef:
            ar[float, mode="c"] x_edges = self.x_edges
            ar[float, mode="c"] y_edges = self.y_edges
            ar[float, mode="c"] z_edges = self.z_edges
            Point p1=Point(), p2=Point()

        # Set up the indexing
        cdef size_t nx, ny, nz, xi, yi, zi, bi, mi, n_boxes, n_measurements

        nx = x_edges.shape[0] - 1
        ny = y_edges.shape[0] - 1
        nz = z_edges.shape[0] - 1
        n_boxes = nx*ny*nz
        n_measurements = mdata.shape[0]

        # Now we're ready to get going
        curbox.setDensity(0)

        # Now do the looping
        for bi from 0 <= bi < n_boxes:
            
            # Set up the box; this would be easier to cycle through
            # xi, yi, zi, then get the index, but they will probably
            # both get optimized to the same thing and this keeps
            # consistency with the other routines.

            xi = _idx_to_xi(bi, nx, ny, nz)
            yi = _idx_to_yi(bi, nx, ny, nz)
            zi = _idx_to_zi(bi, nx, ny, nz)

            p1.x = x_edges[xi]
            p2.x = x_edges[xi+1]
            p1.y = y_edges[yi]
            p2.y = y_edges[yi+1]
            p1.z = z_edges[zi]
            p2.z = z_edges[zi+1]

            curbox.setCoordsDirect(p1, p2)
            
            # Now get all the measurements

            for mi from 0 <= mi < n_measurements:
                measure_pt.set(mpts[mi, 0], mpts[mi, 1], mpts[mi, 2])
                
                # Set the gravity field
                setInducedGF_box_nomask_nonrandom(measure_pt, curbox, gf, grid_size)
                
                # C is going to be Txx - Tyy, and D is going to be
                # Txy.  This may be open to updating.
                mT[bi, mi, 0] = gf.Txx
                mT[bi, mi, 1] = gf.Txy
                mT[bi, mi, 2] = gf.Tyy

        # We're done, now run with it!
        return cs_class(mT)

