#!perl

=head1 NAME

Gen_DBD_foo_GetInfo - Generates a DBD::<foo>::GetInfo package.

=head1 SYNOPSIS

  set DBI_DSN=dbi:ODBC:...
  perl Gen_DBD_foo_GetInfo.pl > GetInfo.pm

=head1 DESCRIPTION

This script generates a DBD::<foo>::GetInfo package from the data source you
specified in the environment variable DBI_DSN.
DBD::<foo>::GetInfo should help a DBD author implementing the DBI get_info()
method.
Because you are just creating this package, it's very unlikly that DBD::<foo>
already provides a good implementation for get_info(). Thus you will probable
connect via DBD::ODBC.

=head1 NOTES

If you connect via DBD::ODBC, you should use version 0.38 or greater;

Please have a critical look at the data returned! ODBC driver vary dramatically
in their quality.

The generator assumes that most values are static and places these values
directly in the %info hash. A few examples show the use of CODE references
and the implementation via subroutines.
It's very likely that you have to write additional subroutines for values
depending on the session state or server version, e.g. SQL_DBMS_VER.

A possible implementation of DBD::<foo>::get_info() may look like:

  sub get_info {
    my($dbh, $info_type) = @_;
    require DBD::<foo>::GetInfo;
    my $v = $DBD::<foo>::GetInfo::info{int($info_type)};
    $v = $v->($dbh) if ref $v eq 'CODE';
    return $v;
  }

Please replace <foo> with the name of your driver.

=head1 SEE ALSO

DBI, DBD::ODBC

=head1 COPYRIGHT

Copyright (c) 2002 Steffen Goeldner. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut

use DBI();
use Local::SQL::CLI::Const::GetInfo();

our $VERSION = '0.01';

my $dbh = DBI->connect or die $DBI::errstr;
#  $dbh->{ RaiseError } = 1;
   $dbh->{ PrintError } = 1;

print <<'PERL';
package DBD::<foo>::GetInfo;

use DBD::<foo>();

my $fmt = '%02d.%02d.%1d%1d%1d%1d';   # ODBC version string: ##.##.#####

my $sql_driver_ver = sprintf $fmt, split (/\./, $DBD::<foo>::VERSION);

my @Keywords = qw(
PERL
{
  local $\ = "\n";
  local $, = "\n";
  print split /,/, $dbh->get_info( 89 );
}
print <<'PERL';
);

sub sql_data_source_name {
    my $dbh = shift;
    return 'dbi:<foo>:' . $dbh->{Name};
}
sub sql_keywords {
    return join ',', @Keywords;
}
sub sql_user_name {
    my $dbh = shift;
    return $dbh->{CURRENT_USER};
}

%info = (
PERL

my $h = \%Local::SQL::CLI::Const::GetInfo::InfoTypes;
my $Comma = ' ';

for ( sort keys %$h ) {
  my $nr = $h->{$_};
  my $Val = $dbh->get_info( $nr );
  $Comma = '#' unless defined $Val;
  printf "$Comma %5d => ", $nr;
  if ( $_ eq 'SQL_DATA_SOURCE_NAME') {
    $Val = '\&sql_data_source_name';
  }
  elsif ( $_ eq 'SQL_KEYWORDS') {
    $Val = '\&sql_keywords';
  }
  elsif ( $_ eq 'SQL_DRIVER_VER') {
    $Val = '$sql_driver_ver';
  }
  elsif ( $_ eq 'SQL_USER_NAME') {
    $Val = '\&sql_user_name';
  }
  elsif ( not defined $Val ) {
    $Val = 'undef';
  }
  elsif ( $Val eq '') {
    $Val = "''";
  }
  elsif ( $Val =~ /\D/) {
    $Val =~ s/\\/\\\\/g;
    $Val =~ s/'/\\'/g;
    $Val = "'$Val'";
  }
  printf '%-30s', $Val;
}
continue {
  $Comma = ',';
  print '  # ', $_, "\n";
}

print <<'PERL';
);

1;
PERL
