/*
 * sort_pkgs.c - sort one Packages file by ascending version number
 *
 * Copyright © 2013 J.-H. Chatenet <jhcha54008@free.fr>
 * 
 * This include some code from pkgdetails.c (package base-installer). 
 * base-installer is written by Tollef Fog Heen <tfheen@debian.org>.
 * This package is under the GNU General Public License, version 2, which
 * can usually be found in /usr/share/common-licenses/GPL-2 on Debian
 * systems.
 *
 * For the remainder :
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "dpkg/version.h"
#include "dpkg/parsehelp.h"

#define MAX_LINE 1000

char buf[MAX_LINE];
FILE *f;

struct version_entry {
    long stanza_begin;
    long stanza_end;
    struct dpkg_version *version;
    struct version_entry *next;
};

struct name_node {
    char *name;
    struct name_node *left;
    struct name_node *right;
    struct version_entry *versions;
};

struct name_node *letter_heads[26], *figure_heads[10];

static void oom_die(void)
{
    fputs("Out of memory!\n", stderr);
    exit(1);
}

static int compare_string(char *p1, char *p2) {
    while((*p1 != '\0') && (*p2 != '\0')) {
        if (*p1 > *p2)
            return +1;
        if (*p1 < *p2)
            return -1;
        p1++;
        p2++;
    }
    if ((*p1 == '\0') && (*p2 == '\0'))
        return 0;
    /* shorter strings are sorted after longer ones */
    if (*p1 == '\0')
        return +1;

    return -1;
}

static void output_stanza(long stanza_begin, long stanza_end) {
    long stanza_length;
    size_t n, n_read, n_written;

    if (fseek(f, stanza_begin, SEEK_SET) == -1) {
        fprintf(stderr, "error : fseek(f, %li)\n", stanza_begin);
        exit(-1);
    }

    stanza_length = stanza_end - stanza_begin;
    while(stanza_length > 0) {
        n = stanza_length > MAX_LINE ? MAX_LINE : stanza_length;
        n_read = fread(&buf, 1, n, f);
        if (n != n_read) {
            fprintf(stderr, "read %i of %i bytes\n", n_read, n);
            exit(-1);
        }
        n_written = fwrite(&buf, 1, n, stdout);
        if (n != n_written) {
            fprintf(stderr, "read %i of %i bytes to stdout\n", n_written, n);
            exit(-1);
        }
        stanza_length -= n;
    }
}

static void insert_version(struct name_node *old_node, struct version_entry *new_entry) {
    struct version_entry *ptr;

    if (dpkg_version_compare(new_entry->version, 
                             old_node->versions->version) <0) {
                new_entry->next = old_node->versions;
                old_node->versions = new_entry;
                return;
    } else {
        ptr = old_node->versions;
        while(ptr->next) {
            if (dpkg_version_compare(new_entry->version,
                ptr->next->version) <0) {
                new_entry->next = ptr->next;
                ptr->next = new_entry;
                return;
            }
            ptr = ptr->next;
        }
        ptr->next = new_entry;
    }
}

static void insert_in_bstree(char *cur_pkg, struct dpkg_version *pkgversion,
        long stanza_begin, long stanza_end) {
    int ret;
    struct version_entry *new_entry;
    struct name_node *new_node, *ptr, **pptr;

/* Allocate new entry */
    if ((new_entry = (struct version_entry *) malloc(sizeof(struct version_entry))) == NULL)
        oom_die();

    if ((new_node = (struct name_node *) malloc(sizeof(struct name_node))) == NULL)
        oom_die();

    new_node->name = strdup(cur_pkg);
    new_node->left = NULL;
    new_node->right = NULL;
    new_node->versions = new_entry;

    new_entry->version = pkgversion;
    new_entry->stanza_begin = stanza_begin;
    new_entry->stanza_end = stanza_end;
    new_entry->next = NULL;

/* Insert in list */
    if (isdigit(cur_pkg[0]))
       pptr = figure_heads + (cur_pkg[0] - '0');
    else
       pptr = letter_heads + (cur_pkg[0] - 'a');

    if (*pptr == NULL) {
        *pptr = new_node;
        return;
    }

    ptr = *pptr;
    while (ptr) {
        ret = compare_string(cur_pkg, ptr->name);
        if (ret < 0) {
            if (!ptr->left) {
                ptr->left = new_node;
                return;
            } else
                ptr = ptr->left;
        } else if (ret > 0) {
            if (!ptr->right) {
                ptr->right = new_node;
                return;
            } else
                ptr = ptr->right;
        } else { /* package names match */
            insert_version(ptr, new_entry);
            /* Clean unused new_node */
            free(new_node->name);
            free(new_node);
            return;
        }
    }
}

static void subbstree_walk(struct name_node* p){
    struct version_entry *pv;

    if (!p)
        return;

    if (p->left)
        subbstree_walk(p->left);

    if (p->versions) {
        pv = p->versions;
        while (pv) {
            output_stanza(pv->stanza_begin, pv->stanza_end);
            pv = pv->next;
        }
    }

    if (p->right)
        subbstree_walk(p->right);
}

static void bstree_walk(void) {
    int i;

    for (i=0; i < 10; i++)
        subbstree_walk(figure_heads[i]);

    for (i=0; i < 26; i++)
        subbstree_walk(letter_heads[i]);
}

static void verify_package_entry(char *cur_pkg, char *cur_version, long stanza_begin, long stanza_end) {
    const char *name_check_error_msg = NULL;
    int ret;
    struct dpkg_version *pkgversion = NULL;

    name_check_error_msg = pkg_name_is_illegal(cur_pkg);
    if (name_check_error_msg) {
        fprintf(stderr, "package name : %s %s\n", cur_pkg, name_check_error_msg);
        return;
    }

    pkgversion = malloc(sizeof(struct dpkg_version));
    if (pkgversion == NULL)
        oom_die();

    ret = parseversion(pkgversion, cur_version);
    if (ret == -1 || ! dpkg_version_is_informative(pkgversion)) {
        free(pkgversion);
        return;
    }

    insert_in_bstree(cur_pkg, pkgversion, stanza_begin, stanza_end);
}

static char *fieldcpy(char *dst, char *fld) {
    while (*fld && *fld != ':') 
        fld++;
    if (!*(fld++)) 
        return NULL;
    while (isspace(*fld)) fld++;
    return strcpy(dst, fld);
}

int main(int argc, char *argv[]) {

    char buf[MAX_LINE];
    char cur_pkg[MAX_LINE];
    char cur_version[MAX_LINE];
    int next_line_is_continuation = 0, this_line_is_continuation = 0;
    long stanza_begin = 0, stanza_end;

    if (argc != 2) {
        fprintf(stderr, "usage: %s packagesfile\n", argv[0]);
        exit(1);
    }

    f = fopen(argv[1], "r");
    if (f == NULL) {
        perror(argv[1]);
        exit(1);
    }

    {
        int i;
        for (i=0; i<26; i++)
            letter_heads[i] = NULL;

        for (i=0; i<10; i++)
            figure_heads[i] = NULL;
    }

/* Packages file analysis */

    while (fgets(buf, sizeof(buf), f)) {
        if (next_line_is_continuation)
            this_line_is_continuation = 1;
        else
            this_line_is_continuation = 0;
        if (*buf && buf[strlen(buf)-1] == '\n') {
                buf[strlen(buf)-1] = '\0';
                next_line_is_continuation = 0;
            } else
                next_line_is_continuation = 1;
        if (strncasecmp(buf, "Package:", 8) == 0) {
            fieldcpy(cur_pkg, buf);
        } else if (strncasecmp(buf, "Version:", 8) == 0) {
            fieldcpy(cur_version, buf);
        } else if (! this_line_is_continuation && *buf == '\0') {
            stanza_end = ftell(f);
        /* stanza_end is the first char after the blank line (usually 'P' of "Package: ") */
            if (stanza_end == -1) {
                perror("ftell");
                exit(1);
            }
            verify_package_entry(cur_pkg, cur_version, stanza_begin, stanza_end);
            cur_pkg[0] = '\0';
            cur_version[0] = '\0';
            stanza_begin = stanza_end;
        }
    }

/* Output */
    bstree_walk();
}

