
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <glob.h>
#include <regex.h>
#include <cdebconf/debconfclient.h>

#define DEBCONF_BASE "di-utils-mkfs/"
#define MAX_LINE_LENGTH (100)
#define GLOB_MKFS "/sbin/mkfs."

struct debconfclient *debconf;

/* proto types */
int main(int argc, char *argv[]);
void log(const char *msg);
char *debconf_ask(char *prio, char *template, char *value);
char *get_partitons();
char *get_filesystems();
char *ask_partitions();
char *ask_filesystem();
int ask_confirm(const char *parts);
int do_mkfs(const char *parts, const char *fs);
int check_mount(const char *part);
int check_type(const char *part);

void log(const char *msg) {
	FILE *test;

	printf("%s\n", msg);

	test = fopen("/var/log/messages", "a");
	if(test == NULL)
		return;
	fprintf(test, "%s\n", msg);
	fclose(test);
}

char *get_partitions() {
	int cline = 0;
	FILE *fs = NULL;
	char line[MAX_LINE_LENGTH];
	int len, size = 1;
	char *ret = NULL;
	regex_t preg;

	fs = fopen("/proc/partitions", "r");
	if(fs == NULL)
		return(NULL);

	if(regcomp(&preg, "/disc$", REG_EXTENDED|REG_NEWLINE) != 0)
		return(NULL);

	while(fgets(line, MAX_LINE_LENGTH, fs) > 0) {
		char *pos = NULL;
		char *device = NULL;

		if(++cline < 3)		/* skip the first two lines */
			continue;

		pos = strrchr(line, 32);
		if(pos == NULL)
			continue;
		pos = pos+sizeof(char);

		device = (char*) malloc(sizeof(char)*
			(strlen("/dev/") + (strlen(pos)-1)));
		if(device == NULL)
			return(NULL);

		strcpy(device, "/dev/");
		strncat(device, pos, strlen(pos)-1);

		if(regexec(&preg, device, 0, NULL, 0) == 0)
			continue;

		if(check_mount(device) != 0)
			continue;

		/* TODO: doesn't work */
		if(check_type(device) == 0)
			continue;

		len = strlen(device);
		size += len;
		ret = realloc(ret, size+(2*sizeof(char)));
		memcpy(ret+size-len-sizeof(char), device, len);
		ret[size++ -1] = ',';

		free(device);
	}
	ret[size-2] = '\0';

	fclose(fs);
	regfree(&preg);

	return(ret);
}

char *get_filesystems() {
	glob_t glres;
	int c = 0;
	int len, size = 1;
	char *ret = NULL;

	glob(GLOB_MKFS "*", 0, NULL, &glres);

	if(glres.gl_pathc == 0)
		return(NULL);
	
	for(c=0; c<glres.gl_pathc; c++) {
		char *pos = NULL;

		if(access(glres.gl_pathv[c], R_OK|X_OK))
			continue;
		pos = strrchr(glres.gl_pathv[c], '.');
		if(pos == NULL)
			continue;
		pos = pos+sizeof(char);

		len = strlen(pos);
		size += len;
		ret = realloc(ret, size+(2*sizeof(char)));
		memcpy(ret+size-len-sizeof(char), pos, len);
		ret[size++ -1] = ',';
	}
	ret[size-2] = '\0';

	globfree(&glres);
	return(ret);
}

char *ask_partitions() {
	char *parts = NULL;

	parts = get_partitions();
	if(parts == NULL) {
		debconf->command(debconf, "INPUT", "high",
			DEBCONF_BASE "noparts", NULL);
		return(NULL);
	}

	debconf->command(debconf, "SUBST", DEBCONF_BASE "partitions",
		"PARTITIONS", parts, NULL);
	debconf->command(debconf, "FSET", DEBCONF_BASE "partitions",
		"seen", "false", NULL);
	debconf->command(debconf, "INPUT", "high",
		DEBCONF_BASE "partitions", NULL);
	debconf->command(debconf, "GO", NULL);
	debconf->command(debconf, "GET", DEBCONF_BASE "partitions", NULL);

	/* nothing selected */
	if(strlen(debconf->value) == 0) {
		debconf->command(debconf, "FSET", DEBCONF_BASE "noselection",
			"seen", "false", NULL);
		debconf->command(debconf, "INPUT", "high",
			DEBCONF_BASE "noselection", NULL);
		exit(EXIT_SUCCESS);
	}

	return(debconf->value);
}

char *ask_filesystem() {
	char *fs = NULL;

	fs = get_filesystems();
	if(fs == NULL) {
		debconf->command(debconf, "INPUT", "high",
			DEBCONF_BASE "nomkfs", NULL);
		return(NULL);
	}

	debconf->command(debconf, "SUBST", DEBCONF_BASE "filesystem",
		"FILESYSTEMS", fs, NULL);
	debconf->command(debconf, "FSET", DEBCONF_BASE "filesystem",
		"seen", "false", NULL);
	debconf->command(debconf, "INPUT", "high",
		DEBCONF_BASE "filesystem", NULL);
	debconf->command(debconf, "GO", NULL);
	debconf->command(debconf, "GET", DEBCONF_BASE "filesystem", NULL);

	return(debconf->value);
}

int ask_confirm(const char *parts) {
	debconf->command(debconf, "SUBST", DEBCONF_BASE "confirm",
		"PARTITIONS", parts, NULL);
	debconf->command(debconf, "FSET", DEBCONF_BASE "confirm",
		"seen", "false", NULL);
	debconf->command(debconf, "INPUT", "high",
		DEBCONF_BASE "confirm", NULL);
	debconf->command(debconf, "GO", NULL);
	debconf->command(debconf, "GET", DEBCONF_BASE "confirm", NULL);

	if(strstr(debconf->value, "true"))
		return(EXIT_SUCCESS);
	return(EXIT_FAILURE);
}

int do_mkfs(const char *parts, const char *fs) {
	char *new = NULL;

	if((parts == NULL) || (fs == NULL))
		return(EXIT_FAILURE);

	new = (char*)parts;
	while(1) {
		char *part = NULL;
		int ret = 0;

		part = malloc(sizeof(char)*1024);
		if(part == NULL)
			exit(EXIT_FAILURE);

		strcpy(part, GLOB_MKFS);
		sprintf(part, "%s%s ", GLOB_MKFS, fs);

		if(strchr(new, ',') != NULL)
			strncat(part, new, strchr(new, ',')-new);
		else
			strncat(part, new, strlen(new));

		log(part);
		strcat(part, " 1>>/var/log/messages 2>&1");
		ret = system(part);
		if(ret != 0) {
			debconf->command(debconf, "SUBST", DEBCONF_BASE "failed",
				"COMMAND", part, NULL);
			debconf->command(debconf, "FSET", DEBCONF_BASE "failed",
				"seen", "false", NULL);
			debconf->command(debconf, "INPUT", "high",
				DEBCONF_BASE "failed", NULL);
			return(EXIT_FAILURE);
		}

		free(part);

		new = strchr(new, ',');
		if(new == NULL)
			break;
		new++;
	}

	return(EXIT_SUCCESS);
}

int check_mount(const char *part) {
	FILE *mounts = NULL;
	char line[MAX_LINE_LENGTH];
	int ret = 0;

	mounts = fopen("/proc/mounts", "r");
	if(mounts == NULL)
		return(1);

	while(fgets(line, MAX_LINE_LENGTH, mounts) > 0) {
		char *pos, *dev;

		pos = strchr(line, 32);
		if(pos == NULL)
			continue;

		dev = (char*) malloc(sizeof(char)*(pos-line+1));
		if(dev == NULL)
			return(1);

		strncpy(dev, line, pos-line);
		dev[pos-line] = '\0';

		if(!strcasecmp(dev, part)) {
			free(dev);
			ret = 1;
			break;
		}

		free(dev);
	}

	fclose(mounts);
	return(ret);
}

int check_type(const char *part) {
	regex_t preg;
	char disc_str[] = "disc";
	char *pos = NULL, *disc = NULL;
	FILE *grep = NULL;
	char line[MAX_LINE_LENGTH];
	char sfdisk[MAX_LINE_LENGTH];

	/* check, if we have real discs (scsi/ide/...) */
	if(regcomp(&preg, "(ide|scsi)", REG_EXTENDED|REG_NEWLINE) != 0)
		return(1);
	if(regexec(&preg, part, 0, NULL, 0) != 0) {
		regfree(&preg);
		return(0);
	}
	regfree(&preg);

	pos = strrchr(part, '/');
	if(pos == NULL)
		return(1);
	pos++;

	disc = (char*) malloc(sizeof(char)*((pos-part)+strlen(disc_str)+1));
	if(disc == NULL)
		return(1);

	strncpy(disc, part, pos-part);
	strcat(disc, disc_str);

	snprintf(sfdisk, MAX_LINE_LENGTH, "/usr/sbin/sfdisk -d %s 2>/dev/null", disc);
	grep = popen(sfdisk, "r");
	if(grep == NULL)
		return(1);

	while(fgets(line, MAX_LINE_LENGTH, grep) > 0) {
		char *regstr = NULL;

		regstr = (char*)malloc(sizeof(char)*
			(strlen(part)+strlen("83")+8));
		if(regstr == NULL)
			return(1);
		sprintf(regstr, "^%s.*Id=%d", part, 83);

		if(regcomp(&preg, regstr, REG_EXTENDED|REG_NEWLINE) != 0)
			return(1);
		if(regexec(&preg, line, 0, NULL, 0) == 0) {
			/* TODO: segfault
			xfree(regstr);
			regfree(&preg);
			pclose(grep);
			free(disc);
			*/
			return(0);
		}

		free(regstr);
		regfree(&preg);
	}

	pclose(grep);
	free(disc);
	return(1);
}

int main(int argc, char *argv[]) {
	char *parts = NULL;
	char *fs = NULL;

	debconf = debconfclient_new();
	debconf->command(debconf, "CAPB", "backup", NULL);
	debconf->command(debconf, "TITLE", "Create filesystem", NULL);

	parts = strdup(ask_partitions());
	fs = strdup(ask_filesystem());

	/* ask the user and then mkfs the partitions */
	if(ask_confirm(parts) == EXIT_SUCCESS)
		do_mkfs(parts, fs);

	return(EXIT_SUCCESS);
}

