/*
 * Copyright (C) 2002, Nathan Hawkins
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY NATHAN HAWKINS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JULIE HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>

static int pw_parse_line(char *buf,size_t buflen,struct passwd *resbuf)
{
	int cfields;
	char *fields[10]={NULL};
	char *temp;

	buf[buflen-1]='\0';	/* get rid of \n */
	fields[0]=buf;
	for(cfields=1;cfields<10;cfields++) {
		temp=strchr(fields[cfields-1],':');
		if(temp==NULL) 
			break;
		fields[cfields]=temp+1;
		*temp='\0';
	}
	resbuf->pw_name=fields[0];
	resbuf->pw_passwd=fields[1];
	resbuf->pw_uid=atoi(fields[2]);
	resbuf->pw_gid=atoi(fields[3]);
	switch(cfields) {
	case 10:
		/* FreeBSD master.passwd file format */
		resbuf->pw_class=fields[4];
		resbuf->pw_change=atol(fields[5]);
		resbuf->pw_expire=atol(fields[6]);
		resbuf->pw_gecos=fields[7];
		resbuf->pw_dir=fields[8];
		resbuf->pw_shell=fields[9];
		break;
	case 7:
		/* Linux passwd file format */
		resbuf->pw_class=&buf[buflen]; /* pointer to NUL */
		resbuf->pw_change=0;
		resbuf->pw_expire=0;
		resbuf->pw_gecos=fields[4];
		resbuf->pw_dir=fields[5];
		resbuf->pw_shell=fields[6];
		break;
	default:
		/* invalid format */
		return(0);
	}
	return(1);
}

static int gr_parse_line(char *buf,size_t linelen,size_t buflen,struct group *resbuf);

int fgetpwent_r (FILE *stream,struct passwd *resbuf,char *buffer,size_t buflen, struct passwd **result)
{
	char *p;

	*result=NULL;
	
	while(*result==NULL) {
		p=fgets(buffer,buflen,stream);

		if(p==NULL && feof(stream)) {
			errno=ENOENT;
			return(ENOENT);
		}
		/* skip leading spaces */
		while(isspace(*p))
			*p++;
		/* skip empty and comment lines */
		if(*p=='\0' || *p=='#')
			continue;
		if(pw_parse_line(buffer,strlen(buffer),resbuf))
			*result=resbuf;
	}
	return(0);
}

int fgetgrent_r (FILE *stream,struct group *resbuf,char *buffer,size_t buflen,
struct group **result)
{
	char *p;

	*result=NULL;
	
	while(*result==NULL) {
		p=fgets(buffer,buflen,stream);

		if(p==NULL && feof(stream)) {
			return(ENOENT);
		}
		/* skip leading spaces */
		while(isspace(*p))
			*p++;
		/* skip empty and comment lines */
		if(*p=='\0' || *p=='#')
			continue;
		if(gr_parse_line(buffer,strlen(buffer),buflen,resbuf))
			*result=resbuf;
	}
	return(0);
}

#ifdef TEST
int main(void)
{
	FILE *f;
	char buf[8192];
	struct group resbuf;
	struct group *result;
	char **gr_mem,*member;
	

	f=fopen("t","r");
	while(!fgetgrent_r(f,&resbuf,buf,(size_t)50,&result)) {
		printf("%s:%s:%d:",result->gr_name,result->gr_passwd,result->gr_gid);
		gr_mem=result->gr_mem;
		member=*gr_mem++;
		while(member!=NULL) {
			printf("%s",member);
			member=*gr_mem++;
		}
		printf("\n");
	}
}
#endif
static int gr_parse_line(char *buf,size_t linelen,size_t buflen,struct group *resbuf)
{
	int cfields;
	char *fields[4];
	char *temp,**member;

	buf[linelen-1]='\0';	/* get rid of \n */
	fields[0]=buf;
	for(cfields=1;cfields<4;cfields++) {
		temp=strchr(fields[cfields-1],':');
		if(temp==NULL) {
			break;
		}
		fields[cfields]=temp+1;
		*temp='\0';
	}
	resbuf->gr_name=fields[0];
	resbuf->gr_passwd=fields[1];
	resbuf->gr_gid=atoi(fields[2]);
	/* might need to think about pointer aligning this sometime */
	member=(char **)&buf[linelen];
	resbuf->gr_mem=member;
	*member=fields[3];
	while(1) {
		if(&member[1]>&buf[buflen]) {
			return(0);
		}
		temp=strchr(*member,',');
		member=&member[1];
		if(temp==NULL) {
			*member=temp;
			break;
		} else {
			*temp='\0';
			temp+=1;
			*member=temp;
		}
	}
	return(1);
}
