/*
 * Copyright (C) 2002, Nathan Hawkins
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY NATHAN HAWKINS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JULIE HAUGH OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <stdio.h>
#include <limits.h>
#include <time.h>
#include <pwd.h>
#include "shadow.h"

static struct spwd *pw2sp(struct passwd *pw,struct spwd *spw)
{
	time_t sectochg,now;

	if(pw==NULL)
		return(NULL);
	spw->sp_expire=pw->pw_expire?pw->pw_expire/86400:0;
	spw->sp_inact=0;
	spw->sp_namp=pw->pw_name;
	spw->sp_pwdp=pw->pw_passwd;

	if(pw->pw_change==0) {
		/* sane defaults for no change required */
		spw->sp_lstchg=0;
		spw->sp_max=INT_MAX;
		spw->sp_min=0;
		spw->sp_warn=7;
		spw->sp_inact=0;
	} else {
		spw->sp_warn=7;
		spw->sp_min=0;
		now=time(NULL);
		/* BSD passwd tracks when the password must be changed. The
		 * shadow routines track when the password was changed, and
		 * how often it must be changed. This code tries to fake out
		 * the aging.
		sectochg=(pw->pw_change-now); /* negative if already past time */
		if(sectochg<=0) {
			spw->sp_max=0;
			spw->sp_lstchg=pw->pw_change/86400;
		} else {
			spw->sp_max=sectochg/86400;
			spw->sp_lstchg=now/86400;
		}
	}
	return(spw);
}

int fgetspent_r (FILE *stream,struct spwd *resbuf,char *buffer,size_t buflen,
struct spwd **result)
{
	struct passwd pw,*pwp;
	struct spwd *sp;
	int rv;

	if((rv=fgetpwent_r(stream,&pw,buffer,buflen,&pwp))!=0)
		return(rv);
	pw2sp(pwp,resbuf);
	*result=resbuf;
	return(0);

}

static FILE *spstream;

static inline void sp_makeopen(void)
{
	if(spstream==NULL)
		spstream=fopen(_PATH_MASTERPASSWD,"r");
}

struct spwd *getspent(void)
{
	static struct spwd sp;
	static char buf[256];
	struct spwd *spp;
	int rv;

	sp_makeopen();
	if(spstream==NULL)
		return(NULL);
	rv=fgetspent_r(spstream,&sp,&buf,256,&spp);
	if(rv)
		return(NULL);
	return(spp);
}

void endspent(void)
{
	if(spstream==NULL)
		return;
	fclose(spstream);
	spstream=NULL;
}

void setspent(void)
{
	if(spstream==NULL)
		return;
	rewind(spstream);
}

/* didn't feel like doing a reentrant version of this... */
struct spwd *getspnam(const char *name)
{
	struct spwd *sp;
	setspent();
	for(;;) {
		sp=getspent();
		if(sp==NULL)
			return(sp);
		if(strcmp(name,sp->sp_namp)==0)
			return(sp);
	}
}

