package main

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promhttp"
)

const (
	lockFile    = "mail.lock"
	listFile    = "mail.list"
	listFileNew = "mail.list.new"
)

var (
	metrics = prometheus.NewHistogramVec(prometheus.HistogramOpts{
		Subsystem: "dovecotsim",
		Name:      "operations",
		Help:      "Simulated opeations",
	}, []string{"path", "op"})

	rmetrics prometheus.Observer
	wmetrics prometheus.Observer
)

func main() {
	prometheus.MustRegister(metrics)

	path := os.Args[1]
	if err := os.Chdir(path); err != nil {
		log.Fatal(err)
	}

	rmetrics = metrics.WithLabelValues(path, "read")
	wmetrics = metrics.WithLabelValues(path, "write")

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	go func() {
		sigchan := make(chan os.Signal, 1)
		signal.Notify(sigchan, syscall.SIGINT, syscall.SIGTERM)

		<-sigchan

		log.Print("Exiting")

		cancel()
	}()

	wg := &sync.WaitGroup{}
	wg.Add(1)
	go func() {
		defer wg.Done()

		w := &writer{}

		w.run(ctx)
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()

		w := &reader{}

		w.run(ctx)
	}()

	go http.ListenAndServe(":9090", promhttp.Handler())

	wg.Wait()
}
