package main

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promhttp"
)

const (
	lockFile    = "mail.lock"
	listFile    = "mail.list"
	listFileNew = "mail.list.new"
)

var (
	metrics = prometheus.NewHistogramVec(prometheus.HistogramOpts{
		Subsystem: "dovecotsim",
		Name:      "operations",
		Help:      "Simulated opeations",
	}, []string{"path", "inst", "op"})

	pmetrics prometheus.ObserverVec
)

func main() {
	prometheus.MustRegister(metrics)

	path := os.Args[1]
	if err := os.Chdir(path); err != nil {
		log.Fatal(err)
	}

	pmetrics = metrics.MustCurryWith(prometheus.Labels{"path": path})

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	go func() {
		sigchan := make(chan os.Signal, 1)
		signal.Notify(sigchan, syscall.SIGINT, syscall.SIGTERM)

		<-sigchan

		log.Print("Exiting")

		cancel()
	}()

	wg := &sync.WaitGroup{}

	log.Print("Starting readers and writers")

	for i := 0; i < 5; i++ {
		wg.Add(1)
		go func(instance int) {
			defer wg.Done()

			w := &reader{}

			w.run(ctx, instance)
		}(i)
	}

	for i := 0; i < 5; i++ {
		wg.Add(1)
		go func(instance int) {
			defer wg.Done()

			w := &writer{}

			w.run(ctx, instance)
		}(i)
	}

	go http.ListenAndServe(":9090", promhttp.Handler())

	wg.Wait()
}
