import java.io.File;

/**
 * Thread with native priority control.
 * On some systems, Thread.setPriority has no effect.
 * This class defines a new method, setNativePriority(),
 * to use native OS system calls to control Thread priority directly.
 * On systems with no native thread support, a call to
 * setNativePriority() will be equivalent to setPriority().
 */
public abstract class NativeThread extends Thread {

	/**
	 * library name
	 */
	private static final String JAVA_LIB_NAME = "libNativeThread.so";

	// constants to rescale from java Thread priority (from MIN_PRIORITY to MAX_PRIORITY)
	// to native thread priority (from 0 to 19).
	
	/**
	 * java Thread priority range.
	 */
	private static final int PRIO_RANGE = MAX_PRIORITY - MIN_PRIORITY;

	/**
	 * native thread priority range
	 */
	private static final int NATIVE_PRIO_RANGE = 19;

	/**
	 * load native thread library.
	 */
	static {
		try {
			File cwd = new File(".");
			Runtime.getRuntime().load(cwd.getCanonicalPath() + '/' + JAVA_LIB_NAME);
		} catch (Throwable t) {	// ignore any errors
		}
	}

	// constructors from Thread
	
	public NativeThread() {
	}

	public NativeThread(Runnable target) {
		super(target);
	}

	public NativeThread(ThreadGroup group, Runnable target) throws SecurityException {
		super(group, target);
	}

	public NativeThread(String name) {
		super(name);
	}

	public NativeThread(ThreadGroup group, String name) throws SecurityException {
		super(group, name);
	}

	public NativeThread(Runnable target, String name) {
		super(target, name);
	}

	public NativeThread(ThreadGroup group, Runnable target, String name) throws SecurityException {
		super(group, target, name);
	}

	public NativeThread(ThreadGroup group, Runnable target, String name, long stackSize) throws SecurityException {
		super(group, target, name, stackSize);
	}

	/**
	 * private native method to set raw native thread priority.
	 *
	 * @param prio priority. Must be in range from 0 to NATIVE_PRIO_RANGE.
	 *
	 * @return true if the call succeeded, false otherwise.
	 */
	private native boolean setRawNativePriority(int prio);

	/**
	 * set native priority.
	 * This method is an alternative to setPriority().
	 * Falls back on setPriority on failure.
	 *
	 * @param prio priority. Must be in range from MIN_PRIORITY to MAX_PRIORITY.
	 *
	 * @throws IllegalArgumentException if prio is not in the required range.
	 * @throws SecurityException if the current thread cannot modifiy this thread.
	 */
	public final void setNativePriority(int prio) throws IllegalArgumentException, SecurityException {
		try {
			int nPrio = NATIVE_PRIO_RANGE - (NATIVE_PRIO_RANGE * (prio - MIN_PRIORITY)) / PRIO_RANGE;
			if (setRawNativePriority(nPrio))
				return;
		} catch (LinkageError e) {
		}
		setPriority(prio);
	}

}
