package org.xwiki.xmlrpc;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Map objects are used to transport data using XML RPC in a convenient way. Instance fields of a
 * map object are converted to a map('field name' -> 'value') that is used to send the representation of
 * the object through XMLRPC. Since there could be some fields that should not appear in the
 * representation (e.g., fields used as a cache for synthesized values), only the fields annotated
 * with the Property annotation are included in the map conversion.
 * 
 * In order to be correctly transported, field types must be one of the following: String, Integer,
 * Boolean, Double, Date, Long (converted to Integer). All other types are implicitly converted to
 * String by using the toString() method. Arrays, Lists and Maps of the previous elements are also
 * supported (see {@link XmlRpcUtils#convertToXmlRpc(Object)}).
 * 
 * In order to create a map object you need to instantiate a class that derives from MapObject and
 * override the MapObject(Map<String, Object>) constructor, calling the super(Map<String, Object>)
 * constructor. You may need to perform some additional check in this constructor in order to
 * control that all the variables have consistent values. In fact, if a field name is not in the
 * provided map, its corresponding value will not be initialized (i.e., it wil default to null), and
 * this might not be what you wanted.
 * 
 * @author fmancinelli
 * 
 */
public abstract class MapObject
{
    private static final Log log = LogFactory.getLog(MapObject.class);

    public MapObject()
    {
    }

    /**
     * Initialize the map object's field using the data provided through a map. For each map key,
     * field with the key's name is assigned to the corresponding value, provided that the value is
     * compatible with the actual type of the field.
     * 
     * @param map The map containing the associations between field names and their value.
     */
    protected MapObject(Map<String, Object> map)
    {
        this();

        /**
         * Usually 'this' is a subclass of MapObject. We need to go up through the inheritance
         * hierarchy in order to find all the fields declared in all the super classes. This is
         * necessary because reflection mechanisms are not able to list the inherited fields.
         */
        Class currentClass = this.getClass();
        while (!currentClass.equals(MapObject.class)) {
            for (Field field : currentClass.getDeclaredFields()) {
                if (field.getAnnotation(Property.class) != null) {
                    Object value = map.get(field.getName());

                    try {
                        if (value != null) {
                            if (checkCompatibility(value, field.getType())) {
                                field.set(this, value);
                            } else {
                                throw new IllegalArgumentException(String
                                    .format(
                                        "Cannot initialize from map. Field '%s' has type '%s' but the value provided is of type '%s'",
                                        field.getName(), field.getType(), value.getClass()));
                            }
                        }
                    } catch (IllegalAccessException e) {
                        log
                            .warn(String
                                .format(
                                    "Cannot access field '%s' in %s. Please change its visibility level to protected.",
                                    field.getName(), this.getClass()));
                    }
                }
            }
            currentClass = currentClass.getSuperclass();
        }
    }

    /**
     * Build a representation of the map object. Only the fields annotated with the Property
     * annotation will be added to the representation.
     * 
     * @return A map containing the associations 'field name' -> 'value'.
     */
    public Map<String, Object> toMap()
    {
        Map<String, Object> result = new HashMap<String, Object>();

        Class currentClass = this.getClass();
        while (!currentClass.equals(MapObject.class)) {
            for (Field field : currentClass.getDeclaredFields()) {
                try {
                    if (field.getAnnotation(Property.class) != null) {
                        result.put(field.getName(), XmlRpcUtils.convertToXmlRpc(field.get(this)));
                    }
                } catch (Exception e) {
                    log
                        .warn(String
                            .format(
                                "Cannot access field '%s' in %s. Please change its visibility level to protected.",
                                field.getName(), this.getClass()));
                }
            }
            currentClass = currentClass.getSuperclass();
        }

        return result;
    }

    /**
     * @param value The value to check the compatibility for.
     * @param type The target type.
     * @return True if the value can be assigned to a variable of the given type. False otherwise.
     */
    protected boolean checkCompatibility(Object value, Class type)
    {
        if (value.getClass().isAssignableFrom(type)) {
            return true;
        }

        // isAssignableFrom doesn't look at interfaces... So if the target type is an interface then
        // we have to look at if the type of value implements that interface.
        if (type.isInterface()) {
            List<Class> interfaces = Arrays.asList(value.getClass().getInterfaces());
            if (interfaces.contains(type)) {
                return true;
            }
        }

        return false;
    }
}