package org.xwiki.xmlrpc;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public class XWikiObject extends MapObject
{
    @Property
    protected String pageId;

    @Property
    protected String className;
    
    @Property
    protected Integer id;

    @Property
    protected String prettyName;
    
    @Property
    protected Map<String, Object> fieldToTypeMap;

    @Property
    protected Map<String, Object> fieldToValueMap;

    public XWikiObject(String pageId, String className, Integer id, String prettyName, Map<String, Object> fieldToTypeMap,
        Map<String, Object> fieldToValueMap)
    {
        this.pageId = pageId;
        this.className = className;
        this.id = id;
        this.prettyName = prettyName;
        this.fieldToTypeMap = new HashMap<String, Object>(fieldToTypeMap);
        this.fieldToValueMap = new HashMap<String, Object>(fieldToValueMap);
    }

    public XWikiObject(Map<String, Object> map)
    {
        super(map);

        for (String fieldName : fieldToTypeMap.keySet()) {
            Object o = fieldToTypeMap.get(fieldName);
            if (o instanceof String) {
                String className = (String) o;
                try {
                    Class c = Class.forName(className);
                    fieldToTypeMap.put(fieldName, c);
                } catch (ClassNotFoundException e) {
                    throw new IllegalArgumentException(String.format(
                        "Cannot initialize XWikiObject. '%s' not found.", className));
                }                
            }                        
        }
    }

    public String getClassName()
    {
        return className;
    }

    public Set<String> getFields()
    {
        return new HashSet<String>(fieldToValueMap.keySet());
    }

    public Object get(String fieldName)
    {
        Object value = fieldToValueMap.get(fieldName);

        /* Convert arrays to lists for easier access */
        if (value.getClass().isArray()) {
            value = Arrays.asList((Object[]) value);
            fieldToValueMap.put(fieldName, value);
        }

        return value;
    }

    public void set(String fieldName, Object object)
    {
        Class type = (Class)fieldToTypeMap.get(fieldName);
        if(type != null) {
            if(!checkCompatibility(object, type)) {
                throw new IllegalArgumentException(String.format("Incompatible types. Trying to assign a '%s' to '%s'.", object.getClass(), type.getName()));
            }
        }
        
        fieldToValueMap.put(fieldName, object);
    }

    public String getPageId()
    {
        return pageId;
    }
    
    public Class getFieldType(String fieldName) {
        return (Class)fieldToTypeMap.get(fieldName);
    }

    public Integer getId()
    {
        return id;
    }

    public String getPrettyName()
    {
        return prettyName;
    }

}
