package org.xwiki.xmlrpc;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Utility methods for XMLRPC communication.
 * 
 * @author fmancinelli
 *
 */
public class XmlRpcUtils
{    
    /**
     * Convert a value to a suitable form for being transfered using XML RPC.
     * 
     * Strings, Integers, Booleans, Doubles, Dates are left untouched. Longs are converted to
     * Integers. Classes are converted to their fully qualified names (this is convenient for
     * transferring XWikiObject fields' type information). All other objects are implicitly converted
     * to String by using the toString() method. Arrays, Lists and Maps of the previous elements are
     * also supported and all their elements are recursively converted as well.
     * 
     * @param o The object to convert.
     * @return The converted object.
     */
    public static Object convertToXmlRpc(Object o)
    {
        if (o.getClass().isArray()) {
            Object[] objects = (Object[]) o;
            List result = new ArrayList();
            for (Object element : objects) {
                result.add(convertToXmlRpc(element));
            }

            return result;
        } else if (o instanceof List) {
            List list = (List) o;
            List result = new ArrayList();
            for (Object element : list) {
                result.add(convertToXmlRpc(element));
            }

            return result;
        } else if (o instanceof Map) {
            Map map = (Map) o;
            Map result = new HashMap();
            for (Object key : map.keySet()) {
                Object object = map.get(key);
                result.put(key, convertToXmlRpc(object));
            }

            return result;
        } else if (o instanceof Integer) {
            return o;
        } else if (o instanceof Boolean) {
            return o;
        } else if (o instanceof Date) {
            return o;
        } else if (o instanceof Double) {
            return o;
        } else if (o instanceof Long) {
            Long l = (Long) o;
            return new Integer(l.intValue());
        } else if (o instanceof Class) {
            /*
             * For classes we get their names and not their string representation. This is
             * convenient for transferring XWikiObjects' field types.
             */
            Class c = (Class) o;
            return c.getName();
        }

        return o.toString();
    }
}
