﻿// Written in the D programming language.

/**
Fortran-style arrays (alpha version)

Copyright: Copyright by authors 2011-.

License: $(WEB boost.org/LICENSE_1_0.txt, Boost License 1.0).

Authors: Denis Shelomovskij
 */
module rarray;

import core.exception;
import std.exception;
import std.array;
import std.traits;
import std.typetuple;
import std.conv;
import std.string;
import std.range;
import std.algorithm;

struct AllR { }

struct R
{
	size_t from, to;
	
	@disable void opCall(); //@@BUG@@ 6688 workaround @disable this();
	
	static AllR opSlice()
	{
		return AllR();
	}
	
	static R opSlice(size_t from, size_t to)
	{
		R r = void;
		r.from = from, r.to = to;
		return r;
	}
}

private template RCount(T...)
{
	static if(T.length)
		enum RCount = (is(Unqual!(T[0]) == R) || is(Unqual!(T[0]) == AllR)) + RCount!(T[1 .. $]);
	else
		enum RCount = 0;
}

private template isROrSize(T)
{
	enum isROrSize = RCount!T || isImplicitlyConvertible!(T, const(size_t));
}

struct RArray(T, size_t N) if(N >= 1)
{
	static if(N == 1)
		alias TypeTuple!(size_t) SizeTypes;
	else
		alias TypeTuple!(size_t, RArray!(T, N-1).SizeTypes) SizeTypes;

	//@@BUG@@ 6763 workaround (or it makes SizeTypes ref too)
	static if(N == 1)
		enum RefSizeTypes = "ref size_t";
	else
		enum RefSizeTypes = "ref size_t, " ~ RArray!(T, N-1).RefSizeTypes;
	mixin(`alias int delegate(`~RefSizeTypes~`, ref T) RefDelegate;`);
	
	immutable SizeTypes lengths;
	
	this(SizeTypes lengths)
	{
		SizeTypes strides;
		strides[$-1] = 1;
		foreach_reverse(i, n; lengths[1 .. $])
			strides[i] = n * strides[i+1];
		this(new T[strides[0] * lengths[0]], strides, lengths);
	}
	
	@property size_t empty() const
	{
		foreach(l; lengths[1..$])
			if(!l) return true;
		return false;
	}
	
	@property size_t length() const
	{
		size_t res = lengths[0];
		foreach(l; lengths[1..$])
			res *= l;
		return res;
	}
	
	int opApply(RefDelegate dg) //@@BUG@@ 6763 workaround int delegate(ref SizeTypes, ref T)
	{
		if(empty)
			return 0;
		SizeTypes indices = 0;
		indices[$-1] = -1;
		for(;;)
		{
			foreach_reverse(plane, index; indices) //@@@GUG@@@ ??? ref index -> no storage class for value index
				if(++indices[plane] < lengths[plane])
					break;
				else if(plane)
					indices[plane] = 0;
				else
					return 0;
			if(int res = dg(indices, data[offset(indices)]))
				return res;
		}
		assert(0);
	}
	
	int opApply(int delegate(ref T) dg)
	{
		//@@BUG@@ 6763 workaround
	//	return opApply(delegate(ref SizeTypes, ref T t) { return dg(t); });
		return opApply(mixin("delegate("~RefSizeTypes~", ref T t) { return dg(t); }"));
	}
	
	size_t getLength(size_t plane) const
	in { assert(plane < N); }
	body
	{
		foreach(i, length; lengths)
			if(i == plane)
				return length;
		assert(0);
	}
	
	alias size_t[N] size_t_N; //@@BUG@@ ####2 workaround
	
	ref opIndex()(in size_t_N indices...) //@@BUG@@ ####2 workaround size_t[N]
	in { foreach(plane, T; SizeTypes) enforceEx!RangeError(indices[plane] >= 0 && indices[plane] < lengths[plane]); }
	body {
		return data[offset(indices)];
	}
	
	T opIndexAssign(A...)(T value, A args) if(args.length == N && allSatisfy!(isROrSize, A) && !RCount!A)
	in { foreach(plane, index; args) enforceEx!RangeError(index >= 0 && index < lengths[plane]); }
	body {
		return data[offset(args)] = value;
	}
	
	auto opIndex(A...)(A args) if(args.length == N && allSatisfy!(isROrSize, A) && RCount!A)
	{
		alias RArray!(T, RCount!A) ResultType;
		
		SizeTypes firstIndices;
		ResultType.SizeTypes newLengths;
		static if(RCount!A == N)
			alias strides newStrides;
		else
			ResultType.SizeTypes newStrides;
		
		foreach(i, a; args)
		{
			alias Unqual!(A[i]) UnqualAi;
			
			static if(RCount!UnqualAi)
			{
				enum j = RCount!(A[0 .. i]);
				static if(RCount!A != N)
					newStrides[j] = strides[i];
			}
			
			static if(is(UnqualAi == R))
			{
				assert(a.from <= a.to);
				testBoundIndex(a.from, i);
				testBoundIndex(a.to, i);
				firstIndices[i] = a.from;
				newLengths[j] = a.to - a.from;
			}
			else static if(is(UnqualAi == AllR))
			{
				firstIndices[i] = 0;
				newLengths[j] = lengths[i];
			}
			else
			{
				testBoundIndex(a, i);
				firstIndices[i] = a;
			}
		}
		
		return ResultType(data[offset(firstIndices) .. $], newStrides, newLengths); //TODO $ -> actual bound
	}
	
	Range opSliceAssign(Range)(Range value) if(isForwardRange!Range)
	{
		Range t = value.save;
		foreach(ref el; this)
		{
			enforce(!value.empty, format("RArray opIndexAssign: value doesn't contain enough elements (< %s)", length));
			el = value.front;
			value.popFront();
		}
		enforce(value.empty, format("RArray opIndexAssign: value contains too many elements (> %s)", length));
		return t;
	}
	
	Range opIndexAssign(Range, A...)(Range value, A args) if(args.length == N && allSatisfy!(isROrSize, A) && RCount!A && isForwardRange!Range)
	{
		return this[args][] = value;
	}
	
	@property auto mapIndices(string pred)()// if (is(typeof(unaryFun!pred)))
	{
		static struct Result
		{
			RArray rarr;
			size_t a;
			size_t[N] indices;
			@property bool empty() const { return !rarr.goodGetOffset(indices); }
			@property auto save() { return Result(rarr, a, indices); }
			@property ref front() { return rarr[indices]; }
			void popFront() {
				++a;
				indices = mixin('['~pred~']'); //TODO slow unless @@@BUG@@@ ???? will be fixed
			}
		}
		auto res = Result(this, -1);
		res.popFront();
		return res;
	}
	
	//Dirty toString() functions just for debug purposes
	static if(N == 1)
		string toString(size_t elementLength = 0) {
			string res = "[";
			foreach(i, el; this)
				res ~= to!string(el).rightJustify(elementLength) ~ (i == lengths[0]-1 ? "" : ", ");
				res ~= ']';
			return res;
		}
	else static if(N == 2)
		string toString(size_t elementLength = 0) {
			foreach(el; this)
				elementLength = max(elementLength, to!string(el).length);
			
			string res;
			foreach(i; 0 .. lengths[0])
				res ~= this[i, R[]].toString(elementLength) ~ '\n';
			return res;
		}
	else static if(N == 3)
		string toString(size_t elementLength = 0) {
			foreach(el; this)
				elementLength = max(elementLength, to!string(el).length);
			
			string res = "[";
			foreach(i; 0 .. lengths[0])
				res ~= '\n' ~ this[i, R[], R[]].toString(elementLength);
			res ~= "]\n";
			return res;
		}
	
private:
	T[] data;
	immutable SizeTypes strides;
	
	private this(T[] data, TypeTuple!(SizeTypes, SizeTypes) strides_lengths)
	in
	{
		/*TODO do some check size_t minimalDataLength = 1;
		foreach(i, length; lengths) {
			assert(length <= strides[i]);
			minimalDataLength *= n;
		}
		assert(data.length >= minimalDataLength);*/
	}
	body
	{
		this.data = data;
		this.strides = strides_lengths[0 .. N];
		this.lengths = strides_lengths[N .. $];
	}
	
	void testBoundIndex(int index, int plane) const
	{
		debug enforceEx!RangeError(index >= 0 && index <= getLength(plane));
	}
	
	size_t offset(in size_t[N] indices...) const
	in { foreach(plane, index; indices) testBoundIndex(index, plane); }
	body {
		size_t res = 0;
		foreach(plane, T; SizeTypes)
			res += indices[plane] * strides[plane];	
		return res;
	}
	
	bool goodGetOffset(in size_t[N] indices...) const
	{
		foreach(plane, T; SizeTypes)
			if(indices[plane] < 0 || indices[plane] >= lengths[plane])
				return false;
		return true;
	}
}