// designs a complex tap fir filter akin to the hilbert transformer.
//
// The hilbert transformer is classified as a linear phase fir
// with allpass magnitude response and 90 degree phase response for
// positive frequencies and -90 degrees phase for negative frequencies.
// Or, if you prefer, normalized frequencies between .5 and 1 since
// negative frequencies don't really have much meaning outside the complex
// domain.
//
// Normally one would use the hilbert transformer in one leg of a complex
// processing block and a compensating delay in the other.
//
// This one differs in the following respects:
//  It is low pass with a cutoff of .078125
//  The filter is a lowpass kaiser windowed filter with parameter 3
//  The phase response is 45 degrees for positive frequencies and -45
//   for negative frequencies.
//  The coefficent set is used in one path and the same coefficients
//   are used time reversed in the other. This results in the net effect
//   of +/- 90 degrees as in the usual hilbert application.
//
// The coefficient set can be used in the gnuradio frequency translating
// fir filter for ssb demodulation.
//
// This isn't as computationally efficient as using the hilbert transformer
// and compensating delay but fascinating none the less.
//
// This program is for the scilab language a very powerful free math 
// package similar to Matlab with infinitely better price/performace.
//
// compute the prototype lowpass fir
// length is 255 (odd) for the same symmetry reasons as the hilbert transformer

h = wfir( 'lp', 255, [20.0/256 0], 'kr', [3 0] );

H = fft(h);

H(1:127)=H(1:127)*exp(%i*%pi/4);
H(128)=0+%i*0;
H(129:255)=H(129:255)*exp(-%i*%pi/4);

j=real(ifft(H));
k(1:255)=j(255:-1:1);

f = file('open','taps')
for i=(1:255)
  fprintf( f, '%f%+fj', j(i), k(i) )
end

file('close',f)

