/* -*- mode: objc -*-
 * FYStackTrace.m created by Wim Oudshoorn on Mon 11-Apr-2006
 */

/***************************************************************
 *
 * FYStackTrace
 *
 ***************************************************************/

#include "FYStackTrace.h"
#include <bfd.h>
#include <Foundation/NSDebug.h>

/*
 * Needed because somehow the bfd lib wants to call this function
 * and I can't find where it is defined.
 */
//int __mingw_fseeko64 (FILE *stream, long long offset, int whence)
//{
//    return fseek (stream, offset, whence);
//}



@interface FYBFDWrapper: NSObject
{
    bfd*    _abfd;
    asymbol **_symbols;
    long    _symbolCount;
    BOOL    _initialized;

    // Variables used by callback functions
    bfd_vma   _address;
    NSString *_fileName;
    NSString *_functionName;
    int       _lineNo;
}

- (void) readSymbols;
- (asymbol**) symbols;


// callback functions while traversing bfd
- (void) setFileName: (const char*) fileName
        functionName: (const char*) functionName
           andLineNo: (int) lineNo;

- (void) openBFDForCurrentProcess;
- (BOOL) foundAddress;
- (bfd_vma) address;

@end

static FYBFDWrapper *theWrapper = nil;

static void find_address (bfd *abfd, asection *section, FYBFDWrapper *wrapper)
{
    bfd_vma vma;
    unsigned size;
    const char *fileName;
    const char *functionName;
    int line = 0;
    BOOL found;
    
    bfd_vma address;
    
    if ([wrapper foundAddress])
        {
        return;
        }

    if (!(bfd_get_section_flags (abfd, section) & SEC_ALLOC))
        {
        return;
        }
    address = [wrapper address];
    vma = bfd_get_section_vma (abfd, section);
    size = bfd_get_section_size (section);
    
    if (address < vma || address >= vma + size)
        {
        return;
        }

    found = bfd_find_nearest_line (abfd, section, [wrapper symbols], address - vma,
                                   &fileName, &functionName, &line);

    if (found)
        {
        [wrapper setFileName: fileName functionName: functionName andLineNo: line];
        }
}

@implementation FYBFDWrapper

- init
{
    if (theWrapper)
        {
        [self dealloc];
        return [theWrapper retain];
        }
    [self openBFDForCurrentProcess];
    [self readSymbols];

    theWrapper = [self retain];
    
    return self;
}

+ bfdWrapper
{
    if (theWrapper)
        {
        return theWrapper;
        }
    else
        {
        return [[[self alloc] init] autorelease];
        }
}

- (void) freeSymbols
{
    if (_symbols)
        {
        free (_symbols);
        _symbols = NULL;
        }
    _symbolCount = 0;
}

- (asymbol**) symbols
{
    return _symbols;
}

- (void) readSymbols
{
    unsigned neededSpace;

    [self freeSymbols];
    
    if (!_abfd)
        {
        NSLog (@"PhS-FYST: No BFD set in read symbols");
        return;
        }
    
    if (!(bfd_get_file_flags (_abfd) & HAS_SYMS))
        {
        NSLog (@"PhS-FYST: BFD does not contain any symbols");
        return;
        }
    
    neededSpace = bfd_get_symtab_upper_bound (_abfd);

    if (neededSpace < 0)
        {
        NSLog (@"PhS-FYST: BFD error while deducing needed space");
        return;
        }
    
    if (neededSpace == 0)
        {
        NSLog (@"PhS-FYST: BFD no space for symbols needed");
        return;
        }
    _symbols = malloc (neededSpace);

    _symbolCount = bfd_canonicalize_symtab (_abfd, _symbols);

    if (_symbolCount < 0)
        {
        NSLog (@"PhS-FYST: BFD reading symbols");
        [self freeSymbols];
        return;
        }
}

- (void) freeBFD
{
    [self freeSymbols];
    if (_abfd)
        {
        bfd_close (_abfd);
        }
}

- (void) openBFDForCurrentProcess
{
    NSString *processName = [[[NSProcessInfo processInfo] arguments] objectAtIndex: 0];
    bfd_boolean matching = NO;
    
    if (!_initialized)
        {
        NSLog (@"PhS-FYST: BFD initialized, processName: %@", processName);
        bfd_init ();
        _initialized = YES;
        }

    _abfd = bfd_openr ([processName cString], NULL);

    if (bfd_check_format (_abfd, bfd_archive))
        {
        NSLog (@"PhS-FYST: BFD format error!");
        [self freeBFD];
        return;
        }
    if (!bfd_check_format_matches (_abfd, bfd_object, &matching))
        {
        NSLog (@"PhS-FYST: BFD format object error");
        [self freeBFD];
        return;
        }
}

- (BOOL) foundAddress
{
    return _lineNo >= 0;
}

- (bfd_vma) address
{
    return _address;
}

- (void) freeCallBackData
{
    _lineNo = -1;
    DESTROY (_fileName);
    DESTROY (_functionName);
}

- (void) setFileName: (const char*) fileName
        functionName: (const char*) functionName
           andLineNo: (int) lineNo
{
    [self freeCallBackData];
    
    _lineNo = lineNo;
    _fileName = [[NSString alloc] initWithCString: fileName];
    _functionName = [[NSString alloc] initWithCString: functionName];
}

    
- (NSDictionary*) informationForAddress: (void*) address
{
    [self freeCallBackData];
    
    _address = (bfd_vma) address;
    bfd_map_over_sections (_abfd, find_address, self);

    if ([self foundAddress])
        {
        return [NSDictionary dictionaryWithObjectsAndKeys:
                                 [NSValue valueWithPointer: address], @"Address",
                             _fileName, @"FileName",
                             _functionName, @"FunctionName",
                             [NSNumber numberWithInt: _lineNo], @"LineNumber",
                             nil, nil];
        }
    else
        {
        return [NSDictionary dictionaryWithObjectsAndKeys:
                                 [NSValue valueWithPointer: address], @"Address",
                             nil, nil];
        }
}

@end

@implementation FYStackTrace

+ stackTrace
{
    FYBFDWrapper *wrapper = [FYBFDWrapper bfdWrapper];
    NSMutableArray *stack = [NSMutableArray array];
    void *address;
    int  index = 0;
    FYStackTrace *result = [[self alloc] init];
    
    address = NSReturnAddress (index++);

    while (address && (index < 100))
        {
        [stack addObject: [wrapper informationForAddress: address]];

        address = NSReturnAddress (index++);
        }

    ASSIGN (result->_stack, stack);

    return [result autorelease];
}


- (void) dealloc
{
    DESTROY (_stack);

    [super dealloc];
}

- (NSString*) description
{
    NSEnumerator *num = [_stack objectEnumerator];
    NSMutableString *result = [NSMutableString string];
    int index = 0;
    NSDictionary *stackLine;
    
    while ((stackLine = [num nextObject]))
        {
        [result appendFormat: @"%3d: %p %@  %@:%@\n",
                ++index, [[stackLine objectForKey: @"Address"] pointerValue],
                [stackLine objectForKey: @"FunctionName"],
                [stackLine objectForKey: @"FileName"],
                [stackLine objectForKey: @"LineNumber"]];
        }
    
    return result;
}


@end
