/* -*- c-file-style: "java"; indent-tabs-mode: nil; fill-column: 78; -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2007 by Ihar Filipau <thephilips@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/* subst_rcmd.c -- Parse command line option "--subst-rcmd". */

#include <stdio.h>	/* sscanf() */
#include <sys/socket.h>	/* inet_aton() */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>	/* strlen() */
#include <stdlib.h>	/* malloc()/free() */

#include "trace.h"
#include "subst_rcmd.h"


/*
    Function to parse parameter of command like option "--subst-rcmd". 
    The parameter looks like "gcc@192.168.1.1=ppc-gcc".
 */
struct subst_rcommand *
dcc_subst_rcommand_parse(const char *cmd_line_param)
{
	char *remote_command=0, *client_ipaddr=0, *local_command=0;
	struct subst_rcommand *scc=0, *ret=0;
	struct in_addr ina;

	size_t clp_len = strlen( cmd_line_param );
	int i;

	do {
		remote_command = (char *)malloc( clp_len );
		client_ipaddr = (char *)malloc( clp_len );
		local_command = (char *)malloc( clp_len );
		scc = (struct subst_rcommand *)malloc( sizeof(*scc) );

		if (!remote_command || !client_ipaddr || !local_command || !scc) {
			rs_log_crit("malloc failed");
			break;
		}

		i = sscanf( cmd_line_param, "%[^@]@%[^=]=%s", remote_command, 
			client_ipaddr, local_command );

		if (i != 3) {
			rs_log_error("can't parse: \"%s\"", cmd_line_param);
			break;
		}

		i = inet_aton( client_ipaddr, &ina );
		if (i == 0) {
			rs_log_error("can't parse internet address \"%s\"", 
				client_ipaddr);
			break;
		}

		rs_trace("remote command subsitution: \"%s\" from client \"%s\" "
			"to local \"%s\"", remote_command,
			client_ipaddr, local_command);

		memset( scc, 0, sizeof(*scc) );
		scc->ipv4_addr = ina.s_addr;
		scc->remote_command = remote_command;
		scc->local_command = local_command;

		ret = scc;
	} while (0);

	if (ret == 0) {
		if (scc) free(scc);
		if (remote_command) free(remote_command);
		if (client_ipaddr) free(client_ipaddr);
		if (local_command) free(local_command);
	}

	return ret;
}

/*
    Utility function to add another command substition to global list.
 */
void
dcc_subst_rcommand_append(struct subst_rcommand **head, 
	struct subst_rcommand *newsc)
{
	struct subst_rcommand *x;	/* and x marks the spot */

	/* assert(head); */

	newsc->next = 0;
	x = *head;

	if (x == 0) {
		*head = newsc;
		return;
	} else {
		while (x->next) {
			x = x->next;
		}
		/* assert( x->next == 0 ); */
		x->next = newsc;
	}
}
