/** \file texmem.c
 * Implements all of the device-independent texture memory management.
 * 
 * Currently, only a simple LRU texture memory management policy is
 * implemented.  In the (hopefully very near) future, better policies will be
 * implemented.  The idea is that the DRI should be able to run in one of two
 * modes.  In the default mode the DRI will dynamically attempt to discover
 * the best texture management policy for the running application.  In the
 * other mode, the user (via some sort of as yet TBD mechanism) will select
 * a texture management policy that is known to work well with the
 * application.
 */

#include <stdio.h>
#include "texmem.h"
#include "simple_list.h"
#include "mem.h"

#include <assert.h>




/**
 * Calculate \f$\log_2\f$ of a value.  This is a particularly poor
 * implementation of this function.  However, since system performance is in
 * no way dependent on this function, the slowness of the implementation is
 * irrelevent.
 * 
 * \param n Value whose \f$\log_2\f$ is to be calculated
 */

static unsigned
driLog2( unsigned n )
{
    unsigned   log2;
    
    
    for ( log2 = 1 ; n > 1 ; log2++ )
    {
	n >>= 1;
    }

    return log2;
}




/**
 * Determine if a texture is resident in textureable memory.  Depending on
 * the driver, this may or may not be on-card memory.  It could be AGP memory
 * or anyother type of memory from which the hardware can directly read
 * texels.
 * 
 * This function is intended to be used as the \c IsTextureResident function
 * in the device's \c dd_function_table.
 * 
 * \param ctx GL context pointer (currently unused)
 * \param texObj Texture object to be tested
 */

GLboolean
driIsTextureResident( GLcontext * ctx, 
		      struct gl_texture_object * texObj )
{
    driTextureObject * t;


    t = (driTextureObject *) texObj->DriverData;
    return( (t != NULL) && (t->memBlock != NULL) );
}




/**
 * (Re)initialize the global circular LRU list.  The last element
 * in the array (\a heap->nrRegions) is the sentinal.  Keeping it
 * at the end of the array allows the other elements of the array
 * to be addressed rationally when looking up objects at a particular
 * location in texture memory.
 * 
 * \param heap Texture heap to be reset
 */

static void resetGlobalLRU( driTexHeap * heap )
{
   driTexRegion * list = heap->global_regions;
   unsigned       sz = 1U << heap->logGranularity;
   unsigned       i;

   heap->local_age = ++heap->global_age[0];

   for (i = 0 ; (i+1) * sz <= heap->size ; i++) {
      list[i].prev = i-1;
      list[i].next = i+1;
      list[i].age = 0;
   }

   i--;
   list[0].prev = heap->nrRegions;
   list[i].prev = i-1;
   list[i].next = heap->nrRegions;
   list[heap->nrRegions].prev = i;
   list[heap->nrRegions].next = 0;
   heap->global_age[0] = 0;
}




/**
 * Called by the client whenever it touches a local texture.
 * 
 * \param t Texture object that the client has accessed
 */

void driUpdateTextureLRU( driTextureObject * t )
{
    driTexHeap   * heap;
    driTexRegion * list;
    unsigned   shift;
    unsigned   start;
    unsigned   end;
    unsigned   i;

    
    heap = t->heap;
    if ( heap != NULL )
    {
	shift = heap->logGranularity;
	start = t->memBlock->ofs >> shift;
	end = (t->memBlock->ofs + t->memBlock->size - 1) >> shift;


	heap->local_age = ++heap->global_age[0];
	list = heap->global_regions;


	/* Update the context's local LRU 
	 */

	move_to_head( & heap->texture_objects, t );


	for (i = start ; i <= end ; i++) 
	{
	    list[i].in_use = 1;
	    list[i].age = heap->local_age;

	    /* remove_from_list(i)
	     */
	    list[(unsigned)list[i].next].prev = list[i].prev;
	    list[(unsigned)list[i].prev].next = list[i].next;

	    /* insert_at_head(list, i)
	     */
	    list[i].prev = heap->nrRegions;
	    list[i].next = list[heap->nrRegions].next;
	    list[(unsigned)list[heap->nrRegions].next].prev = i;
	    list[heap->nrRegions].next = i;
	}
    }
}




/**
 * Keep track of swapped out texture objects.
 * 
 * \param t Texture object to be "swapped" out of its texture heap
 */

void driSwapOutTextureObject( driTextureObject * t )
{
    if ( t->memBlock != NULL ) {
	assert( t->heap != NULL );
	mmFreeMem( t->memBlock );
	t->memBlock = NULL;

	move_to_tail( t->heap->swapped_objects, t );
	t->heap = NULL;
    }
    else
    {
	assert( t->heap == NULL );
    }


    t->dirty_images = ~0;
}




/**
 * Destroy hardware state associated with texture \a t.  Calls the
 * \a destroy_texture_object method associated with the heap from which
 * \a t was allocated.
 * 
 * \param t Texture object to be destroyed
 */

void driDestroyTextureObject( driTextureObject * t )
{
    driTexHeap * heap;


   if ( 0 ) {
       fprintf( stderr, "[%s:%d] freeing %p (tObj = %p, DriverData = %p)\n",
	        __FILE__, __LINE__,
	        t,
	        (t != NULL) ? t->tObj : NULL,
	        (t != NULL && t->tObj != NULL) ? t->tObj->DriverData : NULL );
   }

    if ( t != NULL )
    {
	if ( t->memBlock ) 
	{
	    heap = t->heap;
	    assert( heap != NULL );

	    mmFreeMem( t->memBlock );
	    t->memBlock = NULL;

	    heap->destroy_texture_object( heap->driverContext, t );
	    t->heap = NULL;
	}

	if ( t->tObj != NULL )
	{
	    assert( t->tObj->DriverData == t );
	    t->tObj->DriverData = NULL;
	}

	remove_from_list( t );
	FREE( t );
    }

   if ( 0 ) {
       fprintf( stderr, "[%s:%d] done freeing %p\n",
	        __FILE__, __LINE__,
	        t );
   }
}




/**
 * Update the local heap's representation of texture memory based on
 * data in the SAREA.  This is done each time it is detected that some other
 * direct rendering client has held the lock.  This pertains to both our local
 * textures and the textures belonging to other clients.  Keep track of other
 * client's textures by pushing a placeholder texture onto the LRU list --
 * these are denoted by \a tObj being \a NULL.
 * 
 * \param heap Heap whose state is to be updated
 * \param offset Byte offset in the heap that has been stolen
 * \param size Size, in bytes, of the stolen block
 * \param in_use Non-zero if the block is in-use by another context
 */

static void driTexturesGone( driTexHeap * heap, int offset, int size, 
			     int in_use )
{
    driTextureObject * t;
    driTextureObject * tmp;


    foreach_s ( t, tmp, & heap->texture_objects ) {
	if ( (t->memBlock->ofs < (offset + size))
	     && ((t->memBlock->ofs + t->memBlock->size) > offset) )
	{
	    /* It overlaps - kick it out.  If the texture object is just a
	     * place holder, then destroy it all together.  Otherwise, mark
	     * it as being swapped out.
	     */

	    if ( t->tObj != NULL )
	    {
		driSwapOutTextureObject( t );
	    }
	    else
	    {
		driDestroyTextureObject( t );
	    }
	}
    }


    if ( in_use ) {
	t = (driTextureObject *) CALLOC( heap->texture_object_size );
	if ( t == NULL ) return;

	t->memBlock = mmAllocMem( heap->memory_heap, size, 0, offset );
	if ( t->memBlock == NULL ) {
	    fprintf( stderr, "Couldn't alloc placeholder sz %x ofs %x\n",
		    (int)size, (int)offset );
	    mmDumpMemInfo( heap->memory_heap );
	    return;
	}
	insert_at_head( & heap->texture_objects, t );
    }
}




/**
 * Called by the client on lock contention to determine whether textures have
 * been stolen.  If another client has modified a region in which we have
 * textures, then we need to figure out which of our textures have been
 * removed and update our global LRU.
 * 
 * \param heap Texture heap to be updated
 */

void driAgeTextures( driTexHeap * heap )
{
    driTexRegion * list = heap->global_regions;
    unsigned       sz = 1U << (heap->logGranularity);
    unsigned       i, nr = 0;


    /* Have to go right round from the back to ensure stuff ends up
     * LRU in the local list...  Fix with a cursor pointer.
     */

    for (i = list[heap->nrRegions].prev ; 
	 i != heap->nrRegions && nr < heap->nrRegions ; 
	 i = list[i].prev, nr++)
    {
	/* If switching texturing schemes, then the SAREA might not have been
	 * properly cleared, so we need to reset the global texture LRU.
	 */

	if ( (i * sz) > heap->size ) {
	    nr = heap->nrRegions;
	    break;
	}

	if (list[i].age > heap->local_age) 
	    driTexturesGone( heap, i * sz, sz, 1); 
    }


    /* Loop or uninitialized heap detected.  Reset.
     */

    if (nr == heap->nrRegions) {
	driTexturesGone( heap, 0, heap->size, 0);
	resetGlobalLRU( heap );
    }
	
    heap->local_age = heap->global_age[0];
}




/**
 * Allocate memory from a texture heap to hold a texture object.  This
 * routine will attempt to allocate memory for the texture from the heaps
 * specified by \c heap_array in order.  That is, first it will try to
 * allocate from \c heap_array[0], then \c heap_array[1], and so on.
 *
 * \param heap_array Array of pointers to texture heaps to use
 * \param nr_heaps Number of heap pointer in \a heap_array
 * \param t Texture object for which space is needed
 * \return The ID of the heap from which memory was allocated, or -1 if
 *         memory could not be allocated.
 *
 * \bug The replacement policy implemented by this function is horrible.
 */

int
driAllocateTexture( driTexHeap * const * heap_array, unsigned nr_heaps,
		    driTextureObject * t )
{
    driTexHeap       * heap;
    driTextureObject * temp;
    unsigned           heap_id;


    /* In case t already has texture space, initialize heap.  This also
     * prevents GCC from issuing a warning that heap might be used
     * uninitialized.
     */

    heap = t->heap;


    /* Run through each of the existing heaps and try to allocate a buffer
     * to hold the texture.
     */

    for ( heap_id = 0 
	  ; (t->memBlock == NULL) && (heap_id < nr_heaps) 
	  ; heap_id++ )
    {
	heap = heap_array[ heap_id ];
	t->memBlock = mmAllocMem( heap->memory_heap, t->totalSize, 
				  heap->alignmentShift, 0 );
    }


    /* Kick textures out until the requested texture fits.
     */

    if ( t->memBlock == NULL )
    {
	for ( heap_id = 0 
	     ; (t->memBlock == NULL) && (heap_id < nr_heaps) 
	     ; heap_id++ )
	{
	    heap = heap_array[ heap_id ];
	    if ( t->totalSize <= heap->size ) 
	    { 
		while ( (t->memBlock == NULL)
		        && ! is_empty_list( & heap->texture_objects ) )
		{
		    /* The the LRU element.  If the texture is bound to one of
		     * the texture units, then we cannot kick it out.
		     */

		    temp = last_elem( & heap->texture_objects );
		    if ( temp->bound != 0 )
		    {
			break;
		    }

		    driSwapOutTextureObject( temp );
		    t->memBlock = mmAllocMem( heap->memory_heap, t->totalSize, 
					      heap->alignmentShift, 0 );
		}
	    }     /* if ( t->totalSize <= heap->size ) ... */
	}
    }


    if ( t->memBlock != NULL )
    {
	/* heap_id and heap->heapId may or may not be the same value here.
	 */

	assert( heap != NULL );
	assert( (t->heap == NULL) || (t->heap == heap) );

	t->heap = heap;
	return heap->heapId;
    }
    else
    {
	assert( t->heap == NULL );

	fprintf( stderr, "[%s:%d] unable to allocate texture\n",
		 __FUNCTION__, __LINE__ );
	return -1;
    }
}




/**
 * Create a new heap for texture data.
 * 
 * \param heap_id             Device-dependent heap identifier.  This value
 *                            will returned by driAllocateTexture when memory
 *                            is allocated from this heap.
 * \param context             Device-dependent driver context.  This is
 *                            supplied as the first parameter to the
 *                            \c destroy_tex_obj function.
 * \param size                Size, in bytes, of the texture region
 * \param alignmentShift      Alignment requirement for textures.  If textures 
 *                            must be allocated on a 4096 byte boundry, this
 *                            would be 12.
 * \param nr_regions          Number of regions into which this texture space
 *                            should be partitioned
 * \param global_regions      Array of \c driTexRegion structures in the SAREA
 * \param global_age          Pointer to the global texture age in the SAREA
 * \param swapped_objects     Pointer to the list of texture objects that are
 *                            not in texture memory (i.e., have been swapped
 *                            out).
 * \param texture_object_size Size, in bytes, of a device-dependent texture
 *                            object
 * \param destroy_tex_obj     Function used to destroy a device-dependent
 *                            texture object
 *
 * \sa driDestroyTextureHeap
 */

driTexHeap *
driCreateTextureHeap( unsigned heap_id, void * context, unsigned size,
		      unsigned alignmentShift, unsigned nr_regions,
		      driTexRegion * global_regions, unsigned * global_age,
		      driTextureObject * swapped_objects, 
		      unsigned texture_object_size,
		      destroy_texture_object_t * destroy_tex_obj
		    )
{
    driTexHeap * heap;
    unsigned     l;
    
    
    if ( 0 )
	fprintf( stderr, "%s( %u, %p, %u, %u, %u )\n",
		 __FUNCTION__,
		 heap_id, context, size, alignmentShift, nr_regions );

    heap = (driTexHeap *) CALLOC( sizeof( driTexHeap ) );
    if ( heap != NULL )
    {
	l = driLog2( (size - 1) / nr_regions );
	if ( l < alignmentShift )
	{
	    l = alignmentShift;
	}

	heap->logGranularity = l;
	heap->size = size & ~((1L << l) - 1);

	heap->memory_heap = mmInit( 0, heap->size );
	if ( heap->memory_heap != NULL )
	{
	    heap->heapId = heap_id;
	    heap->driverContext = context;

	    heap->alignmentShift = alignmentShift;
	    heap->nrRegions = nr_regions;
	    heap->global_regions = global_regions;
	    heap->global_age = global_age;
	    heap->swapped_objects = swapped_objects;
	    heap->texture_object_size = texture_object_size;
	    heap->destroy_texture_object = destroy_tex_obj;

	    heap->local_age = 0;
	    make_empty_list( & heap->texture_objects );
	}
	else
	{
	    FREE( heap );
	    heap = NULL;
	}
    }


    if ( 0 )
	fprintf( stderr, "%s returning %p\n", __FUNCTION__, heap );

    return heap;
}




/** Destroys a texture heap
 * 
 * \param heap Texture heap to be destroyed
 */

void
driDestroyTextureHeap( driTexHeap * heap )
{
    driTextureObject * t;
    driTextureObject * temp;


    if ( heap != NULL )
    {
	foreach_s( t, temp, & heap->texture_objects )
	{
	    driDestroyTextureObject( t );
	}

	mmDestroy( heap->memory_heap );
	FREE( heap );
    }
}
