/** \file texmem.h
 * Public interface to the DRI texture memory management routines.
 * 
 * \sa texmem.c
 */

#ifndef DRI_TEXMEM_H
#define DRI_TEXMEM_H

#include "mtypes.h"
#include "mm.h"

struct dri_tex_heap;
typedef struct dri_tex_heap driTexHeap;

struct dri_texture_object;
typedef struct dri_texture_object driTextureObject;

struct dri_texture_region;
typedef struct dri_texture_region driTexRegion;

/**
 * Base texture object type.  Each driver will extend this type with its own
 * private data members.
 */

struct dri_texture_object {
	struct dri_texture_object * next;
	struct dri_texture_object * prev;

	driTexHeap * heap;		/**< Texture heap currently stored in */
	struct gl_texture_object * tObj;
	PMemBlock   memBlock;		/**< Memory block containing texture */
	unsigned    bound;		/**< Texture unit currently bound to */

	unsigned    totalSize;		/**< Total size of the texture,
					 * including all mipmap levels 
					 */

	unsigned    dirty_images;	/**< Flags for whether or not images
					 * need to be uploaded to local or
					 * AGP texture space 
					 */
};


/**
 * Inter-context texture memory management structure.
 * 
 * Each driver will have an array of driTexRegion structures in its SAREA.
 * Each element in the array corresponds to a specific block of texture
 * memory.
 */
struct dri_texture_region {
	unsigned char next;	/**< indices to form a circular LRU  */
	unsigned char prev;	/**< indices to form a circular LRU  */
	unsigned char in_use;	/**< owned by a client, or free? */
	unsigned      age;	/**< tracked by clients to update local LRU's */
};




typedef void (destroy_texture_object_t)( void * driverContext,
				        driTextureObject * t );

/**
 * Client-private representation of texture memory state.
 *
 * Clients will place one or more of these structs in their driver
 * context struct to manage one or more global texture heaps.
 */

struct dri_tex_heap {

	/** Client-supplied heap identifier 
	 */
	unsigned heapId;	

	/** Pointer to the client's private context 
	 */
	void *driverContext;

	/** Total size of the heap, in bytes
	 */
	unsigned size;

	/** \brief \f$log_2\f$ of size of single heap region
	 *
	 * Each context takes memory from the global texture heap in
	 * \f$2^{logGranularity}\f$ byte blocks.  The value of
	 * \a logGranularity is based on the amount of memory represented
	 * by the heap and the maximum number of regions in the SAREA.  Given
	 * \a b bytes of texture memory an \a n regions in the SAREA,
	 * \a logGranularity will be \f$\lfloor\log_2( b / n )\rfloor\f$.
	 */
	unsigned logGranularity;

	/** \brief Required alignment of allocations in this heap
	 * 
	 * The alignment shift is supplied to \a mmAllocMem when memory is
	 * allocated from this heap.  The value of \a alignmentShift will
	 * typically reflect some require of the hardware.  This value has
	 * \b no \b relation to \a logGranularity.  \a alignmentShift is a
	 * per-context value.
	 *
	 * \sa mmAllocMem
	 */
	unsigned alignmentShift;

	/** Number of elements in global list (the SAREA).
	 */
	unsigned nrRegions;	 

	/** Pointer to SAREA \a driTexRegion array
	 */
	driTexRegion * global_regions;
    
	/** Pointer to the texture state age (generation number) in the SAREA
	 */
	unsigned     * global_age;

	/** Local age (generation number) of texture state
	 */
	unsigned local_age;

	/** Memory heap used to manage texture memory represented by
	 * this texture heap.
	 */
	memHeap_t    * memory_heap;

	/** List of objects that we currently believe to be in texture
	 * memory.
	 */
	driTextureObject     texture_objects;
    
	/** Pointer to the list of texture objects that are not in
	 * texture memory.
	 */
	driTextureObject   * swapped_objects;

	/** Size of the driver-speicific texture object.
	 */
	unsigned       texture_object_size;


	/**
	 * \brief Function to destroy driver-specific texture object data.
	 * 
	 * This function is supplied by the driver so that the texture manager
	 * can release all resources associated with a texture object.  This
	 * function should only release driver-specific data.  That is,
	 * \a driDestroyTextureObject will release the texture memory
	 * associated with the texture object, it will release the memory
	 * for the texture object itself, and it will unlink the texture
	 * object from the texture object lists.
	 *
	 * \param driverContext Pointer to the driver supplied context
	 * \param t Texture object that is to be destroyed
	 * \sa driDestroyTextureObject
	 */

	destroy_texture_object_t * destroy_texture_object;
};




/**
 * Called by the client on lock contention to determine whether textures have
 * been stolen.  If another client has modified a region in which we have
 * textures, then we need to figure out which of our textures have been
 * removed and update our global LRU.
 * 
 * \param heap Texture heap to be updated
 * \hideinitializer
 */

#define DRI_AGE_TEXTURES( heap )			\
   do {							\
       if ( (heap)->local_age > (heap)->global_age[0] )	\
	   driAgeTextures( heap );			\
   } while( 0 )




/* This should be called whenever there has been contention on the hardware
 * lock.  driAgeTextures should not be called directly.  Instead, clients
 * should use DRI_AGE_TEXTURES, above.
 */

void driAgeTextures( driTexHeap * heap );

void driUpdateTextureLRU( driTextureObject * t );
void driSwapOutTextureObject( driTextureObject * t );
void driDestroyTextureObject( driTextureObject * t );
int driAllocateTexture( driTexHeap * const * heap_array, unsigned nr_heaps,
    driTextureObject * t );

GLboolean driIsTextureResident( GLcontext * ctx, 
    struct gl_texture_object * texObj );

driTexHeap * driCreateTextureHeap( unsigned heap_id, void * context,
    unsigned size, unsigned alignmentShift, unsigned nr_regions,
    driTexRegion * global_regions, unsigned * global_age,
    driTextureObject * swapped_objects, unsigned texture_object_size,
    destroy_texture_object_t * destroy_tex_obj );
void driDestroyTextureHeap( driTexHeap * heap );

#endif /* DRI_TEXMEM_H */
