/*
 * Gradient demo (using 2D texture)
 * Brian Paul
 * 21 April 2004
 */


#define GL_GLEXT_PROTOTYPES
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GL/glut.h>

#define PI 3.14159

#define LINEAR 0
#define RADIAL 1
static int GradientMode = LINEAR;

#define CLAMP 0
#define REPEAT 1
static int WrapMode = CLAMP;

static GLint WinWidth = 500, WinHeight = 500;
static GLfloat Xpos, Ypos;

static GLuint LinearTex = 1, RadialTex = 2;


static void
PrintString(const char *s)
{
   while (*s) {
      glutBitmapCharacter(GLUT_BITMAP_8_BY_13, (int) *s);
      s++;
   }
}

static void
Draw(void)
{
   int i;

   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   {
      GLfloat sPlane[4];
      GLfloat tPlane[4];
      sPlane[0] = 2.0 / WinWidth;
      sPlane[1] = 0;
      sPlane[2] = 0;
      sPlane[3] = 0;

      tPlane[0] = 0;
      tPlane[1] = 2.0 / WinHeight;
      tPlane[2] = 0;
      tPlane[3] = 0;

      glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
      glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
      glTexGenfv(GL_S, GL_EYE_PLANE, sPlane);
      glTexGenfv(GL_T, GL_EYE_PLANE, tPlane);
      glEnable(GL_TEXTURE_GEN_S);
      glEnable(GL_TEXTURE_GEN_T);
   }

   if (GradientMode == LINEAR)
      glBindTexture(GL_TEXTURE_2D, LinearTex);
   else
      glBindTexture(GL_TEXTURE_2D, RadialTex);

   if (WrapMode == CLAMP) {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
   }
   else {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   }


   glEnable(GL_TEXTURE_2D);

   glPushMatrix();
   glTranslatef(Xpos, Ypos, 0);

   /* draw star */
   glBegin(GL_POLYGON);
   glVertex2f(0, 0);
   for (i = 0; i <= 10; i++) {
      float radius = 80.0 + 120.0 * (i & 1);
      float angle = i / 10.0 * PI * 2.0;
      float x = radius * sin(angle);
      float y = -radius * cos(angle);
      glVertex2f(x, y);
   }
   glEnd();

   glPopMatrix();

   glDisable(GL_TEXTURE_2D);

   {
      static const char *gradients[] = { "Linear", "Radial" }; 
      static const char *wraps[] = { "Clamp", "Repeat" }; 
      char s[1000];
      sprintf(s, "Gradient Mode [g]: %s", gradients[GradientMode]);
      glWindowPos2iARB(10, 10);
      PrintString(s);

      sprintf(s, "Texture Wrap Mode [w]: %s", wraps[WrapMode]);
      glWindowPos2iARB(10, 30);
      PrintString(s);
   }

   glutSwapBuffers();
}


static void
Reshape(int width, int height)
{
   WinWidth = width;
   WinHeight = height;
   glViewport(0, 0, width, height);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glOrtho(0, width, 0, height, -1, 1);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}


static void
Key(unsigned char key, int x, int y)
{
   (void) x;
   (void) y;
   switch (key) {
      case 'g':
         GradientMode = (GradientMode + 1) % 2;
         break;
      case 'w':
         WrapMode = (WrapMode + 1) % 2;
         break;
      case 27:
         exit(0);
         break;
   }
   glutPostRedisplay();
}


static GLboolean moving = GL_FALSE;
static int startX, startY;
static GLfloat startXpos, startYpos;


static void
Motion(int x, int y)
{
   if (moving) {
      float dx = x - startX;
      float dy = (WinHeight - y) - startY;
      Xpos = startXpos + dx;
      Ypos = startYpos + dy;
      glutPostRedisplay();
   }

}


static void
Mouse(int button, int state, int x, int y)
{
   if (button == GLUT_LEFT_BUTTON && state == GLUT_DOWN) {
      moving = GL_TRUE;
      startX = x;
      startY = WinHeight - y;
      startXpos = Xpos;
      startYpos = Ypos;
   }
   if (button == GLUT_LEFT_BUTTON && state == GLUT_UP) {
      moving = GL_FALSE;
   }
   glutPostRedisplay();
}



static void
Init(void)
{
   GLfloat gradient[256][256][3];
   GLint i, j;

   /* linear gradient texture */
   for (i = 0; i < 256; i++) {
      for (j = 0; j < 256; j++) {
         gradient[i][j][0] = j / 255.0;
         gradient[i][j][1] = i / 255.0;
         gradient[i][j][2] = 0.0;
      }
   }
   glBindTexture(GL_TEXTURE_2D, LinearTex);
   glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 256, 256, 0, GL_RGB, GL_FLOAT, gradient);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

   /* radial gradient texture */
   for (i = 0; i < 256; i++) {
      for (j = 0; j < 256; j++) {
         float dx = i - 128;
         float dy = j - 128;
         float r = sqrt(dx * dx + dy * dy) / sqrt(128*128 + 128*128);
         r *= 1.0;
         if (r > 1.0)
            r = 1.0;
         gradient[i][j][0] = 1.0 - r;
         gradient[i][j][1] = r;
         gradient[i][j][2] = 0.0;
      }
   }
   glBindTexture(GL_TEXTURE_2D, RadialTex);
   glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 256, 256, 0, GL_RGB, GL_FLOAT, gradient);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

   Xpos = 0.25 * WinWidth;
   Ypos = 0.25 * WinHeight;
}


int
main(int argc, char *argv[])
{
   glutInit(&argc, argv);
   glutInitWindowPosition(0, 0);
   glutInitWindowSize(WinWidth, WinHeight);
   glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
   glutCreateWindow(argv[0]);
   glutReshapeFunc(Reshape);
   glutKeyboardFunc(Key);
   glutMouseFunc(Mouse);
   glutMotionFunc(Motion);
   glutDisplayFunc(Draw);
   Init();
   glutMainLoop();
   return 0;
}
