#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#include "ffp.h"


// The different parts of the program is stored in strings
// which can be updated independently. When one string is
// updated, the whole program string must be reassembled.

// initialization holds the declaration part of the vertex program
static char * initialization;

// modelviewtransform holds the modelview transformation
// Since the modelview instructions are the same for all states in
// our implementation, the modelviewTransform() function have been
// dropped, and it is kept only as a string literal.
static char * modelviewtransform =
"\n# -------- Modelview calculations --------- \n"
"# Transform the vertex to eye coordinates \n"
"DP4 eyePosition.x, mvm[0], iPos;\n"
"DP4 eyePosition.y, mvm[1], iPos;\n"
"DP4 eyePosition.z, mvm[2], iPos;\n"
"DP4 eyePosition.w, mvm[3], iPos;\n"
"# --------- /Modelview calculations --------\n";

// normaltransform holds the normal transformation
static char * normaltransform;

// normalizationstr holds the normalization calculation
// The string has the added subfix str due to name clashes
static char * normalizationstr;

// rescalingstr holds the rescaling calculation
// The string has the added subfix str due to name clashes
static char * rescalingstr;

// projectiontransform holds the projection transformation
// Since the projection instructions are the same for all states
// in our implementation, the projectionTransform() function have
// been dropped, and it is kept only as a string literal.
static char * projectiontransform =
"\n# ---------- Projection calculations ----------\n"
"# Transform the vertex to clip-coordinates\n"
"DP4 oPos.x, proj[0], eyePosition;\n"
"DP4 oPos.y, proj[1], eyePosition;\n"
"DP4 oPos.z, proj[2], eyePosition;\n"
"DP4 oPos.w, proj[3], eyePosition;\n"
"# ---------- /Projection Calculations ---------\n";

// lighting holds the lighting calculations.
static char * lighting;

// initialized is used to keep track of which of the above strings
// has been initialized, to keep memory allocation problems to
// a minimum.
static GLint initialized = 0; // bitmask where each bit indicates
                                            // whether a part is initialized,
                                  // that is whether the string has
                     // been malloced (malloc()).
                                            // 1 is initialization
                                            // 2 is normal transform
                                            // 4 is normalization
                                            // 8 is rescaling
                                            // 16 is lighting

// maxlights is used to hold the maximum number of lights
// supported by the OpenGL implementation. The number is used to
// loop through the lights that can be enabled by the application.
static GLint maxlights = 0;


/*
Utility method to add a string to another. The first
string needs to be allocated through malloc, calloc or realloc before it is
sent to this function.
The second string can be a format string with it's parameters following
immediately. When the function has completed, a string                       which holds both
the content of dst and the content of src and eventual parameters of src,
is returned.
*/

#if 0
static char *
addString(char *dst, const char *src, ...)
{
   // sets up the necessary variables
   int newsize;
   char * newstr = "\0";
   char * newstr2 = "\0";
   va_list args;
   
   //printf("called\n");
   
   va_start(args, src);
   // enters this if clause if src is a format strings with
   // parameters
   if (sizeof(args)) {
      // The following call will lead to a printout to the standard output
      // stream. It is not necessarily trivial to avoid such a printout and
      // so we have kept it as it can be helpful for debugging purposes.
      newsize = vprintf(src, args) + 1;
      va_end(args);
      newstr2 = (char*)malloc(newsize);
      va_start(args, src);
      vsprintf(newstr2, src, args);
      va_end(args);
   }
   if (*newstr2 != '\0')
      newsize = 10000;//strlen(str1) + strlen(newstr2) + 1;
   else
      newsize = strlen(dst)+strlen(newstr2)+1; //strlen(str1) + strlen(str2) + 1;
   newstr = (char*)malloc(newsize);
   strcpy(newstr, /*str1*/dst);

   if (newstr2) {
      strcat(newstr, newstr2);
      free(newstr2);
   }
   else
      strcat(newstr, src);

   free(dst);
   //printf("called end\n");
   return newstr;
}
#endif
char *addString(char *dst, const char *src, ...)
{
	va_list ap;

	dst=realloc(dst, strlen(dst)+strlen(src)+10);
	va_start(ap, src);
	vsprintf(dst+strlen(dst), src, ap);
	va_end(ap);
	
	return dst;
}
		
/*
Utility function that cleans up the resources that has been allocated by
the vertex program system.
*/
void
cleanResources()
{
   // frees the memory allocated to the strings
   free(initialization);
   free(normaltransform);
   free(normalizationstr);
   free(rescalingstr);
   free(lighting);
} // cleanResources()

/* Constructs the string calculates the VP_e vector and normalizes it. */
static void computeVertexEyeVector()
{
	const char * vevector =
	"\n# Compute the vector pointing from the vertex position to the eye position\n"
	"SUB temp1, eyePositionConst, eyePositionNH;\n"
	"# Compute eye vector \n"
	"# d = |eyePositionConst - eyePositionNH| \n"
	"DP3 temp1.w, temp1, temp1; # temp1.w = d\n"
	"RSQ reflEyeLightVector.w, temp1.w; # reflEyeLightVector.w = 1/d \n"
	"MUL sphereEyeVector, temp1, reflEyeLightVector.w; # Normalized direction (VP_e unit vector) \n"
	"#DST reflEyeLightVector, temp1.w, reflEyeLightVector.w; #reflEyeLightVector = (1, d, d, 1/d) \n"
	
			
	;
	lighting = addString(lighting, vevector);
     
	//lighting = addString(lighting, "MOV oColorPrimFront, sphereEyeVector;\n");
} // computeVertexEyeVector()


/* Constructs the string that calculates the distance attenuation
   for the lightsource */
static void computeAttenuation(GLint lightindex)
{
	/*
	lighting = addString(lighting, "MOV temp1, {0.0, 0.3, 0.0, 0.0};\n");
	lighting = addString(lighting, "MOV distVector, {0.0, 0.0, 0.3, 0.0};\n");
	lighting = addString(lighting, "MOV reflEyeLightVector, {0.0, 1.0, 0.0, 0.0};\n");
	
	return ;
	*/
   const char * attcomp =
      "\n# Compute light direction and distance vector \n"
      "# d = |lightposition - eyeposition| \n"
      "DP3 temp1.w, temp1, temp1; # temp1.w = d\n"
      "RSQ distVector.w, temp1.w; # distVector.w = 1/d \n"
      "MUL reflEyeLightVector, temp1, distVector.w; # Normalized direction (VP_pli unit vector) \n"
      "DST distVector, temp1.w, distVector.w; # distVector = (1, d, d, 1/d) \n";

   lighting = addString(lighting, "\n# Compute the vector pointing from the vertex position to the light position\n");
   lighting = addString(lighting, "SUB temp1, lightPos%i, eyePositionNH; # temp1 = lightposition - eyeposition (VP_pli) \n", lightindex);
   lighting = addString(lighting, attcomp);
   lighting = addString(lighting, "\n# Compute Distance attenuation\n");
   lighting = addString(lighting, "DP3 distVector.w, distVector, lightAttCoeff%i; \n", lightindex);
   lighting = addString(lighting, "RCP distVector.w, distVector.w; \n");
} // computeAttenuation()

/*
Constructs the string that calculatest the spotlight component
for the lightsource
 */
static void computeSpotlight(GLint lightindex)
{
	GLfloat angle[1];
	glGetLightfv(GL_LIGHT0+lightindex, GL_SPOT_CUTOFF, angle);

	if (angle[0] != 180.0) {
		const char * spot =
		"LIT temp1, temp1; # temp1.z = 0.0, if P_pliV * S_dli - cos(c_rli) < 0 \n"
		"MUL distVector.w, distVector.w, temp1.z; # Light attenuation and spotlight attenaution is now multiplied together \n";
		lighting = addString(lighting, "\n# Compute spotlight cone attenuation\n");
		lighting = addString(lighting, "DP3 temp1.y, reflEyeLightVector, -lightSpotDir%i; \n", lightindex);
		//lighting = addString(lighting, "ADD temp1.x, temp1.y, -lightSpotCos%i.w; \n", lightindex);
		lighting = addString(lighting, "SUB temp1.x, temp1.y, lightSpotCos%i.w; \n", lightindex);
		lighting = addString(lighting, "MOV temp1.w, lightAttCoeff%i.w; # now: temp1 = ( P_pliV * S_dli - cos(c_rli) , P_pliV * s_dli , - , s_rli) \n", lightindex);
		//lighting = addString(lighting, "MOV temp1.z, lightAttCoeff%i.w; # now: temp1 = ( P_pliV * S_dli - cos(c_rli) , P_pliV * s_dli , - , s_rli) \n", lightindex);
		lighting = addString(lighting, spot);
////////		lighting = addString(lighting, "MOV oColorPrimFront, temp1; \n");
	}
} // computeSpotlight()
/*
      //spot cone computation
      float3 L2 = mul((float3x3)matViewIT, -normalize(lights[i].vDir));
      float rho = dot(L, L2);
      fAttenSpot *= pow(saturate((rho - lights[i].vSpot.y)/(lights[i].vSpot.x - lights[i].vSpot.y)), lights[i].vSpot.z);
*/
		
/*
Constructs the string that calculates the diffuse and specular
components of the lightsource.
*/
static void computeDiffuseAndSpecularComponents(GLboolean locallight,
                                          GLboolean front,
                                          GLboolean localviewer,
                                          GLint lightindex)
{
  if (!localviewer) { // Infinite viewer
    if (front) { // Frontside
       lighting = addString(lighting, "\n#Compute the light coefficients for diffuse and specular light for Frontmaterial \n");
       if (locallight) { // Both V and P are points
            lighting = addString(lighting, "DP3 temp1.x, normalEye, reflEyeLightVector;\n", lightindex);
       } else { // V is a point and P is a direction
            lighting = addString(lighting, "DP3 temp1.x, normalEye, lightPosNorm%i;\n", lightindex);
       }
       lighting = addString(lighting, "DP3 temp1.y, normalEye, halfDir%i; \n", lightindex);
       lighting = addString(lighting, "MOV temp1.w, specExp.x;    # now: temp1 = (n * VP_pli , n * h_i , - , s_rm)\n");
       lighting = addString(lighting, "LIT temp1, temp1;\n");

       if (locallight) { //multiply each coefficient (front side) with the specular/diffuse product
            lighting = addString(lighting, "MUL temp1, temp1, distVector.w;             # Light attenuation and spotlight is now multiplied with\n");
            lighting = addString(lighting, "# the ambient, diffuse and specular coefficients \n");
       }
    } else { // Backside
       lighting = addString(lighting, "\n#Compute the light coefficients for diffuse and specular light for Backmaterial \n");
       if (locallight) { // Both V and P are points
		lighting = addString(lighting, "DP3 temp1.x, -normalEye, reflEyeLightVector;\n", lightindex);
       } else { // V is a point and P is a direction
	       lighting = addString(lighting, "DP3 temp1.x, -normalEye, lightPosNorm%i; \n", lightindex);
       }

      lighting = addString(lighting, "DP3 temp1.y, -normalEye, halfDir%i; \n", lightindex);
      lighting = addString(lighting, "MOV temp1.w, specExp.x;   # now: temp1 = (-n * VP_pli , -n * h_i , - , s_rm) \n");
      lighting = addString(lighting, "LIT temp1, temp1; \n");
      if (locallight) { // multiply each coefficient (back side) with the specular/diffuse product
         lighting = addString(lighting, "MUL temp1, temp1, distVector.w;      # Light attenuation and spotlight is now multiplied with\n");
         lighting = addString(lighting, "# the ambient, diffuse and specular coefficients \n");
      }

    }
  }
  else { // Local viewer
    if (front) {
      lighting = addString(lighting, "\n# Compute the normalized half-angle vector \n");
      lighting = addString(lighting, "ADD halfVector, lightPos%i, sphereEyeVector; \n", lightindex);
      lighting = addString(lighting, "DP3 halfVector.w, halfVector, halfVector; # halfVector.w = halfVector\n");
      lighting = addString(lighting, "RSQ halfVector.w, halfVector.w; # halfVector.w = 1/|halfVector\n");
      lighting = addString(lighting, "MUL halfVector, halfVector, halfVector.w; # normalized halfangle\n");
      lighting = addString(lighting, "\n#Compute the light coefficients for diffuse and specular light for Frontmaterial \n");
      if (locallight) { // Both V and P are points
         lighting = addString(lighting, "DP3 temp1.x, normalEye, reflEyeLightVector;\n", lightindex);
      } else { // V is a point and P is a direction
         lighting = addString(lighting, "DP3 temp1.x, normalEye, lightPosNorm%i;\n", lightindex);
      }
      lighting = addString(lighting, "DP3 temp1.y, normalEye, halfVector; \n");
      lighting = addString(lighting, "MOV temp1.w, specExp.x;   # now: temp1 = (n * VP_pli , n * h_i , - , s_rm)\n");
      lighting = addString(lighting, "LIT temp1, temp1;\n");
      if (locallight) { //multiply each coefficient (front side) with the specular/diffuse product
         lighting = addString(lighting, "MUL temp1, temp1, distVector.w; # Light attenuation and spotlight is now multiplied with\n");
         lighting = addString(lighting, "# the ambient, diffuse and specular coefficients \n");
      }
    }
    else { // Backside
      lighting = addString(lighting, "\n#Compute the light coefficients for diffuse and specular light for Backmaterial \n");
      if (locallight) { // Both V and P are points
         lighting = addString(lighting, "DP3 temp1.x, -normalEye, reflEyeLightVector;\n", lightindex);
      } else { // V is a point and P is a direction
         lighting = addString(lighting, "DP3 temp1.x, -normalEye, lightPosNorm%i; \n", lightindex);
      }
      lighting = addString(lighting, "DP3 temp1.y, -normalEye, halfVector; \n");
      lighting = addString(lighting, "MOV temp1.w, specExp.x; # now: temp1 = (-n * VP_pli , -n * h_i , - , s_rm) \n");
      lighting = addString(lighting, "LIT temp1, temp1; \n");
      if (locallight) { //multiply each coefficient (back side) with the specular/diffuse product
         lighting = addString(lighting, "MUL temp1, temp1, distVector.w; # Light attenuation and spotlight is now multiplied with\n");
            lighting = addString(lighting, "# the ambient, diffuse and specular coefficients \n");

       }
    }
  }
} // computeDiffuseAndSpecularComponents()

/*
  Constructs the string that multiplies the ambient, diffuse (and specular,
if not separated) light/material product
  with the ambient, diffuse (and specular if not separated) coefficients
(which are earlier multiplied with spotlight
  and distance attenuation if they are enabled) for each lightsource.
  The colors are then summed up in front (or back) primary color
*/
static void
computePrimaryColor(GLboolean front,
                              GLboolean separate_specular,
                              GLint lightindex)
{
  if (front) {
    lighting = addString(lighting, "\n# Adding up all the color components for Front Material\n");
    lighting = addString(lighting, "MAD primFront.xyz, temp1.x, ambientCol%i, primFront; \n", lightindex);
    lighting = addString(lighting, "MAD primFront.xyz, temp1.y, diffuseCol%i, primFront; \n", lightindex);
    if (!separate_specular) { // The specular color is added up with the primary color
       //lighting = addString(lighting, "MUL oColorPrimFront, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD oColorPrimFront, oColorPrimFront, primFront; \n");
       //lighting = addString(lighting, "MOV oColorPrimFront, primFront; \n");
       lighting = addString(lighting, "MAD primFront, temp1.z, specularCol%i, primFront; \n", lightindex);
       
       //lighting = addString(lighting, "MUL secFront, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD secFront, secFront, primFront; \n");
       //lighting = addString(lighting, "MOV oColorPrimFront, secFront; \n");
       //lighting = addString(lighting, "ADD oColorPrimFront, secFront, primFront; \n");
	    
       //lighting = addString(lighting, "ADD secFront, primFront, {0.2, 0.2, 0.2, 0.0}; \n");
       //lighting = addString(lighting, "MOV oColorPrimFront, secFront; \n");
       
       //secFront
//       lighting = addString(lighting, "MAD primFront.xyz, temp1.z, specularCol%i, primFront; \n", lightindex);
//	    oColorPrimFront
       //lighting = addString(lighting, "MOV secBack, {0.0, 0.0, 0.0, 0.0}; \n");
       //lighting = addString(lighting, "MOV primBack, {0.0, 0.0, 0.0, 0.0}; \n");
       //lighting = addString(lighting, "TEMP foos;\n");
	    
       //lighting = addString(lighting, "MUL primBack, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD primFront.xyz, primBack, primFront; \n");
	    
       //lighting = addString(lighting, "MUL secFront, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD primFront.xyz, secFront, primFront; \n");
	    
       //lighting = addString(lighting, "MOV normalEye, {0.2, 0.5, 1.7, 1.2}; \n");
       //lighting = addString(lighting, "MUL normalEye, temp1.z, {0.2, 0.5, 1.7, 1.2}; \n");
       //lighting = addString(lighting, "MUL normalEye, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD primFront, primFront, normalEye; \n");
       
       //lighting = addString(lighting, "MUL temp1, temp1.z, specularCol%i; \n", lightindex);
       //lighting = addString(lighting, "ADD primFront, temp1, primFront; \n");
       
    }
    //lighting = addString(lighting, "MOV oColorPrimFront, primFront; \n");
  }
  else { // Back material
    lighting = addString(lighting, "\n# Adding up all the color components for Back Material\n");
    lighting = addString(lighting, "MAD primBack.xyz, temp1.x, ambientCol%i, primBack; \n", lightindex);
    lighting = addString(lighting, "MAD primBack.xyz, temp1.y, diffuseCol%i, primBack; \n", lightindex);
    if (!separate_specular) { // The specular color is added up with the primary color
	    lighting = addString(lighting, "MAD primBack.xyz, temp1.z, specularCol%i, primBack; \n", lightindex);
    }
  }
} // computePrimaryColor()

/*
  Constructs the string that multiplies the specular light/material product
  with the specular coefficient (which is earlier multiplied with spotlight
  and distance attenuation if they are enabled) for each lightsource.
  The colors are then summed up in front (or back) secondary color
 */
static void computeSecondaryColor(GLboolean front,
                                GLint lightindex)
{
  if (front) {
    lighting = addString(lighting, "MAD secFront.xyz, temp1.z, specularCol%i, secFront; \n", lightindex);
  }
  else {
    lighting = addString(lighting, "MAD secBack.xyz, temp1.z, specularCol%i, secBack; \n", lightindex);
  }
} // computeSecondaryColor()

/*
  Constructs the string that adds the color of the global ambient light
  and the emissive light/material color product to the final primary color.
 */
static void
computeSceneColor(GLboolean front, GLboolean separate_specular)
{
  if (front) {
    lighting = addString(lighting, "\n# The final addition of scene color and moving it to the output for Front Material\n");
    lighting = addString(lighting, "ADD oColorPrimFront, primFront, sceneLight;\n");
    //lighting = addString(lighting, "MOV primFront.w, {0.0}; \n");
    //lighting = addString(lighting, "MOV oColorPrimFront, primFront; \n");
    if (separate_specular) { // The secondary color is moved to the output register
       lighting = addString(lighting, "MOV oColorSecFront, secFront;\n");
    }
  }
  else { // Back material
    lighting = addString(lighting, "\n# The final addition of scene color and moving it to the output for Back Material\n");
    lighting = addString(lighting, "ADD oColorPrimBack, primBack, sceneLight;\n");
    if (separate_specular) { // The secondary color is moved to the output register
       lighting = addString(lighting, "MOV oColorSecBack, secBack;\n");
    }
  }
} // computeSceneColor()


/*
  Creates the string that moves the input color into the output register
  Is called when lighting is disabled
 */
static void constructNoLighting(GLboolean c_es)
{
   lighting = addString(lighting, "MOV oColorPrimFront, iColorPrim;\n");
   if (c_es) {
       lighting = addString(lighting, "MOV oColorSecFront, iColorSec;\n");
   }
} // constructNoLighting()

/*
  This is the main function of constructing a vertex program
  It calls the functions necessary to construct the vertex program strings
 */
void constructVertexProgram()
{
	initialized=0;
   constructInitialization();
   constructTransforms();
   constructLighting();
} // constructVertexProgram()


/* This function creates a program string out of the
different parts that it should consist of and returning
it as a string. Also note that the function allocates memory to hold the
string, thus requiring the driver to free the memory when it is finished
with the program string.
*/
char *
getVertexProgram()
{
    // allocates the necessary amount of memory to
    // the program parameter
    char *program = (GLubyte*)malloc(strlen(initialization) +
                                          strlen(modelviewtransform) +
                                          strlen(normaltransform) +
                                          strlen(normalizationstr) +
                                          strlen(rescalingstr) +
                                          strlen(projectiontransform) +
                                          strlen(lighting) + 1 +1000);

    // The concatenation of the different parts of the vertex
    // program using the strcat() function of the C library.
    // After the last call to strcat(), program holds the whole
    // vertex program.
    strcat(program, initialization);
    strcat(program, "MOV primFront, {0.0, 0.0, 0.0, 0.0};\n");
    strcat(program, modelviewtransform);
    strcat(program, normaltransform);
    strcat(program, normalizationstr);
    strcat(program, rescalingstr);
    strcat(program, projectiontransform);
    strcat(program, lighting);
    strcat(program, "END\n");
    return program;
} // getVertexProgram()

	

/*
  Constructs the string for parameter initialization part of the vertex program
 */
void
constructInitialization()
{
   int i, c_es;
   GLboolean v_bs, t_bs;
   const char *start1 =
    "!!ARBvp1.0\n"

    "ATTRIB iPos = vertex.position; \n"
    "ATTRIB iNormal = vertex.normal; \n";

   const char *start2 =
    "PARAM mvm[4] = { state.matrix.modelview };\n"
    "PARAM mvinv[4] = { state.matrix.modelview.invtrans }; # The inversetransposed modelview matrix \n"
    "PARAM proj[4] = { state.matrix.projection }; # The projection matrix\n";

   const char *start3 =
       "PARAM specExp = state.material.shininess; # Specular exponent \n"
       "PARAM sceneLight = state.lightmodel.scenecolor; # front scene color = a_cs * a_cm + e_cm \n"
       "PARAM eyePositionConst = { 0.0, 0.0, 0.0, 1.0 }; # Position of eye \n";
   const char *start4 =
       "TEMP normalEye, eyePosition, eyePositionNH, reflEyeLightVector, sphereEyeVector, halfVector, temp1, primFront, secBack, primBack, secFront, distVector;\n"
       "OUTPUT oPos = result.position; \n";

   if (!(initialized & 1)) {
       initialized += 1;
   } else {
       free(initialization);
   }
   initialization = (char*)malloc(strlen(start1) + 1);

   if (!maxlights) {
       glGetIntegerv(GL_MAX_LIGHTS, &maxlights);
   }
   glGetBooleanv(GL_LIGHT_MODEL_LOCAL_VIEWER, &v_bs);
   glGetIntegerv(GL_LIGHT_MODEL_COLOR_CONTROL, &c_es);
   glGetBooleanv(GL_LIGHT_MODEL_TWO_SIDE, &t_bs);

   strcpy(initialization, start1);
   //initialization = addString(initialization, start);
   if (glIsEnabled(GL_LIGHTING)) { // Lighting is enabled
       if (glIsEnabled(GL_RESCALE_NORMAL)) {
          initialization = addString(initialization, "PARAM rescaleScale = program.local[40];\n");
       }
       initialization = addString(initialization,start2);
       for (i=0; i < maxlights; i++) {
           if (glIsEnabled(GL_LIGHT0+i) ) { // Checking if the lightsource is enabled
               GLfloat position[] = {0, 0, 0, 0};
               GLfloat cutoff[] = {0.0};

            glGetLightfv(GL_LIGHT0+i, GL_POSITION, position);
            glGetLightfv(GL_LIGHT0+i, GL_SPOT_CUTOFF, cutoff);
            initialization = addString(initialization, "PARAM lightPos%i = state.light[%i].position; # Position of the lightsource \n", i, i);
            initialization = addString(initialization, "PARAM ambientCol%i = state.lightprod[%i].ambient; # Product of ambient color of material and color of scene \n", i, i);
            initialization = addString(initialization, "PARAM diffuseCol%i = state.lightprod[%i].diffuse; # Product of diffuse color of material and diffuse  intensity of light \n", i, i);
            initialization = addString(initialization, "PARAM specularCol%i = state.lightprod[%i].specular; # Product of specular color of material and specular intensity of light \n", i, i);
            if (!v_bs) { // Viewer is infinite
               initialization = addString(initialization, "PARAM halfDir%i = state.light[%i].half; # Halfangle of lightsource, for infinite eyecoordinates \n", i, i);
            }
            if (position[3] != 0) { // Lightsource is local
               initialization = addString(initialization, "PARAM lightAttCoeff%i = state.light[%i].attenuation; # Attenuation coefficients \n", i, i);
               if (cutoff[0] != 180.0) { // Lightsource is a spotlight
                  initialization = addString(initialization, "PARAM lightSpotCos%i = state.light[%i].spot.direction; # The normalized vector of the spotlight direction\n", i, i);
                  //initialization = addString(initialization, "PARAM lightSpotCos%i = program.local[50]; # The normalized vector of the spotlight direction\n", i);
                  initialization = addString(initialization, "PARAM lightSpotDir%i = program.local[%i]; # The normalized vector of the spotlight direction\n", i, i+16);
               }
            } else { // Lightsource is infinite
               initialization = addString(initialization, "PARAM lightPosNorm%i = program.local[%i]; # The normalized vector of the light position \n", i, i);
            }
         }
      }

      initialization = addString(initialization, start3);
   }
   else { // Lighting is disabled
      initialization = addString(initialization, "ATTRIB iColorPrim = vertex.color.primary; \n");
      if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is separated into secondary color
         initialization = addString(initialization, "ATTRIB iColorSec = vertex.color.secondary; \n");
      }
      initialization = addString(initialization, start2);
   }
   initialization = addString(initialization, start4);
   initialization = addString(initialization, "OUTPUT oColorPrimFront = result.color.front.primary; \n");
   //initialization = addString(initialization, "OUTPUT oColorPrimFront = result.color; \n");
   
   if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is separated into secondary color
      initialization = addString(initialization, "OUTPUT oColorSecFront = result.color.front.secondary; \n");
   }
   if (glIsEnabled(GL_LIGHTING)) { // Lighting is enabled
     if (t_bs) { // Two-side coloring is enabled
         initialization = addString(initialization, "OUTPUT oColorPrimBack = result.color.back.primary; \n");
         if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is separated into secondary color
            initialization = addString(initialization, "OUTPUT oColorSecBack = result.color.back.secondary; \n");
         }
       }
   }
} // constructInitialization()

/*
  Calls the functions that do normal transformations, normalization and
rescaling.
  Modelview and projection transformations are defined at top of the
program in separate strings.
 */
void constructTransforms()
{
   normalTransform();
   normalization();
   rescaling();
} // constructTransforms()

/*
  Constructs the string for the lighting part of the
  vertex program
 */
void
constructLighting()
{
   GLboolean t_bs, v_bs;
   GLint c_es, i;

   if (!(initialized & 16)) {
       lighting = (char*)malloc(strlen("\0"));
       initialized += 16;
       //if (!glIsEnabled(GL_LIGHTING)) {
           *lighting = '\0';
       // return;
       //}
   } else if (*lighting != '\0') {
       lighting = (char*)realloc(lighting, strlen("\0"));
       *lighting = '\0';
   }

   glGetBooleanv(GL_LIGHT_MODEL_LOCAL_VIEWER, &v_bs);
   glGetIntegerv(GL_LIGHT_MODEL_COLOR_CONTROL, &c_es);
   glGetBooleanv(GL_LIGHT_MODEL_TWO_SIDE, &t_bs);

   if (glIsEnabled(GL_LIGHTING)) { // Lighting is enabled
       computeVertexEyeVector();
       for (i = 0; i < maxlights; i++) {
            if (glIsEnabled(GL_LIGHT0+i)) { // Checking if lightsource is enabled
               GLfloat position[] = {0, 0, 0, 0};
               glGetLightfv(GL_LIGHT0+i, GL_POSITION, position);
               lighting = addString(lighting, "\n#------- LIGHTSOURCE %i ----- ---\n", i);
                  // computeHalfAngleVector(GL_LIGHT0+i, v_bs);

               if (position[3] != 0.0) { // LOCAL LIGHTSOURCE
                   computeAttenuation(i);
                   computeSpotlight(i);
                   computeDiffuseAndSpecularComponents(GL_TRUE, GL_TRUE, v_bs, i);
                   if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is to be separated into secondary color
                       computePrimaryColor(GL_TRUE, GL_TRUE, i);
                       computeSecondaryColor(GL_TRUE, i);
                   } else { // primary and secondary color are put together
                       computePrimaryColor(GL_TRUE, GL_FALSE, i);
                   }
                   if (t_bs) { // Backside lighting
                       computeDiffuseAndSpecularComponents(GL_TRUE, GL_FALSE, v_bs, i);
                       if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is to be separated into secondary color
                          computePrimaryColor(GL_FALSE, GL_TRUE, i);
                          computeSecondaryColor(GL_FALSE, i);
                       } else { // primary and secondary color are put together
                          computePrimaryColor(GL_FALSE, GL_FALSE, i);
                       }
                   }
               } else { // INFINITE LIGHTSOURCE
                   computeDiffuseAndSpecularComponents(GL_FALSE, GL_TRUE, v_bs, i);
                   if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is to be separated into secondary color
                       computePrimaryColor(GL_TRUE, GL_TRUE, i);
                       computeSecondaryColor(GL_TRUE, i);
                   } else { // primary and secondary color are put together
                       computePrimaryColor(GL_TRUE, GL_FALSE, i);
                   }
                   if (t_bs) { // Backside lighting
                       computeDiffuseAndSpecularComponents(GL_FALSE, GL_FALSE, v_bs, i);
                       if (c_es == GL_SEPARATE_SPECULAR_COLOR) { // Specular color is to be separated into secondary color
                          computePrimaryColor(GL_FALSE, GL_TRUE, i);
                          computeSecondaryColor(GL_FALSE, i);
                       } else { // primary and secondary color are put together
                          computePrimaryColor(GL_FALSE, GL_FALSE, i);
                       }
                   }
               } // end: INFINITE LIGHTSOURCE
           } // end: lightsource enabled
       } // end: for-loop
       computeSceneColor(GL_TRUE, c_es == GL_SEPARATE_SPECULAR_COLOR);
       if (t_bs) { // Backside color is calculated
           computeSceneColor(GL_FALSE, c_es == GL_SEPARATE_SPECULAR_COLOR);
       }
   } else { // Lighting is disabled
       constructNoLighting(c_es == GL_SEPARATE_SPECULAR_COLOR);
   }
   //lighting = addString(lighting, "END\n");
} // constructLighting()


/*
  Constructs the string that calculates the normal transformation
  of the vertex program
 */
void normalTransform()
{
   char * normalstring =
      "# ------------ Normal calculations ---------\n"
      "# Transform the normal to eye coordinates. \n"
      "DP3 normalEye.x, mvinv[0], iNormal;\n"
      "DP3 normalEye.y, mvinv[1], iNormal;\n"
      "DP3 normalEye.z, mvinv[2], iNormal;\n"
      "\n# Non-homogeneous Eye position\n"
      "RCP temp1.w, eyePosition.w;\n"
      "MUL eyePositionNH, eyePosition, temp1.w;\n"
      "# ------------ /Normal Calculations -----------\n";

   if (!(initialized & 2)) { // normalstring is not initialized
     if (glIsEnabled(GL_LIGHTING)) { // Lighting is enabled
         normaltransform = (char*)malloc(strlen(normalstring) + 1);
      } else { // Lighting is disabled
         normaltransform = (char*)malloc(strlen("\0"));
      }
      initialized += 2;
      *normaltransform = '\0';
   }
   if (glIsEnabled(GL_LIGHTING)) { // Lighting is enabled
     if (*normaltransform == '\0') { // The initialized string is empty
         normaltransform = (char*)realloc(normaltransform, strlen(normalstring) + 1);
         strcpy(normaltransform, normalstring);
      }
   } else { // Lighting is disabled
     if (*normaltransform != '\0') { // The initialized string is empty
         normaltransform = (char*)realloc(normaltransform, strlen("\0"));
         *normaltransform = '\0';
      }
   }
} // normalTransform()

/*
  Constructs the string that calculates the normalization of normals
  part of the vertex program.
 */
void normalization()
{
   char * normalizationstring =
       "\n# Normalize Eye Normal\n"
       "DP3 normalEye.w, normalEye, normalEye;\n"
       "RSQ normalEye.w, normalEye.w;\n"
       "MUL normalEye, normalEye, normalEye.w;\n";

   if (!(initialized & 4)) { // normalizationstring is not initialized
       if (glIsEnabled(GL_NORMALIZE)) { // Lighting is enabled
           normalizationstr = (char*)malloc(strlen(normalizationstring) + 1);
       } else { // Lighting is disabled
           normalizationstr = (char*)malloc(strlen("\0"));
       }
       initialized += 4;
       *normalizationstr = '\0';
   }

   if (glIsEnabled(GL_NORMALIZE)) { // Normalization is enabled
       if (*normalizationstr == '\0') { // The initialized string is empty
           normalizationstr = (char*)realloc(normalizationstr,
                                              strlen(normalizationstring) + 1);
           strcpy(normalizationstr, normalizationstring);
       }
   } else { // Normalization is disabled
       if (*normalizationstr != '\0') { // The initialized string is empty
           normalizationstr = (char*)realloc(normalizationstr, strlen("\0"));
           *normalizationstr = '\0';
       }
   }
} // normalization()

void rescaling()
{
   char * rescalingstring =
       "\n#Rescaling of normalvector\n"
       "MUL normalEye, normalEye, rescaleScale;\n";

   if (!(initialized & 8)) { // rescalingstring is not initialized
       if (glIsEnabled(GL_RESCALE_NORMAL)) { // Rescaling is enabled
           rescalingstr = (char*)malloc(strlen(rescalingstring) + 1);
       } else { // Rescaling is disabled
           rescalingstr = (char*)malloc(strlen("\0"));
       }
       initialized += 8;
       *rescalingstr = '\0';
   }
   if (glIsEnabled(GL_RESCALE_NORMAL)) { // Rescaling is enabled
       if (*rescalingstr == '\0') { // The initialized string is empty
           rescalingstr = (char*)realloc(rescalingstr,
                                        strlen(rescalingstring) + 1);
           strcpy(rescalingstr, rescalingstring);
       }
   } else { // Rescaling is disabled
       if (*rescalingstr != '\0') { // The initialized string is empty
           rescalingstr = (char*)realloc(rescalingstr, strlen("\0"));
           *rescalingstr = '\0';
       }
   }
} // rescaling()
