//#include "extgl.h"
//#include <GL/openglut.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include "ffp.h"

// -----------------------------------------------------------------------------

static char *program_variables = {
	"!!ARBvp1.0\n"
        "# variable declarations ---------------------------------\n"
	"ATTRIB input_position	= vertex.position;\n"
	"ATTRIB input_normal	= vertex.normal;\n"
	"PARAM  modelview[4]	= { state.matrix.modelview };\n"
	"PARAM  modelview_inv[4]\t= { state.matrix.modelview.invtrans };\n"
	"PARAM  projection[4]	= { state.matrix.projection };\n"
	"PARAM  scene_light	= state.lightmodel.scenecolor;\n"
	"PARAM  shininess	= state.material.shininess;\n"
};

// -----------------------------------------------------------------------------

static char *program_codes = {
	"TEMP   primary_f, primary_b, secondary_f, secondary_b;\n"
	"TEMP   temp1, temp2, position, normal;\n"
	"OUTPUT output_position\t= result.position;\n"
	"OUTPUT output_color_f   = result.color;\n"
	"MOV primary_f, {0.0, 0.0, 0.0, 0.0};\n"
	"MOV primary_b, {0.0, 0.0, 0.0, 0.0};\n"
	"MOV secondary_f, {0.0, 0.0, 0.0, 0.0};\n"
	"MOV secondary_b, {0.0, 0.0, 0.0, 0.0};\n"
        "# transform the vertex to model view --------------------\n"
	"DP4 position.x, modelview[0], input_position;\n"
	"DP4 position.y, modelview[1], input_position;\n"
	"DP4 position.z, modelview[2], input_position;\n"
	"DP4 position.w, modelview[3], input_position;\n"
        "# project the vertex ------------------------------------\n"
	"DP4 output_position.x, projection[0], position;\n"
	"DP4 output_position.y, projection[1], position;\n"
	"DP4 output_position.z, projection[2], position;\n"
	"DP4 output_position.w, projection[3], position;\n"
	"# transform normal --------------------------------------\n"
	"DP3 normal.x, modelview_inv[0], input_normal;\n"
	"DP3 normal.y, modelview_inv[1], input_normal;\n"
	"DP3 normal.z, modelview_inv[2], input_normal;\n"
	"DP3 normal.w, normal, normal;\n"
	"RSQ normal.w, normal.w;\n"
	"MUL normal, normal, normal.w;\n"
};

// -----------------------------------------------------------------------------

/*
 * WARNING : allocating fixed size buffer for unknown size data
 */
static char *add_string(char *dst, const char *src, ...)
{
	va_list ap;

	dst = (char*) realloc(dst, strlen(dst) + strlen(src) + 128);
	va_start(ap, src);
	vsprintf(dst+strlen(dst), src, ap);
	va_end(ap);

	return dst;
}

// -----------------------------------------------------------------------------

/* Add half light parameter
 */
static void add_half_light(GLint id, char **variables)
{
	char tmp[64];

	// Check if already present
	sprintf(tmp, "light%i_half", id);
	if (strstr((*variables), tmp))
		return;

	static char *light_half = {
		"PARAM  light%i_half\t= state.light[%i].half;\n"
	};

	(*variables) = add_string((*variables), light_half, id, id);
}

// -----------------------------------------------------------------------------

/* Add temp light_vector distance
 */
static void add_temp_spotlight(char **variables)
{
	// Check if already present
	if (strstr((*variables), "distance"))
		return;

	const char *temp = {
                "TEMP   distance;\n"
	};

	(*variables) = add_string((*variables), temp);
}

// -----------------------------------------------------------------------------

/* Add non homogeneous position
 */
static void non_homogeneous_position(char **variables, char **codes)
{
	// Check if already present
	if (strstr((*variables), "position_nh"))
		return;

	const char *non_homogeneous = {
		"# non-homogeneous position ------------------------------\n"
		"RCP temp1.w, position.w;\n"
		"MUL position_nh, position, temp1.w;\n"
	};

	(*variables) = add_string((*variables), "TEMP   position_nh;\n");
	(*codes) = add_string((*codes), non_homogeneous);
}

// -----------------------------------------------------------------------------

/* Constructs the string that calculates the distance attenuation
 * for the lightsource
 */
static void attenuation(GLint id, char **variables, char **codes)
{
	static char *light_attenuation = {
		"PARAM  light%i_attenuation\t= state.light[%i].attenuation;\n"
	};

	const char *attenuation = {
		"# compute light direction and distance vector -----------\n"
                "SUB temp1, light%i_position, position_nh;\n"
		"# d = |lightposition - eyeposition|\n"
		"DP3 temp1.w, temp1, temp1;\n"
		"RSQ distance.w, temp1.w;\n"
		"MUL temp2, temp1, distance.w;\n"
		"DST distance, temp1.w, distance.w;\n"
		"# compute distance attenuation --------------------------\n"
		"DP3 distance.w, distance, light%i_attenuation;\n"
		"RCP distance.w, distance.w;\n"
	};

	(*variables) = add_string((*variables), light_attenuation, id, id);
	(*codes) = add_string((*codes), attenuation, id, id);
}


// -----------------------------------------------------------------------------

/*
 * Constructs the string that calculatest the spotlight component
 * for the lightsource
 */
static void spot_light(GLint id, char **variables, char **codes)
{
	GLfloat angle[1];
	glGetLightfv(GL_LIGHT0 + id, GL_SPOT_CUTOFF, angle);
	if (angle[0] == 180.0)
		return;

	static char *spot_light = {
		"PARAM  spot%i_direction\t= state.light[%i].spot.direction;\n"
	};

	const char *spot = {
		"# compute spotlight cone attenuation --------------------\n"
		"DP3 temp1.w, spot%i_direction, spot%i_direction;\n"
		"RSQ temp1.w, temp1.w;\n"
		"MUL temp1,  spot%i_direction, temp1.w;\n"
		"DP3 temp1.y, temp1, -temp2;\n"
		"ADD temp1.x, temp1.y, -spot%i_direction.w;\n"
		"MOV temp1.w, light%i_attenuation.w;\n"
		"LIT temp1, temp1;\n"
		"MUL distance.w, distance.w, temp1.z;\n"
	};

	(*variables) = add_string((*variables), spot_light, id, id);
	(*codes) = add_string((*codes), spot, id, id, id, id, id);
}

// -----------------------------------------------------------------------------

/* Add scene light
 */
static void scene_light( char **codes,
			 GLboolean front,
			 GLboolean separate_specular)
{
	static char *scene = {
		"# final color computation -------------------------------\n"
                "ADD output_color_%c, primary_%c, scene_light;\n"
	};

	static char *scene_specular = {
		"# final color computation -------------------------------\n"
                "MOV output_specular_%c, secondary_%c;\n"
	};

	// Front or back
	char c = 'f';
	if (!front) {
		c = 'b';
	}

	(*codes)=add_string((*codes), scene, c, c);
	if (separate_specular) {
		(*codes)=add_string((*codes), scene_specular, c, c);
	}
}

// -----------------------------------------------------------------------------

/*
 * Constructs the string that calculates the diffuse and specular
 * components of the lightsource.
 */
static void light_diffuse_specular_components( GLint id,
					       char **variables,
					       char **codes,
					       GLboolean locallight,
					       GLboolean front,
					       GLboolean localviewer )
{
	static char *local_light = {
		"DP3 temp1.x, %cnormal, temp2;\n"
	};
	static char *light_normalize = {
		"DP3 temp2.w, light%i_position, light%i_position;\n"
		"RSQ temp2.w, temp2.w;\n"
		"MUL temp2, light%i_position, temp2.w;\n"
	};
	static char *infinite_light = {
		"DP3 temp1.x, %cnormal, temp2;\n"
	};

	if (!localviewer) {
		// Infinite viewer

		// Add half light to parameter
                add_half_light(id, variables);

		// Negate if we are doing backface
		char negate = ' ';
		if (!front) {
                        negate = '-';
		} else if (!locallight) {
			
			(*codes) = add_string( (*codes), light_normalize,
					       id, id, id );
		}

		if (locallight) {
			(*codes) = add_string((*codes), local_light, negate);
		} else {
			(*codes) = add_string((*codes), infinite_light, negate);
		}

		static char *infinite_viewer = {
			"DP3 temp1.y, %cnormal, light%i_half;\n"
			"MOV temp1.w, shininess.x;\n"
			"LIT temp1, temp1;\n"
		};
		(*codes) = add_string((*codes), infinite_viewer, negate, id);

		if (locallight) {
			(*codes) = add_string( (*codes),
					       "MUL temp1,temp1,distance.w;\n");
		}
	} else {
#if 0
		// Local viewer
		if (front) {
			lighting = add_string(lighting, "\n# Compute the normalized half-angle vector \n");
			lighting = add_string(lighting, "ADD halfVector, lightPos%i, sphereEyeVector; \n", lightindex);
			lighting = add_string(lighting, "DP3 halfVector.w, halfVector, halfVector; # halfVector.w = halfVector\n");
			lighting = add_string(lighting, "RSQ halfVector.w, halfVector.w; # halfVector.w = 1/|halfVector\n");
			lighting = add_string(lighting, "MUL halfVector, halfVector, halfVector.w; # normalized halfangle\n");
			lighting = add_string(lighting, "\n#Compute the light coefficients for diffuse and specular light for Frontmaterial \n");
			if (locallight) {
				// Both V and P are points
				lighting = add_string(lighting, "DP3 temp1.x, normalEye, reflEyeLightVector;\n", lightindex);
			} else {
				// V is a point and P is a direction
				lighting = add_string(lighting, "DP3 temp1.x, normalEye, lightPosNorm%i;\n", lightindex);
			}
			lighting = add_string(lighting, "DP3 temp1.y, normalEye, halfVector; \n");
			lighting = add_string(lighting, "MOV temp1.w, specExp.x;   # now: temp1 = (n * VP_pli , n * h_i , - , s_rm)\n");
			lighting = add_string(lighting, "LIT temp1, temp1;\n");
			if (locallight) {
				// multiply each coefficient (front side) with the specular/diffuse product
				lighting = add_string(lighting, "MUL temp1, temp1, distVector.w; # Light attenuation and spotlight is now multiplied with\n");
				lighting = add_string(lighting, "# the ambient, diffuse and specular coefficients \n");
			}
		} else {
			// Backside
			lighting = add_string(lighting, "\n#Compute the light coefficients for diffuse and specular light for Backmaterial \n");
			if (locallight) {
				// Both V and P are points
				lighting = add_string(lighting, "DP3 temp1.x, -normalEye, reflEyeLightVector;\n", lightindex);
			} else {
				// V is a point and P is a direction
				lighting = add_string(lighting, "DP3 temp1.x, -normalEye, lightPosNorm%i; \n", lightindex);
			}
			lighting = add_string(lighting, "DP3 temp1.y, -normalEye, halfVector; \n");
			lighting = add_string(lighting, "MOV temp1.w, specExp.x; # now: temp1 = (-n * VP_pli , -n * h_i , - , s_rm) \n");
			lighting = add_string(lighting, "LIT temp1, temp1; \n");
			if (locallight) {
				// multiply each coefficient (back side) with the specular/diffuse product
				lighting = add_string(lighting, "MUL temp1, temp1, distVector.w; # Light attenuation and spotlight is now multiplied with\n");
				lighting = add_string(lighting, "# the ambient, diffuse and specular coefficients \n");
			}
		}
#endif
	}
}

// -----------------------------------------------------------------------------

/*
 * Constructs the string that multiplies the ambient, diffuse (and specular,
 * if not separated) light/material product
 * with the ambient, diffuse (and specular if not separated) coefficients
 *(which are earlier multiplied with spotlight
 * and distance attenuation if they are enabled) for each lightsource.
 * The colors are then summed up in front (or back) primary color
 */
static void primary_color( GLint id,
                           char **codes,
			   GLboolean front,
			   GLboolean separate_specular )
{
	static char *primary = {
		"MAD primary_%c.xyz, temp1.x, light%i_ambient, primary_%c;\n"
		"MAD primary_%c.xyz, temp1.y, light%i_diffuse, primary_%c;\n"
	};
	static char *specular = {
		"MAD primary_%c.xyz, temp1.z, light%i_specular, primary_%c; \n"
	};

	// Front or back
	char c = 'f';
	if (!front) {
		c = 'b';
	}

	(*codes) = add_string((*codes), primary, c, id, c, c,id, c);
	if (!separate_specular) {
		// The specular color is added up with the primary color
		(*codes) = add_string((*codes), specular, c, id, c);
	}
}

// -----------------------------------------------------------------------------

/*
 * Constructs the string that multiplies the specular light/material product
 * with the specular coefficient (which is earlier multiplied with spotlight
 * and distance attenuation if they are enabled) for each lightsource.
 * The colors are then summed up in front (or back) secondary color
 */
static void secondary_color(GLint id, char **codes, GLboolean front)
{
	static char *secondary = {
		"MAD secondary_%c.xyz,temp1.z,light%i_specular,secondary_%c;\n"
	};

	if (front) {
		(*codes) = add_string((*codes), secondary, 'f', id, 'f');
	} else {
		(*codes) = add_string((*codes), secondary, 'b', id, 'b');
	}
}

// -----------------------------------------------------------------------------

static void process_light(GLint id, char **variables, char **codes)
{
	static char *light_variables = {
		"PARAM  light%i_position\t= state.light[%i].position;\n"
		"PARAM  light%i_ambient\t= state.lightprod[%i].ambient;\n"
		"PARAM  light%i_diffuse\t= state.lightprod[%i].diffuse;\n"
		"PARAM  light%i_specular\t= state.lightprod[%i].specular;\n"
	};
	static char *light_codes = {
		"# light %i -----------------------------------------------\n"
	};


	GLfloat position[] = {0, 0, 0, 0};
	GLfloat cutoff[] = {0.0};
	glGetLightfv(GL_LIGHT0 + id, GL_POSITION, position);
	glGetLightfv(GL_LIGHT0 + id, GL_SPOT_CUTOFF, cutoff);

	// Add variables for this light
        (*variables) = add_string( (*variables), light_variables,
				 id, id, id, id, id, id, id, id );

	// Add code for this lights
	GLboolean	local_viewer, two_side;
	GLint		color_control;
	glGetBooleanv(GL_LIGHT_MODEL_LOCAL_VIEWER, &local_viewer);
	glGetIntegerv(GL_LIGHT_MODEL_COLOR_CONTROL, &color_control);
	glGetBooleanv(GL_LIGHT_MODEL_TWO_SIDE, &two_side);

	if (position[3] != 0.0) {
                // Local light source
                add_temp_spotlight(variables);
                non_homogeneous_position(variables, codes);
	        (*codes) = add_string((*codes), light_codes, id);
                attenuation(id, variables, codes);
                spot_light(id, variables, codes);
		light_diffuse_specular_components(id, variables, codes, GL_TRUE, GL_TRUE, local_viewer);
		if (color_control == GL_SEPARATE_SPECULAR_COLOR) {
			// Specular color is to be separated into secondary color
                        primary_color(id, codes, GL_TRUE, GL_TRUE);
                        secondary_color(id, codes, GL_TRUE);
		} else {
			// Primary and secondary color are put together
                        primary_color(id, codes, GL_TRUE, GL_FALSE);
		}
		if (two_side) {
			// Backside lighting
			light_diffuse_specular_components(id, variables, codes, GL_TRUE, GL_FALSE, local_viewer);
			if (color_control == GL_SEPARATE_SPECULAR_COLOR) {
				// Specular color is to be separated into secondary color
        	                primary_color(id, codes, GL_FALSE, GL_TRUE);
	                        secondary_color(id, codes, GL_FALSE);
			} else { // Primary and secondary color are put together
	                        primary_color(id, codes, GL_FALSE, GL_FALSE);
			}
		}
	} else {
		// Infinite light source
	        (*codes) = add_string((*codes), light_codes, id);
		light_diffuse_specular_components(id, variables, codes, GL_FALSE, GL_TRUE, local_viewer);
		if (color_control == GL_SEPARATE_SPECULAR_COLOR) {
			// Specular color is to be separated into secondary color
                        primary_color(id, codes, GL_TRUE, GL_TRUE);
                        secondary_color(id, codes, GL_TRUE);
		} else {
			// Primary and secondary color are put together
                        primary_color(id, codes, GL_TRUE, GL_FALSE);
		}
		if (two_side) {
			// Backside lighting
			light_diffuse_specular_components(id, variables, codes, GL_FALSE, GL_FALSE, local_viewer);
			if (color_control == GL_SEPARATE_SPECULAR_COLOR) {
				// Specular color is to be separated into secondary color
        	                primary_color(id, codes, GL_FALSE, GL_TRUE);
	                        secondary_color(id, codes, GL_FALSE);
			} else { // Primary and secondary color are put together
	                        primary_color(id, codes, GL_FALSE, GL_FALSE);
			}
		}
	}
}

char *make_vertex_program(void)
{
	char *program;
	char *variables;
	char *codes;
	GLint max_lights;
	GLint i;

	static char *secondary_f = {
		"OUTPUT output_specular_f\t= result.color.front.secondary;\n"
	};
	static char *secondary_b = {
		"OUTPUT output_specular_b\t= result.color.back.secondary;\n"
	};
	static char *primary_b = {
		"OUTPUT output_color_b\t= result.color.back.primary;\n"
	};

	// Start with an empty program
	program = NULL;

	// Start variable part with constant things
	variables = (char*) malloc(strlen(program_variables) + 1);
	strcpy(variables, program_variables);

	// Start code with invariant part
	codes = (char*) malloc(strlen(program_codes) + 1);
	strcpy(codes, program_codes);

	// Do lighting
	if (glIsEnabled(GL_LIGHTING)) {
		GLint		color_control;
		GLboolean	two_side;
		glGetIntegerv(GL_LIGHT_MODEL_COLOR_CONTROL, &color_control);
		glGetBooleanv(GL_LIGHT_MODEL_TWO_SIDE, &two_side);
		GLboolean       separate = GL_FALSE;

		if (two_side) {
			variables = add_string(variables, primary_b);
		}

		if (color_control == GL_SEPARATE_SPECULAR_COLOR) {
                        separate = GL_TRUE;
			variables = add_string(variables, secondary_f);
			if (glIsEnabled(GL_LIGHT_MODEL_TWO_SIDE)) {
				variables = add_string(variables, secondary_b);
			}
		}

		glGetIntegerv(GL_MAX_LIGHTS, &max_lights);
		for (i = 0; i < max_lights; i++) {
			if (glIsEnabled(GL_LIGHT0 + i)) {
                                process_light(i, &variables, &codes);
			}
		}
                scene_light(&codes, GL_TRUE, separate);
		if (two_side) {
	                scene_light(&codes, GL_FALSE, separate);
		}
	} else {
	}

	// Make the program
	program = (char*) malloc(strlen(variables) + strlen(codes) + 1);
	memcpy(program, variables, strlen(variables));
	memcpy(program + strlen(variables), codes, strlen(codes) + 1);
	program = add_string(program, "END");
	free(variables);
	free(codes);
	return program;
}
