
from __future__ import division

import defeat

class CondorcetMatrix(object):
   """
   Represents a Condorcet Matrix.
   """

   def __init__(self, candidates, votes):
      """
      Initialize the matrix.
      candidates = a list of strings identifying the candidates
      votes = a 2-D array in which votes[i][j] is the number of voters
              who prefer candidates[i] over candidates[j]
      """
      self._candidates = candidates
      self._votes = votes

   def __repr__(self):
      return 'CondorcetMatrix(%r, %r)' % (self._candidates, self._votes)

   def __len__(self):
      """
      Return the number of candidates in the matrix.
      """
      return len(self._candidates)

   def defeat_matrix(self, defeat_strength):
      """
      Return a CondorcetMatrix with the same candidates as self in which
      votes[i][j] = defeat_strength(votes[i][j], votes[j][i]).
      """
      n = len(self)
      matrix = [[None] * n for i in xrange(n)]
      for i in xrange(n):
         for j in xrange(n):
            matrix[i][j] = defeat_strength(self._votes[i][j],
                                           self._votes[j][i])
      return CondorcetMatrix(self._candidates, matrix)

   def beatpath_matrix(self, defeat_strength):
      matrix = [x[:] for x in self.defeat_matrix(defeat_strength)._votes]
      n = len(matrix)
      while True:
         found_improvement = False
         for i in xrange(n):
            for j in xrange(n):
               if i == j:
                  continue
               new_strength = max(min(matrix[i][k], matrix[k][j])
                                  for k in xrange(n))
               if new_strength > matrix[i][j]:
                  matrix[i][j] = new_strength
                  found_improvement = True
         if not found_improvement:
            break
      return CondorcetMatrix(self._candidates, matrix)


def beatpath_ranking(condorcet_matrix, defeat_strength):
   """
   Return the candidates sorted in order of their number of net beatpath
   victories (beatpath winner is last).  If there is a tie, it is broken
   at random.
   """
   import operator
   import random
   beatpath_matrix = condorcet_matrix.beatpath_matrix(defeat_strength)
   defeat_matrix = beatpath_matrix.defeat_matrix(defeat.copeland)
   candidate_scores = [(c, sum(v)) for c, v in zip(defeat_matrix._candidates,
                                                   defeat_matrix._votes)]
   random.shuffle(candidate_scores)
   candidate_scores.sort(key=operator.itemgetter(1))
   return [x[0] for x in candidate_scores]



