"""Representation of ranked ballots."""

from __future__ import division

import condorcet
from gmpy import mpq as _rational


class RankedBallot(object):
   """
   Represents a ranked ballot which allows equal rankings.
   """

   def __init__(self, ranking, weight=1):
      """
      Create a ranked ballot.
      ranking[i] = a collection of candidate(s) at rank i,
         e.g., ranking = [['A', 'B'], ['C'], ['D', 'E']]
         represents the ranked ballot A>B=C>D=E
      weight = the weight of this ballot
      """
      self._ranking = [set(x) for x in ranking if x]
      self.weight = _rational(weight)

   def __repr__(self):
      return 'RankedBallot(%r, %r)' % (self._ranking, self.weight)

   def __str__(self):
      ranking = '>'.join('='.join(candidate for candidate in rank_set)
                         for rank_set in self._ranking)
      return '%s: %s' % (self.weight, ranking)

   def __nonzero__(self):
      """
      Return True iff self is non-exhausted and has nonzero weight.
      """
      return bool(weight) and bool(ranking)

   def copy(self):
      """
      Return a copy of self.
      """
      return RankedBallot(self._ranking, self.weight)

   def eliminate(self, candidates):
      """
      Eliminate candidates from the ballot,
      preserving the relative order of other candidates.
      """
      self._ranking = [y for y in (set(x) - set(candidates)
                                   for x in self._ranking) if y]

   def candidates(self):
      """
      Return a set of the candidates ranked on this ballot.
      """
      result = set()
      for eq in self._ranking:
         for candidate in eq:
            result.add(candidate)
      return result

   def top_candidates(self):
      """
      Return the set of top-ranked candidates on the ballot.
      """
      return self._ranking[0]
      

class BallotSet(object):
   """
   A collection of RankedBallots
   """

   def __init__(self, ballots):
      self._ballots = list(ballots)
      # Ensure that all candidates are ranked on all ballots.
      self._candidates = set()
      for ballot in self._ballots:
         self._candidates |= ballot.candidates()
      # Candidates not ranked go at the end.
      for ballot in self._ballots:
         other_candidates = self._candidates - ballot.candidates()
         if other_candidates:
            ballot._ranking.append(other_candidates)

   def __repr__(self):
      return 'BallotSet(%r)' % self._ballots

   def __str__(self):
      return '\n'.join(str(ballot) for ballot in self._ballots)

   def num_votes(self):
      """
      Return the total number of votes in this BallotSet.
      """
      return sum(ballot.weight for ballot in self._ballots)

   def candidates(self):
      """
      Return the set of candidates ranked on the ballots.
      """
      return set(self._candidates)

   def _pairwise_preferences(self):
      """
      Returns a dict of the form {(A, B): (votes for A>B)}.
      """
      result = {}
      for a in self._candidates:
         for b in self._candidates:
            result[a, b] = _rational(0)
      for ballot in self._ballots:
         n = len(ballot._ranking)
         for i in xrange(n):
            for j in xrange(i + 1, n):
               for candidate_i in ballot._ranking[i]:
                  for candidate_j in ballot._ranking[j]:
                     result[candidate_i, candidate_j] += ballot.weight
      return result

   def condorcet_matrix(self):
      """
      Return the CondorcetMatrix for self.
      """
      candidates = sorted(self._candidates)
      n = len(candidates)
      preferences = self._pairwise_preferences()
      matrix = [[None] * n for i in xrange(n)]
      for i, x in enumerate(candidates):
         for j, y in enumerate(candidates):
            matrix[i][j] = preferences[x, y]
      return condorcet.CondorcetMatrix(candidates, matrix)

   def copy(self):
      """
      Return a copy of self.
      """
      return BallotSet(b.copy() for b in self._ballots)

   def eliminate(self, candidates):
      """
      Eliminiate the given set of candidates from all ballots.
      """
      for ballot in self._ballots:
         ballot.eliminate(candidates)
      self._candidates -= candidates

   def vote_counts(self):
      """
      Return a {candidate: 1st-choice votes for candidate} dict.
      """
      votes = {}
      for candidate in self._candidates:
         votes[candidate] = _rational(0)
      for ballot in self._ballots:
         candidates = ballot.top_candidates()
         votes_per_candidate = ballot.weight * _rational(1, len(candidates))
         for candidate in candidates:
            votes[candidate] += votes_per_candidate
      return votes

   def weight_candidates(self, weights):
      """
      Reweight the ballots in this BallotSet.
      weights = a dict of {candidate: weight}
      """
      for ballot in self._ballots:
         candidates = ballot.top_candidates()
         n = _rational(len(candidates))
         weight = sum(weights[c] for c in candidates) / n
         ballot.weight *= weight

def _str2rat(x):
   """
   Convert a string to a GMPY rational number.
   """
   try:
      return _rational(x)
   except ValueError:
      return _rational(float(x))


def _parse_ballot(line):
   """
   Convert a line of text to a RankedBallot.
   (See parse_ballots for details.)
   """
   if ':' in line:
      weight, ranking = line.split(':')
      weight = _str2rat(weight.strip())
   else:
      weight = _rational(1)
      ranking = line
   ranking = [[x.strip() for x in e.split('=')] for e in ranking.split('>')]
   return RankedBallot(ranking, weight)


def parse_ballots(ballot_iter):
   """
   Convert ballots in weight:ranking form to a BallotSet.
   ballot_iter = an iterable that yields weight:ranking strings.
   """
   return BallotSet(_parse_ballot(ballot) for ballot in ballot_iter
                                          if ballot.strip())


