#!/usr/bin/env python

"""STV-CLE Vote Calculator

Usage: stv_cle.py [options] num_seats vote_file

Options:

   -h    Display help and exit.
   -v    Verbose mode (displays elimination order and vote counts)
   -d:   Specify the defeat strength for the elimination order
         -d margin    Use Margins (default)
         -d rel       Use Relative Margins
         -d wv        Use Winning Votes

num_seats = the number of candidates to be elected

Each line of vote_file must have the format votes:ranking, where ranking
is a list of candidates separated by = or >.  For example,

42: Memphis > Nashville > Chattanooga > Knoxville
26: Nashville > Chattanooga > Knoxville > Memphis
15: Chattanooga > Knoxville > Nashville > Memphis
17: Knoxville > Chattanooga > Nashville > Memphis

If a candidate is not ranked on a particular ballot, it is assumed to be
ranked below the explicitly-ranked candidates.
"""

from __future__ import division

import ballot
import condorcet
import defeat
import getopt
import gmpy
import operator
import sys


def stvcle_winners(ballots, seats, ranking_method=condorcet.beatpath_ranking,
                   defeat_strength=defeat.margin, outfile=sys.stdout):
   """
   Return the STV-CLE winners for a BallotSet.

   ballots = a BallotSet object
   seats = the number of seats for which candidates are elected
   ranking_method = a function with the parameter list
                    (ballot set, defeat strength function)
                    that returns an elimination order.
   defeat_strength = a function with the parameter list
                     (votes for X over Y, votes for Y over X)
                     that returns the defeat strength of X over Y
   outfile = the file to which the vote counts are printed
   """
   # There must be at least seats+1 candidates to have an election
   candidates = ballots.candidates()
   if len(candidates) <= seats:
      if outfile:
         print >>outfile, 'Too few candidates.'
         print >>outfile
      return candidates
   # Compute the elimination order
   elimination_order = ranking_method(ballots.condorcet_matrix(),
                                      defeat_strength)
   if outfile:
      print >>outfile, 'Elimination order is:'
      for candidate in elimination_order:
         print >>outfile, candidate
      print >>outfile
   # Use the N-B quota
   total_votes = ballots.num_votes()
   quota = total_votes * gmpy.mpq(1, seats+1)
   if outfile:
      print >>outfile, 'Total votes = %f' % total_votes
      print >>outfile, 'Quota = %f' % quota
      print >>outfile
   # Main loop
   original_ballots = ballots
   elected_candidates = set()
   eliminated_candidates = set()
   while len(elected_candidates) < seats:
      # Ballots in initial state except for eliminated candidates
      ballots = original_ballots.copy()
      ballots.eliminate(eliminated_candidates)
      while True:
         # count the votes
         vote_counts = ballots.vote_counts()
         if outfile:
            print >>outfile, 'Vote counts:'
            vote_output = [(vote_counts[candidate], candidate)
                           for candidate in vote_counts]
            vote_output.sort(reverse=True)
            for x in vote_output:
               print '%17.6f %s' % x
            print >>outfile
         # Elect candidates that meet quota
         newly_elected = set()
         for candidate, votes in vote_counts.items():
            if votes > quota:
               newly_elected.add(candidate)
         # Eliminate a candidate if nobody met quota
         if not newly_elected:
            for candidate in elimination_order:
               if candidate not in (elected_candidates|eliminated_candidates):
                  new_elimination = candidate
                  break
            eliminated_candidates.add(new_elimination)
            if outfile:
               print >>outfile, 'Eliminating %s' % new_elimination
               print >>outfile
            break
         # Otherwise, adjust the weights
         if outfile:
            for candidate in newly_elected:
               print >>outfile, 'Electing %s' % candidate
            print >>outfile
         elected_candidates |= newly_elected
         if len(elected_candidates) >= seats:
            return elected_candidates
         weights = {}
         for candidate, votes in vote_counts.items():
            if votes > quota:
               weights[candidate] = (votes - quota) / votes
            else:
               weights[candidate] = gmpy.mpq(1)
         ballots.weight_candidates(weights)
         ballots.eliminate(newly_elected)            


def _main(argv=sys.argv[1:]):
   try:
      options, parameters = getopt.getopt(argv, 'd:hv')
   except getopt.GetoptError, e:
      print >>sys.stderr, e
      return 1
   options = dict(options)
   if '-h' in options:
      print __doc__
      return 0
   outfile = (None, sys.stdout)['-v' in options]
   defeat_strength = options.get('-d', 'm')[0].lower()
   try:
      defeat_strength = {'m': defeat.margin,
                         'r': defeat.relative_margin,
                         'w': defeat.winning_votes}[defeat_strength]
   except KeyError:
      print >>sys.stderr, 'defeat strength option not recognized'
      return 1
   try:
      seats, input_file = parameters
      seats = int(seats)
   except ValueError:
      print >>sys.stderr, 'argument list must be: #seats input_file'
      return 1
   # Finally, we can count the votes
   try:
      ballots = ballot.parse_ballots(file(input_file))
   except Exception, e:
      print >>sys.stderr, e
      return 1
   winners = stvcle_winners(ballots, seats, defeat_strength=defeat_strength,
                            outfile=outfile)
   if outfile:
      print >>outfile, 'WINNERS ARE:'
   for candidate in sorted(winners):
      print candidate

if __name__ == '__main__':
   sys.exit(_main())
