/******************************************************************************
 * 
 * Copyright (C) 2006 MTA SZTAKI <haidegger AT sztaki DOT hu>
 * License: GPL version 2
 *
 * Author: Evert Lammerts
 * License: GPL Version 2
 * Copyright (c) 2006 All rights reserved
 * Last change: 01-sept-2006
 * $Revision: 0.4 $
 * $Author: elammerts $
 * $Date: 2006/07/31 12:05:03
 * 
 * This is the driver that handles real-time ethernet communications
 * in EMC, using RTnet (http://www.rtnet.org).
 * 
 * 
 * INSTALLATION ***************************************************************
 * 
 *   This driver has 7 module parameters:
 * 
 *   fp_period=[long]
 *     This parameter specifies the period of the thread to be created.
 *     Default = 0 - no thread.
 * 
 *   action=[send|recieve] ==> vital
 *     This parameter specifies wether the driver will send or 
 *     recieve data. The names correspond to the functions that
 *     will be exported. No default value, is vital for module.
 *
 *   if send functionality is exported, then we have to set parameter:
 *   ip = [xxx.xxx.x.xxx]
 *     This parameter specifies the ip addres of your rtnet Master.
 *     No default value.
 * 
 *   port=[portnumber]
 *     You usually won't change this, unless some other application
 *     uses the default port (I am not aware of any). Remember that
 *     both the sending and the recieving modules need the same port!
 *     Default 49222.
 * 
 *   timeout=[usec]
 *     This parameter specifies the recieve timeout. This is
 *     only necessary when your modules are out of synch. Note 
 *     that the timeout will not fix the synchronization - that
 *     all depends on your thread period timing. It does however
 *     provide you with the possibility to wait for data when 
 *     none is present on the port. Default = 1.
 * 
 *   nr_of_pins=[int]
 *     This parameter specifies the amount of pins to export from
 *     the module. Default = 0.
 * 
 *   If nr_of_pins > 0, you have to specify the parameter:
 *   pin_cfg=[xxx_yyy]
 *     This parameter specifies the type of pins to export. The
 *     possible values are: bit, u8, s8, u16, s16, u32, s32 and
 *     float. You seperate the values by underscore. 
 * 
 *     Example:
 *     nr_of_pins=3 pin_cfg=u16_s8_float
 * 
 *     Provides the pins hal_rtnet.0-u16, hal_rtnet.1-s8 and
 *     hal_rtnet.2-float
 * 
 *   Example installation:
 * 
 *   Machine 1 (RTnet slave):
 *     bin/halcmd loadrt hal_rtnet action=send ip=192.168.1.100 \
 *                nr_of_pins=3 pin_cfg=bit_float_float fp_period=1000000
 * 
 *   Machine 2 (RTnet master)
 *     bin/halcmd loadrt hal_rtnet action=recieve \
 *                nr_of_pins=3 pin_cfg=bit_float_float fp_period=1000000
 * 
 * 
 * HAL PINS *******************************************************************
 * 
 *   The HAL pins that are exported depend on the module parameters.
 *   They will appear in the form:
 *     hal_rtnet.<nr_of_pin>-<data_type>
 * 
 * 
 * HAL FUNCTIONS **************************************************************
 * 
 *   Two HAL functions can be exported, depending on the action to
 *   be performed:
 * 
 *     if action = send, then     hal_rtnet.send
 *     if action = recieve, then  hal_rtnet.recieve
 * 
 * 
 * Miscellaneous notes ********************************************************
 * 
 *   You can check wether you are executing the HAL functions from a
 *   RT context by uncommenting the first line in the recieve and/or send 
 *   function(s).
 * 
 *   At this moment RTnet is not an integral part of EMC2. Therefore
 *   it is essential to install RTnet in its default location, so
 *   that our import statement can find it: /usr/local/rtnet
 * 
 ******************************************************************************
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU General
 * Public License as published by the Free Software Foundation.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111 USA
 *
 * THE AUTHORS OF THIS LIBRARY ACCEPT ABSOLUTELY NO LIABILITY FOR
 * ANY HARM OR LOSS RESULTING FROM ITS USE.  IT IS _EXTREMELY_ UNWISE
 * TO RELY ON SOFTWARE ALONE FOR SAFETY.  Any machinery capable of
 * harming persons must have provisions for completely removing power
 * from all motors, etc, before persons enter any danger area.  All
 * machinery must be designed to comply with local and national safety
 * codes, and the authors of this software can not, and do not, take
 * any responsibility for such compliance.
 *
 ******************************************************************************/

#ifndef RTAPI
#error This is a realtime component only!
#endif

/*
  HEADERS
  
  I do not like that we need to specify the absolute path to rtnet.h... Maybe if the RTnet interface
  will ever become part of EMC2, this can be changed. For now it works like this.
*/

// Linux headers
#include <linux/net.h>                        /* SOCK_DGRAM */
#include <linux/in.h>                         /* sockaddr_in */
#include <linux/string.h>                     /* strncpy(), strcmp() */
#include <linux/ctype.h>                      /* isspace() */

// RTAI, RTnet and EMC2 headers
#include "/usr/local/rtnet/include/rtnet.h"   /* RTnet, absolute path... */
#include "rtapi.h"                            /* RTAPI realtime OS API */
#include "rtapi_app.h"                        /* RTAPI realtime modules decls */
#include "hal.h"                              /* HAL public API decls */

/*
  MODULE INFO & PARAMS
*/

#ifdef MODULE
  // Module information
  MODULE_AUTHOR("Evert Lammerts");
  MODULE_DESCRIPTION("Real-time ethernet HAL implementation");
  #ifdef MODULE_LICENSE
    MODULE_LICENSE("GPL");
  #endif

  // Module parameters
  static int port = 49222; // Standard server port number (should be in 49152 - 65535)
  MODULE_PARM(port, "i");
  MODULE_PARM_DESC(port, "Server & client port");
  
  static char *action;
  MODULE_PARM(action, "s");
  MODULE_PARM_DESC(action, "Specify send or recieve functionality");
  
  static char *ip;
  MODULE_PARM(ip, "s");
  MODULE_PARM_DESC(ip, "Server IP for sending functionality");
  
  static int64_t timeout = 1;
  MODULE_PARM(timeout, "i");
  MODULE_PARM_DESC(timeout, "Recieve time-out in usec");
  
  static int nr_of_pins = 0; // Nr of pins to export
  MODULE_PARM(nr_of_pins, "i");
  MODULE_PARM_DESC(nr_of_pins, "Nr of pins to export (r/w by default)");
  
  static char *pin_cfg;
  MODULE_PARM(pin_cfg, "s");
  MODULE_PARM_DESC(pin_cfg, "Pin types, in order (from 1, .., n) seperated by an underscore _");
  
  static long fp_period = 0;
  MODULE_PARM(fp_period, "l");
  MODULE_PARM_DESC(fp_period, "floating point thread period (nsecs)");
#endif

#define HAL_RTNET_MAX_PACKET_SIZE 4000
#define HAL_RTNET_MAX_PIN 100

/********************************************************
*              STRUCTURES AND GLOBAL VARIABLES          *
********************************************************/

typedef struct {
  hal_bit_t        *hbit;
  hal_u8_t         *hu8;
  hal_s8_t         *hs8;
  hal_u16_t        *hu16;
  hal_s16_t        *hs16;
  hal_u32_t        *hu32;
  hal_s32_t        *hs32;
  hal_float_t      *hfloat;
} hal_rtnet_data;

typedef union {
  hal_u16_t hal_u16;
  unsigned char c[sizeof(hal_u16_t)];
} u16_to_char;

typedef union {
  hal_s16_t hal_s16;
  unsigned char c[sizeof(hal_s16_t)];
} s16_to_char;

typedef union {
  hal_u32_t hal_u32;
  unsigned char c[sizeof(hal_u32_t)];
} u32_to_char;

typedef union {
  hal_s32_t hal_s32;
  unsigned char c[sizeof(hal_s32_t)];
} s32_to_char;

typedef union {
  hal_float_t hal_float;
  unsigned char c[sizeof(hal_float_t)];
} float_to_char;

static hal_rtnet_data *hal_data;					// HAL pin
static int comp_id;							// Component id
static struct sockaddr_in local_addr, server_addr;			// Address information
static int sockfd;							// Socket
static char *argv[HAL_RTNET_MAX_PIN];					// String array to store pin types
static int packet_size;							// Size of the packet to send
static const char terminator = '\0';					// Data seperator
char pack[sizeof(float) + 1];						// Temporary data storage

/********************************************************
*                    FUNCTION DECLARATIONS              *
********************************************************/

static int export_rtnet(int nr);					// Export functions and pins
static int init_recieve(void);						// Initialize recieve functionality
static int init_send(void);						// Initialize sending functionality
static int parse_pin_cfg(char *cfg);					// Parsing the pin config string
static void recieve(void *arg, long period);				// Recieving HAL function
static void send(void *arg, long period);				// Sending HAL function

/********************************************************
*                     INIT AND EXIT CODE                *
********************************************************/

int rtapi_app_main(void)
{
  // Check parameters
  // First, are we sending or recieving?
  if(action == 0) {
    rtapi_print("hal_rtnet: ERROR: set functionality parameter (action=<recieve|send>)\n");
    return -1;
  }
  
  // If we need an IP address to send to, do we have it?
  if((action[0] == 's') && (((ip[0] == '\0') || (ip[0] == 0)) || (ip == NULL))) {
    rtapi_print("hal_rtnet: ERROR: set ip parameter to use send functionality (ip=<xxx.xxx.x.xxx>)\n");
    return -1;
  }
  
  // If we have > 0 pins, do we have a pin-config string?
  if((nr_of_pins > 0) && (pin_cfg <= 0)) {
    rtapi_print("hal_rtnet: ERROR: set pin type(s) parameter (nr_of_pins=<x> pin_cfg=<bit|u8|s8|u16|...>_<bit|u8|s8|u16|...>_....)\n");
    return -1;
  }
  else if(nr_of_pins > 0) {
    if(parse_pin_cfg(pin_cfg) != 0) { // the pin config string or nr_of_pins is not right
      rtapi_print("hal_rtnet: ERROR: nr_of_pins doesn't correspond to the amount of data types in pin_cfg\n");
      return -1;
    }
  }
  // We're good, lets continue

  // Initializing the HAL module
  comp_id = hal_init("hal_rtnet");
  if(comp_id < 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: hal_init() failed\n");
    return -1;
  }
  
  rtapi_set_msg_level(RTAPI_MSG_DBG); // DEBUG PURPOSES

  if(action[0] == 'r') {  // We need to recieve data
    if(init_recieve() < 0) {
      rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: Cannot init recieve functionality\n");
      hal_exit(comp_id);
      return -1;
    }
  }
  else if(action[0] == 's') { // We need to send data
    if(init_send() < 0) {
      rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: Cannot init send functionality\n");
      hal_exit(comp_id);
      return -1;
    }
  }
  else { // No idea what needs to be done, so do nothing and exit
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: action \"%s\" not known\n", action);
    hal_exit(comp_id);
    return -1;
  }

  return 0;
}

void rtapi_app_exit(void)
{
  
  int retval;
  
    
  // Removing the HAL module
  retval = hal_exit(comp_id);
  if(retval != HAL_SUCCESS) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: hal_exit() failed\n");
  }
  
  // Close socket
  if(rt_dev_close(sockfd) == EAGAIN) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: couldn't close socket, \
                    close filedescriptor in /proc/rtai/rtdm/openfildes manually");
  }
 
  rtapi_print_msg(RTAPI_MSG_INFO, "hal_rtnet: Clean-up complete\n");
}

/********************************************************
*             GENERAL FUNCTION IMPLEMENTATIONS          *
********************************************************/

/**
* init_recieve()
* 
* This function initializes sending functionality.
* 
* A socket is created, and we set a timeout.
* 
**/
static int init_recieve(void)
{
  
  rtapi_print_msg(RTAPI_MSG_INFO, "hal_rtnet: initializing recieve functionality\n");
  
  int retval;
  
  // Set structures to 0
  memset(&local_addr, 0, sizeof(struct sockaddr_in));

  // Export HAL functionality
  retval = export_rtnet(nr_of_pins);
  if(retval != 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: export_rtnet() failed\n");
    hal_exit(comp_id);
    return -1;
  }
  // Set address information structure    
  local_addr.sin_family      = AF_INET;
  local_addr.sin_addr.s_addr = INADDR_ANY;
  local_addr.sin_port        = htons(port);
  
  // Create a new socket
  sockfd = rt_dev_socket(AF_INET, SOCK_DGRAM, 0);
  if (sockfd < 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: creating socket failed\n");
  }
  
  // Bind socket to local address specified as parameter.
  retval = rt_dev_bind(sockfd, (struct sockaddr *) &local_addr,
                    sizeof(struct sockaddr_in));
  
  // Set timeout for socket in blocking mode
  retval = rt_dev_ioctl(sockfd, RTNET_RTIOC_TIMEOUT, &timeout);
  if (retval < 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: setting socket option failed with error %d", retval);
  }
  
  return 0;
}

/**
* init_send()
* 
* This function initializes sending functionality.
* 
* A socket is created and we connect to the remote ip address.
* 
**/
static int init_send(void)
{

  rtapi_print_msg(RTAPI_MSG_INFO, "hal_rtnet: initializing send functionality with ip %s\n", ip);

  int retval;

  // Set structures to 0
  memset(&server_addr, 0, sizeof(struct sockaddr_in));
  memset(&local_addr, 0, sizeof(struct sockaddr_in));

  // Export HAL functionality
  retval = export_rtnet(nr_of_pins);
  if(retval != 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: export_rtnet() failed\n");
    hal_exit(comp_id);
    return -1;
  }
  
  // Set address information structures
  local_addr.sin_family      = AF_INET;
  local_addr.sin_addr.s_addr = INADDR_ANY;
  local_addr.sin_port        = htons(port);

  server_addr.sin_family      = AF_INET;
  server_addr.sin_addr.s_addr = rt_inet_aton(ip);
  server_addr.sin_port        = htons(port);
  
  // Create a new socket
  sockfd = rt_dev_socket(AF_INET, SOCK_DGRAM, 0);
  if (sockfd < 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: creating socket failed\n");
  }
  
  // Bind socket to local address specified as parameter
  retval = rt_dev_bind(sockfd, (struct sockaddr *) &local_addr,
                    sizeof(struct sockaddr_in));
  
  // Connect to destination address
  retval = rt_dev_connect(sockfd, (struct sockaddr *) &server_addr,
                          sizeof(struct sockaddr_in));
  if (retval < 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: couldn't connect to %s, error %d\n", ip, retval);
  }
  
  return 0;
}


/**
* export_rtnet()
* 
* This function exports the rtnet HAL functionality.
* 
**/
static int export_rtnet(int nr)
{
  int retval, i;
  char buf[HAL_NAME_LEN + 2];
  
  /* we are sending over a packet with a size of the sum of the data types size plus
     the seperators, which take one byte each (nr_of_pins) */
  packet_size = nr_of_pins;
  
  /* export realtime functions that do the real work */
  if(action[0] == 'r') {
    retval = hal_export_funct("hal_rtnet.recieve", recieve, 0	/* arg 
			     */ , 1 /* uses_fp */ , 0 /* reentrant */ , comp_id);
  }
  else if (action[0] == 's') {
    retval = hal_export_funct("hal_rtnet.send", send, 0	/* arg 
			     */ , 1 /* uses_fp */ , 0 /* reentrant */ , comp_id);  
  }
  else {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: can't find function for action %s\n", action);
    return -1;
  }
  
  if (retval != HAL_SUCCESS) {
    rtapi_print_msg(RTAPI_MSG_ERR,
    "hal_rtnet: failed to export cController function %s\n", action);
    return -1;
  }
  
  /* Allocate shared memory */
  hal_data = hal_malloc(sizeof(hal_rtnet_data) * nr_of_pins);
  if(hal_data == 0) {
    rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: hal_malloc() failed\n");
    return -1;
  }
  
  /* Export pins */
  for(i = 0; i < nr; i++) {
  
    rtapi_snprintf(buf, HAL_NAME_LEN, "hal_rtnet.%d-%s", i, argv[i]);
    
    if((strcmp(argv[i], "bit") == 0) && !(packet_size + sizeof(hal_bit_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_bit_new(buf, HAL_RD_WR, &(hal_data[i].hbit), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hbit) = 0;
      packet_size += sizeof(hal_bit_t);
    }
    else if((strcmp(argv[i], "u8") == 0) && !(packet_size + sizeof(hal_u8_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_u8_new(buf, HAL_RD_WR, &(hal_data[i].hu8), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hu8) = 0;
      packet_size += sizeof(hal_u8_t);
    }
    else if((strcmp(argv[i], "s8") == 0) && !(packet_size + sizeof(hal_s8_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_s8_new(buf, HAL_RD_WR, &(hal_data[i].hs8), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hs8) = 0;
      packet_size += sizeof(hal_s8_t);
    }
    else if((strcmp(argv[i], "u16") == 0) && !(packet_size + sizeof(hal_u16_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_u16_new(buf, HAL_RD_WR, &(hal_data[i].hu16), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hu16) = 0;
      packet_size += sizeof(hal_u16_t);
    }
    else if((strcmp(argv[i], "s16") == 0) && !(packet_size + sizeof(hal_s16_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_s16_new(buf, HAL_RD_WR, &(hal_data[i].hs16), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hs16) = 0;
      packet_size += sizeof(hal_s16_t);
    }
    else if((strcmp(argv[i], "u32") == 0) && !(packet_size + sizeof(hal_u32_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_u32_new(buf, HAL_RD_WR, &(hal_data[i].hu32), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hu32) = 0;
      packet_size += sizeof(hal_u32_t);
    }
    else if((strcmp(argv[i], "s32") == 0) && !(packet_size + sizeof(hal_s32_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_s32_new(buf, HAL_RD_WR, &(hal_data[i].hs32), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hs32) = 0;
      packet_size += sizeof(hal_s32_t);
    }
    else if((strcmp(argv[i], "float") == 0) && !(packet_size + sizeof(hal_float_t) > HAL_RTNET_MAX_PACKET_SIZE)) {
      retval = hal_pin_float_new(buf, HAL_RD_WR, &(hal_data[i].hfloat), comp_id);
      if(retval != HAL_SUCCESS) {
        rtapi_print_msg(RTAPI_MSG_ERR, "Couldn't create pin: %d\n", retval);
        return retval;
      }
      *(hal_data[i].hfloat) = 0.0;
      packet_size += sizeof(hal_float_t);
    }
    else
      return -1;

  }
  
  if (fp_period > 0) {
    /* create a floating point thread */
    retval = hal_create_thread("hal_rtnet.thread", fp_period, 1);
    if (retval < 0) {
      rtapi_print_msg(RTAPI_MSG_ERR, "hal_rtnet: ERROR: could not create FP thread\n");
      hal_exit(comp_id);
      return -1;
    } else {
        rtapi_print_msg(RTAPI_MSG_INFO, "hal_rtnet: created %d uS FP thread\n", fp_period / 1000);
    }
  }
  
  return 0;
}

/**
* parse_pin_cfg()
* 
* This function parses the pin config parameter string and puts the result
* into the global argv string array.
* 
**/
static int parse_pin_cfg(char *cfg)
{

  // Do we have a valid amount of pins?
  if(nr_of_pins > HAL_RTNET_MAX_PIN)
    return -1; // No, return error
  
  int n;
  char *cp;
  cp = cfg;
  for (n = 0; n < nr_of_pins; n++) {
    /* strip leading whitespace */
    while ((*cp != '\0') && ( isspace(*cp) || ( *cp == '_') ))
      cp++;
    
    /* mark beginning of token */
    argv[n] = cp;
    /* find end of token */
    while ((*cp != '\0') && !( isspace(*cp) || ( *cp == '_') ))
      cp++;

    /* mark end of this token, prepare to search for next one */
    if (*cp != '\0') {
      *cp = '\0';
      cp++;
    }
  }
  for (n = 0; n < nr_of_pins; n++) {
    /* is token empty? */
    if (argv[n][0] == '\0') {
      return -1;
    }
  }
  return 0;
}

/********************************************************
*               HAL FUNCTION IMPLEMENTATIONS            *
********************************************************/

static void recieve(void *arg, long period)
{
  //Uncomment the following line to test wether we are operating from a rt context
  //rtapi_print_msg(RTAPI_MSG_INFO, "RECIEVE: rtdm_in_rt_context = %d\n", rtdm_in_rt_context());

  char packet[packet_size];
  int position = 0, retval, i;
  
  /* Block until packet is received. */
  retval = rt_dev_recv(sockfd, &packet, sizeof(packet), 0 /* or for non-blocking, MSG_DONTWAIT*/);

  if(retval < 0) {
    rtapi_print_msg(RTAPI_MSG_INFO, "error number: %d\n", retval);
    return;
  }
  
  for(i = 0; i < nr_of_pins; i++) {
    if(strcmp(argv[i], "bit") == 0) {
      *(hal_data[i].hbit) = *(hal_bit_t *)&packet[position];
      position += sizeof(hal_bit_t) + 1;
    }
    else if(strcmp(argv[i], "u8") == 0) {
      *(hal_data[i].hu8) = *(hal_u8_t *)&packet[position];
      position += sizeof(hal_u8_t) + 1;
    }
    else if(strcmp(argv[i], "s8") == 0) {
      *(hal_data[i].hs8) = *(hal_s8_t *)&packet[position];
      position += sizeof(hal_s8_t) + 1;
    }
    else if(strcmp(argv[i], "u16") == 0) {
      *(hal_data[i].hu16) = *(hal_u16_t *)&packet[position];
      position += sizeof(hal_u16_t) + 1;
    }
    else if(strcmp(argv[i], "s16") == 0) {
      *(hal_data[i].hs16) = *(hal_s16_t *)&packet[position];
      position += sizeof(hal_s16_t) + 1;
    }
    else if(strcmp(argv[i], "u32") == 0) {
      *(hal_data[i].hu32) = *(hal_u32_t *)&packet[position];
      position += sizeof(hal_u32_t) + 1;
    }
    else if(strcmp(argv[i], "s32") == 0) {
      *(hal_data[i].hs32) = *(hal_s32_t *)&packet[position];
      position += sizeof(hal_s32_t) + 1;
    }
    else if(strcmp(argv[i], "float") == 0) {
      *(hal_data[i].hfloat) = *(hal_float_t *)&packet[position];
      position += sizeof(hal_float_t) + 1;
    }
  }

}

static void send(void *arg, long period)
{
  //Uncomment the following line to test wether we are operating from a rt context
  //rtapi_print_msg(RTAPI_MSG_INFO, "RECIEVE: rtdm_in_rt_context = %d\n", rtdm_in_rt_context());
  
  int retval, i, n;
  int position = 0;

  char packet[packet_size];
  
  for(i = 0; i < nr_of_pins; i++) {
    if(strcmp(argv[i], "bit") == 0) {
      packet[position] = *(unsigned char *)hal_data[i].hbit;
      position++;
    }
    else if(strcmp(argv[i], "u8") == 0) {
      packet[position] = *(unsigned char *)hal_data[i].hu8;
      position++;
    }
    else if(strcmp(argv[i], "s8") == 0) {
      packet[position] = *(unsigned char *)hal_data[i].hs8;
      position++;
    }
    else if(strcmp(argv[i], "u16") == 0) {
      u16_to_char tc;
      tc.hal_u16 = *(hal_data[i].hu16);
      for(n = 0; n < sizeof(hal_u16_t); n++) {
        packet[position] = tc.c[n];
        position++;
      }
    }
    else if(strcmp(argv[i], "s16") == 0) {
      s16_to_char tc;
      tc.hal_s16 = *(hal_data[i].hs16);
      for(n = 0; n < sizeof(hal_s16_t); n++) {
        packet[position] = tc.c[n];
        position++;
      }
    }
    else if(strcmp(argv[i], "u32") == 0) {
      u32_to_char tc;
      tc.hal_u32 = *(hal_data[i].hu32);
      for(n = 0; n < sizeof(hal_u32_t); n++) {
        packet[position] = tc.c[n];
        position++;
      }
    }
    else if(strcmp(argv[i], "s32") == 0) {
      s32_to_char tc;
      tc.hal_s32 = *(hal_data[i].hs32);
      for(n = 0; n < sizeof(hal_s32_t); n++) {
        packet[position] = tc.c[n];
        position++;
      }
    }
    else if(strcmp(argv[i], "float") == 0) {
      float_to_char tc;
      tc.hal_float = *(hal_data[i].hfloat);
      for(n = 0; n < sizeof(hal_float_t); n++) {
        packet[position] = tc.c[n];
        position++;
      }
    }
    packet[position] = '\0';
    position++;
  }
  
  /* Send the message. */
  retval = rt_dev_send(sockfd, &packet, sizeof(packet), 0);

  if(retval < 0)
    rtapi_print_msg(RTAPI_MSG_INFO, "error number: %d\n", retval);
}

