;;; clp.el --- Create a digest of a CVS log.
;;;
;;; Copyright 2005 by Yoni Rabkin under the GNU GPL (read below).
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
;;; 02110-1301 USA
;;;
;;; Commentary:
;;;
;;; This software creates a digest of a CVS log.
;;;
;;; Use: 
;;;
;;; Evaluate (clp-directory "/path/to/cvs/directory/"). Note that it
;;; is very important to add the trailing slash, and of course that
;;; the directory will be a CVS directory.
;;;
;;; Bugs:
;;;
;;; * At the time of writing clp.el does not work on Emacs 21.
;;;
;;; * Runs very slowly. The code is not optimised in any way.
;;;
;;; History:
;;;
;;; In the beginning there was nothing. A void without form or
;;; shape...

;;; Code:
(defvar clp-output-buffer
  "*clp-output*"
  "Output buffer.")

(defvar clp-head-time nil
  "Time stamp for cvs head.")

(defvar clp-scan-message t
  "If non-nil, output progress message.")

(defvar clp-pre-buffer 
  "*clp-pre*"
  "Buffer to store intermediate CVS log.")

;;; --------------------------------------
;;; 
;;; --------------------------------------

;; example use: (clp-directory "/CVS/DIRECTORY/FOR/EMMS/")
(defun clp-directory (pathstr)
  (let ((default-directory pathstr))
    (message "Downloading CVS log for %s..." default-directory)
    ;; make sure no old buffers are hanging around
    (when (get-buffer clp-pre-buffer)
      (kill-buffer clp-pre-buffer))
    (call-process "cvs" nil clp-pre-buffer nil "log")
    (with-current-buffer clp-pre-buffer
      (clp-go))
    ;; clean up after us
    (when (get-buffer clp-pre-buffer)
      (kill-buffer clp-pre-buffer))
    (switch-to-buffer clp-output-buffer)
    (goto-char (point-min))))

;;; --------------------------------------
;;; Prerequisites
;;; --------------------------------------

(unless (fboundp 'line-number-at-pos)
  (error "Required function `line-number-at-pos' was not found"))

;;; --------------------------------------
;;; Looking at text
;;; --------------------------------------

(defmacro clp-make-looker (name regexp)
  "Return a looker function NAME.
Argument REGEXP should be a regular expression."
  `(defun ,name (n)
     ,(format "Return t if line N begins with the regexp \"%s\"."
	      regexp)
     (save-excursion
       (goto-line n)
       (goto-char (point-at-bol))
       (looking-at ,regexp))))

(defun clp-eobp (n)
  "Return t if point is on the last line of the buffer.
Argument N should be a line number."
  (>= (- n 1) (count-lines (point-min)
			   (point-max))))

(clp-make-looker clp-section-startp "^RCS file:.*,v$")
(clp-make-looker clp-section-stopp "^=+$")
(clp-make-looker clp-desc-startp "^description:$")

;;; --------------------------------------
;;; Slurp description
;;; --------------------------------------

(defun clp-collect-desc (n)
  "Collect lines of text which consist of the description.
If the line N is the beggining of the comment description
section of a CVS log, return a list containing the elements of
that description."
  (if (clp-desc-startp n)
      (let ((clp-desc nil))
	(save-excursion
	  (while (not (clp-section-stopp n))
	    (goto-line n)
	    (setq clp-desc
		  (append clp-desc
			  (list (buffer-substring-no-properties (point-at-bol)
								(point-at-eol)))))
	    (setq n (1+ n))))
	clp-desc)
    (error "Clp-collect-desc called on a line with no CVS description")))

;;; ------------------------------------
;;; Parsing dates from description
;;; ------------------------------------

(defun clp-date-from-desc (clp-desc-str)
  "Return the date string of the head revision.
Argument CLP-DESC-STR should be a string."
  (if (and clp-desc-str
	   (> (length clp-desc-str) 3))
      (with-temp-buffer
	(insert (nth 3 clp-desc-str))
	(goto-char (point-min))
	(re-search-forward "date:.\\([0-9].*[0-9]\\);"
			   (point-at-eol) t)
	(match-string-no-properties 1))
    (error "No CVS description available")))

(defun clp-cvs-date-to-time (date-str)
  "Convert the CVS time stamp to GNU/Emacs internal time.
Argument DATE-STR should be a string."
  (date-to-time (progn
		  (store-substring date-str 4 "-")
		  (store-substring date-str 7 "-"))))

(defun clp-days-since (timeval)
  "Number of days since TIMEVAL."
  (truncate (/ (/ (/ (float-time
		      (time-subtract (current-time)
				     timeval))
		     60)
		  60)
	       24)))

(defun clp-date-and-since (cdstr)
  "Return a cons cell with date information.
More specifically,
return a list whos cons is time head was commited and the cdr is
the number of days since the commit.
Argument CDSTR should be a string."
  (let ((headtime (clp-head-time
		   (clp-date-from-desc cdstr))))
    (cons (format-time-string "%Y-%m-%dT%T%z"
			      headtime)
	  (clp-days-since headtime))))

(defun clp-head-time (date-from-desc-str)
  "Update the variable `clp-head-time' variable.
Argument DATE-FROM-DESC-STR should be a string."
  (if date-from-desc-str
      (setq clp-head-time
	    (clp-cvs-date-to-time date-from-desc-str))
    (error "Failed to set head time")))

;; Example (clp-date-and-since-pretty (clp-collect-desc CVS_LINE_NUM))
(defun clp-date-and-since-pretty (cdstr)
  "Return a sentence describing how long ago head was commited.
Argument CDSTR should be a string."
  (let* ((r (clp-date-and-since cdstr))
	 (timestr (car r))
	 (ago (cdr r)))
    (cond ((> ago 365)
	   ;; FIXME: improve with modulo
	   (setq ago "over a year ago"))
	  ((and (< 1 ago)
		(< ago 365))
	   (setq ago (format "%s days ago" ago)))
	  ((= 1 ago)
	   (setq ago "yesterday"))
	  ((< ago 1)
	   (setq ago "today")))
    (cons timestr (format "%s, %s" timestr ago))))

;;; --------------------------------------
;;; Author from description
;;; --------------------------------------

;; (clp-author-from-desc (clp-collect-desc CVS_LINE_NUM))
(defun clp-author-from-desc (clp-desc-str)
  "Return the author string of the head revision.
Argument CLP-DESC-STR should be a string."
  (if (and clp-desc-str
	   (> (length clp-desc-str) 3))
      (with-temp-buffer
	(insert (nth 3 clp-desc-str))
	(goto-char (point-min))
	(re-search-forward "author:.\\(.*?\\);"
			   (point-at-eol) t)
	(match-string-no-properties 1))
    (error "No CVS description available")))

;;; --------------------------------------
;;; Commit comment from description
;;; --------------------------------------

(defun clp-comment-from-desc (clp-desc-str)
  "Extract the comment part from the description string.
Argument CLP-DESC-STR should be a string."
  (clp-comment-inter (nthcdr 4 clp-desc-str) nil))

;; FIXME: Crash when (number of comment lines) > (max-lisp-eval-depth)
(defun clp-comment-inter (lst accum)
  "Accumulate the comments within the text.
Argument LST should be a list.
Argument ACCUM should be either nil or a list."
  (if (or (null lst)
	  (string-match "----------------------------"
			(car lst)))
      accum
    (clp-comment-inter (cdr lst)
		       (append accum (list (car lst))))))

;;; --------------------------------------
;;; Scanning is fun and nutritious
;;; --------------------------------------

;; Evil idea: store the function for displaying the data in the CAR of
;; the cons cell.
(defun clp-cvs-log ()
  "Return an alist containing a parsed CVS log."
  (let ((out nil)
	(desc nil)
	(lnap 0))
    (save-excursion
      (goto-char (point-min))
      (when (clp-eobp (line-number-at-pos))
	(error "Empty CVS log"))
      (while (not (clp-eobp (line-number-at-pos)))
	(forward-line 1)
	(setq lnap (line-number-at-pos))
	;; Slows everything by 0.6 factor
	(when clp-scan-message
	  (message "Scanning line %s of %s"
		   lnap
		   (count-lines (point-min)
				(point-max))))
	(cond ((clp-section-startp lnap)
	       (forward-line 1)
	       (setq out
		     (append out
			     (list (cons 'next 'start))))
	       (setq out
		     (append out
			     (list
			      (cons 'filename
				    (buffer-substring-no-properties
				     (+ (point-at-bol) 14)
				     (point-at-eol)))))))
	      ((clp-desc-startp lnap)
	       (setq desc
		     (clp-collect-desc lnap))
	       (setq out
		     (append out
			     (list (cons 'date
					 (clp-date-and-since-pretty desc)))
			     (list (cons 'author
					 (clp-author-from-desc desc)))
			     (list (cons 'comment
					 (clp-comment-from-desc desc))))))
	      ((clp-section-stopp lnap)
	       (setq out
		     (append out
			     (list (cons 'next 'stop))))))))
    out))

;;; --------------------------------------
;;; Preparing for post-processing
;;; --------------------------------------

(defun clp-sublist (accum lst)
  "Starting with ACCUM, return asublist of LST."
  (cond ((null lst) accum)
	((and (consp (car lst))
	      (eq (caar lst) 'next)
	      (eq (cdar lst) 'stop))
	 accum)
	(t (clp-sublist (append accum
				(list (car lst)))
			(cdr lst)))))

;; recusively called (/ (length lst) 2) times
(defun clp-encap (accum lst)
  "Surround the relevant sections of data in sublists.
Argument ACCUM should be either nil or a list.
Argument LST should be a list."
  (cond ((null lst) accum)
	((and (consp (car lst))
	      (eq (caar lst) 'next)
	      (eq (cdar lst) 'start))
	 (clp-encap (append accum
			    (list (cdr (clp-sublist nil lst))))
		    (member (cons 'next 'stop)
			    lst)))
	(t (clp-encap accum
		      (cdr lst)))))

(defun clp-encap-call (accum lst)
  (let ((max-lisp-eval-depth (length lst)))
    (clp-encap accum lst)))

;;; --------------------------------------
;;; Sorting
;;; --------------------------------------

(defun clp-sort-p (a b)
  "Sorting predicate.
Argument A should be a date cell.
Argument B should be a date cell."
  (time-less-p (date-to-time (cadadr b))
	       (date-to-time (cadadr a))))

(defun clp-sort (lst)
  "Sort LST according to `clp-sort-p'."
  (sort lst 'clp-sort-p))

;;; --------------------------------------
;;; Human readable output
;;; --------------------------------------

(defun clp-cell-author (cell)
  "Simple accessor function.
Argument CELL should be a cell."
  (cdr (assoc 'author cell)))
(defun clp-cell-filename (cell)
  "Simple accessor function.
Argument CELL should be a cell."
  (cdr (assoc 'filename cell)))
(defun clp-cell-hdate (cell)
  "Simple accessor function.
Argument CELL should be a cell."
  (cddr (assoc 'date cell)))
(defun clp-cell-comment-l (cell)
  "Simple accessor function.
Argument CELL should be a cell."
  (cdr (assoc 'comment cell)))

;; Use example: (clp-cvs-log-output (clp-sort (clp-encap nil
;; (clp-cvs-log))))
;;
;; FIXME: A better way to do this is with C-style
;; format strings.
(defun clp-cvs-log-output (lst)
  "Output a human readable buffer from LST."
  (when (get-buffer clp-output-buffer)
    (kill-buffer clp-output-buffer))
  (get-buffer-create clp-output-buffer)
  (with-current-buffer clp-output-buffer
    (mapc (lambda (e)
	    (unless (eq (car e) 'next)
	      (insert (format "F: %s, commited %s by %s"
			      (clp-cell-filename e)
			      (clp-cell-hdate e)
			      (clp-cell-author e)))
	      (newline)
	      (mapc (lambda (l)
		      (insert (format "C: %s" l))
		      (newline))
		    (clp-cell-comment-l e))
	      (newline)))
	  lst)))

(defun clp-go ()
  "Parse a CVS log into a digest."
  (interactive)
  ;; profiling code
  ;;  (setq start (current-time))
  (clp-cvs-log-output
   (clp-sort
    (clp-encap-call nil
		    (clp-cvs-log))))
  (message "Done. Check the %s buffer for results."
	   clp-output-buffer)
  ;; profiling code
  ;;  (setq end (current-time))
  ;;  (setq totalt (- (float-time end) 
  ;;		  (float-time start)))
  (with-current-buffer clp-output-buffer
    (goto-char (point-min))))

(provide 'clp)

;;; clp.el ends here
