package dbcodegen;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.Properties;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.exception.ParseErrorException;
import org.apache.velocity.exception.ResourceNotFoundException;

import util.FileUtils;
import dbcodegen.db.Database;
import dbcodegen.db.Table;

/**
 * This is the entry class for generating the java persistence model based on a
 * database schema.  It uses the Empire DB open-source framework to build a 
 * java persistence layer for an application.  The Apache Velocity template 
 * engine is used to create the output interfaces and classes.  
 * 
 * The Empire DB framework doesn't try to hide the underlying database and data 
 * model but instead embraces its power by modeling it within java.  The result 
 * is a persistence layer that uses a more "object-oriented, type safe" SQL to
 * access persistent data. 
 * 
 * NOTE:  THIS VERSION HAS SEVERE RESTRICTIONS:
 * 1. Only tables are currently modeled (we'll add views to a later version).
 * 2. Table indexes are not yet modeled (exception is primary key).  Again, this
 * 		will be added to later editions.
 * 3. It is assumed that each table has a single INTEGER auto-generated primary
 * 		key column that has the same name for all tables.
 * 4. It is assumed that each table has a single TIMESTAMP optimistic locking
 * 		column that has the same name for all tables.
 */
public class DbCodeGenerator {
	private Database db;
	private String basePackageName;
	private File baseDir;
	private File tableDir;
	private File recordDir;
	
	/**
	 * Constructs an instance of the database code generator.
	 * @param db The database.
	 * @param srcLocation The location of the generated source code.
	 * @param packageName The package name for the generated persistence layer.
	 */
	public DbCodeGenerator(Database db) {
		this.db = db;
		try {
			Velocity.init();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Generates the source code for the persistence layer.
	 */
	public void generateCode(String srcLocation, String packageName) {
		this.basePackageName = packageName;
		
		// Prepare directories for generated source files
		this.initDirectories(srcLocation, packageName);
		
		// Create the DB class
		this.createDatabaseClass();
	
		// Create base table class
		this.createBaseTableClass();

		// Create base record class
		this.createBaseRecordClass();
		
		// Create table classes, record interfaces and record classes
		for (Table table: this.db.getTables()) {
			this.createTableClass(table);
			this.createRecordClass(table);
		}
	}
	
	private void initDirectories(String srcLocation, String packageName) {
		// Create the directory structure for the generated source code.
		File baseDir = new File(srcLocation);
		if (!baseDir.exists()) {
			baseDir.mkdirs();
		}		
		StringBuilder sb = new StringBuilder();
		sb.append(srcLocation).append("/");
		sb.append(packageName.replaceAll("\\.", "/"));
		this.baseDir = new File(sb.toString());
		if (!this.baseDir.exists()) {
			this.baseDir.mkdirs();
		}
		
		// Clean out the directory so old code is wiped out.
		FileUtils.cleanDirectory(this.baseDir);
		
		// Create the table package directory
		this.tableDir = new File(this.baseDir, "tables");
		this.tableDir.mkdir();

		// Create the record package directory
		this.recordDir = new File(this.baseDir, "records");
		this.recordDir.mkdir();
	}
	
	private void createDatabaseClass() {
		File file = new File(this.baseDir, this.db.getClassName() + 
				".java");
		VelocityContext context = new VelocityContext();
		context.put("basePackageName", this.basePackageName);		
		context.put("tableSubPackage", "tables");
		context.put("database", this.db);
		this.writeFile(file, DATABASE_TEMPLATE, context);
	}
	
	private void createBaseTableClass() {
		File file = new File(this.tableDir, 
				this.db.getBaseTableClassName()	+ ".java");
		VelocityContext context = new VelocityContext();
		context.put("tablePackageName", this.basePackageName + ".tables");		
		context.put("db", this.db);
		this.writeFile(file, BASE_TABLE_TEMPLATE, context);
	}
	
	private void createTableClass(Table table) {
		File file = new File(this.tableDir, 
				table.getClassName() + ".java");
		VelocityContext context = new VelocityContext();
		context.put("tablePackageName", this.basePackageName + ".tables");		
		context.put("db", this.db);
		context.put("table", table);
		this.writeFile(file, TABLE_TEMPLATE, context);
	}
	
	private void createBaseRecordClass() {
		File file = new File(this.recordDir, 
				"BaseRecord.java");
		VelocityContext context = new VelocityContext();
		context.put("basePackageName", this.basePackageName);		
		context.put("tablePackageName", this.basePackageName + ".tables");		
		context.put("recordPackage", this.basePackageName + ".records");		
		context.put("database", this.db);
		this.writeFile(file, BASE_RECORD_TEMPLATE, context);
	}
	
	private void createRecordClass(Table table) {
		File file = new File(this.recordDir, 
				table.getRecordClassName() + ".java");
		VelocityContext context = new VelocityContext();
		context.put("basePackageName", this.basePackageName);		
		context.put("tablePackageName", this.basePackageName + ".tables");		
		context.put("recordPackage", this.basePackageName + ".records");		
		context.put("database", this.db);
		context.put("table", table);		
		this.writeFile(file, RECORD_TEMPLATE, context);
	}

	private void writeFile(File file, String templatePath, 
			VelocityContext context) {
		try {
			Template template = Velocity.getTemplate(templatePath);
			Writer writer = new FileWriter(file);
			template.merge(context, writer);
			writer.close();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ResourceNotFoundException e) {
			e.printStackTrace();
		} catch (ParseErrorException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		if (args.length != 1) {
			System.out.println("Usage: java DbCodeGenerator <propertyFile>");
			System.exit(-1);
		}
	    Properties properties = new Properties();
	    try {
	        properties.load(new FileInputStream(args[0]));
	    } catch (IOException e) {
			e.printStackTrace();
			System.exit(-1);
	    }
	    String jdbcDriver = properties.getProperty("db-driver");
	    String dbUrl = properties.getProperty("db-url");
	    String dbUser = properties.getProperty("db-user");
	    String dbPassword = properties.getProperty("db-password");
	    String dbSchema = properties.getProperty("db-schema");
	    String dbPkCol = properties.getProperty("db-pk-col-name");
	    String dbLockingCol = properties.getProperty("db-locking-col-name");
	    String codeLocation = properties.getProperty("code-location");
	    String codePackage = properties.getProperty("code-package");
	    
		Database db = new Database(jdbcDriver, dbUrl, dbUser, dbPassword,
				dbSchema);
		db.populateTableMetaData(dbPkCol, dbLockingCol);
		DbCodeGenerator generator = new DbCodeGenerator(db);
		generator.generateCode(codeLocation, codePackage);
	}
	
	public static final String DATABASE_TEMPLATE =
		"resources/templates/Database.vm";
	public static final String BASE_TABLE_TEMPLATE =
		"resources/templates/BaseTable.vm";
	public static final String TABLE_TEMPLATE =
		"resources/templates/Table.vm";
	public static final String BASE_RECORD_TEMPLATE =
		"resources/templates/BaseRecord.vm";
	public static final String RECORD_TEMPLATE =
		"resources/templates/Record.vm";
}
