
/* This is simple support of freedesktop.org menu */
#include "e.h"
#include "e_fd_menu.h"
#include "EXML.h"
#include <fcntl.h>

#define MENU_FILE "/etc/xdg/menus-alt/applications.menu"
#define APP_DIRS "/usr/share/applications-alt/"
#define DESKTOP_DIRS "/usr/share/desktop-directories-alt/"
//#define XML_TAG_TYPE(name, id) { name, id },
//

#undef DEBUG
#ifdef DEBUG
#define DBG fprintf
#else
#define DBG
#endif

E_FD_Menu *global_fd_menu = NULL;
Ecore_Timer *free_timer = NULL;


/* F***d cache */
/* Key -> filename */
Evas_Hash *desktop_entry = NULL;
/* key -> category name */
/* value -> list of E_D_Entry values */
Evas_Hash *category_list = NULL;


static E_FD_Menu* e_freedesktop_menu_parse(EXML* xml, E_FD_Menu* parent);

static E_FD_Menu* _e_fd_tag_name(EXML* xml, E_FD_Menu* menu)
{
	if (exml_value_get(xml))
		menu->name = strdup(exml_value_get(xml));

	return menu;
}

static E_FD_Menu* _e_fd_tag_directory(EXML* xml, E_FD_Menu* menu)
{
	if (exml_value_get(xml))
		menu->directory = strdup(exml_value_get(xml));

	return menu;
}


static E_FD_Rules* _e_fd_tag_rules(EXML* xml, E_FD_Rules* rule);

static E_FD_Rules* _e_fd_tag_rules_filename(EXML* xml, E_FD_Rules* rule)
{
	rule->type = E_FD_RULE_FILE;
	if (exml_value_get(xml))
		rule->data = evas_list_append(rule->data, strdup(exml_value_get(xml)));

	DBG(stderr,"filename = %s\n", exml_value_get(xml));

	return rule;
}

static E_FD_Rules* _e_fd_tag_rules_category(EXML* xml, E_FD_Rules* rule)
{
	rule->type = E_FD_RULE_CATEGORY;
	if (exml_value_get(xml))
		rule->data = evas_list_append(rule->data, strdup(exml_value_get(xml)));

	DBG(stderr,"category = %s\n", exml_value_get(xml));

	return rule;
}

static E_FD_Rules* _e_fd_tag_rules_and(EXML* xml, E_FD_Rules* rule)
{
	E_FD_Rules* r;
	r = _e_fd_tag_rules(xml, NULL);
	r->type = E_FD_RULE_AND;
	rule->children = evas_list_append(rule->children, r);

	return rule;
}

static E_FD_Rules* _e_fd_tag_rules_or(EXML* xml, E_FD_Rules* rule)
{
	E_FD_Rules* r;
	r = _e_fd_tag_rules(xml, NULL);
	r->type = E_FD_RULE_OR;
	rule->children = evas_list_append(rule->children, r);

	return rule;
}

static E_FD_Rules* _e_fd_tag_rules_not(EXML* xml, E_FD_Rules* rule)
{
	E_FD_Rules* r;
	r = _e_fd_tag_rules(xml, NULL);
	r->type = E_FD_RULE_NOT;
	rule->children = evas_list_append(rule->children, r);

	return rule;
}

E_FD_XML_Tag inc_tags[] = 
{
	{ "filename", E_TAG_CB(_e_fd_tag_rules_filename) },
	{ "category", E_TAG_CB(_e_fd_tag_rules_category) },
	{ "and", E_TAG_CB(_e_fd_tag_rules_and) },
	{ "or", E_TAG_CB(_e_fd_tag_rules_or) },
	{ "not", E_TAG_CB(_e_fd_tag_rules_not) }
};

static E_FD_Rules* _e_fd_tag_rules(EXML* xml, E_FD_Rules* rule)
{
	char *tag;

	if (!rule)
		rule = calloc(1, sizeof(E_FD_Rules));

	tag = exml_down(xml);
	if(tag)
		do
		{
			int i;
			for (i = 0; i < sizeof(inc_tags)/sizeof(inc_tags[0]);
					i++)
			{
				if (!strcasecmp(tag, inc_tags[i].tag))
					inc_tags[i].call(xml, rule);
			}
		}
		while ((tag = exml_next_nomove(xml)) != NULL);

	exml_up(xml);

	return rule;
}

static E_FD_Menu* _e_fd_tag_include(EXML* xml, E_FD_Menu* menu)
{
	E_FD_Rules* rule;

	rule = _e_fd_tag_rules(xml, NULL);

	rule->type = rule->type | E_FD_RULE_INCLUDE;
	if(rule)
		menu->rules = evas_list_append(menu->rules, rule);

	return menu;
}

static E_FD_Menu* _e_fd_tag_exclude(EXML* xml, E_FD_Menu* menu)
{
	E_FD_Rules* rule;

	rule = _e_fd_tag_rules(xml, NULL);

	rule->type = rule->type | E_FD_RULE_EXCLUDE;
	if(rule)
		menu->rules = evas_list_append(menu->rules, rule);

	return menu;
}

E_FD_XML_Tag tags[] = 
{
	{ "name", E_TAG_CB(_e_fd_tag_name) },
	{ "directory", E_TAG_CB(_e_fd_tag_directory) },
	{ "include", E_TAG_CB(_e_fd_tag_include) },
	{ "include", E_TAG_CB(_e_fd_tag_exclude) },
	{ "menu", E_TAG_CB(e_freedesktop_menu_parse) }
};

/* Parses one Menu entry */
static E_FD_Menu* e_freedesktop_menu_parse(EXML* xml, E_FD_Menu* parent)
{
	E_FD_Menu* menu;
	char *tag;

	if(!xml)
		return NULL;

//	DBG(stderr,"%s(xml = %p, parentname = %s)\n", __FUNCTION__,
//			xml, parent?parent->name:NULL);
	//if (strcasecmp(exml_down(xml), "menu"))
	//	return NULL;
	menu = calloc(1, sizeof(E_FD_Menu));

	tag = exml_down(xml);
//	DBG(stderr,"exml_down = %s\n", tag);
	if (tag)
		do
		{
			int i;

			DBG(stderr,"Tag = %s\n", tag);
			for (i = 0; i < sizeof(tags)/sizeof(tags[0]); i ++)
			{
				if (!strcasecmp(tag, tags[i].tag))
				{
					tags[i].call(xml, menu);
					break;
				}
			}
		}
		while ((tag = exml_next_nomove(xml)) != NULL);
	exml_up(xml);

	if (parent)
	{
//		DBG(stderr,"append children %s to parent %s\n",
//				menu->name, parent->name);
		parent->submenus = evas_list_append(parent->submenus, menu);
	}

//	DBG(stderr,"menu->childrens = %p\n", menu->childrens);

	/*
	DBG(stderr,"childrens: %d\n", evas_list_count(menu->childrens));
	Evas_List *l;
	l = menu->childrens;
	for(; l; l = evas_list_next(l))
	{
		DBG(stderr,"name = %s\n", ((E_FD_Menu*)l->data)->name);
	}


	DBG(stderr,"menu = %p\n", menu);
	*/
	return menu;
}

static void e_freedesktop_menu_post_deact_cb(void *data __UNUSED__,
		E_Menu* menu)
{
}

static Evas_Hash* e_freedesktop_menu_parse_desktop_file(char* filename)
{
	Evas_Hash* hash = NULL;
	FILE* file;
	char *line;
	int linelen;
	int l;

	file = fopen(filename, "r");

	if (!file)
		return;

	linelen = 1024;
	line = calloc(linelen, sizeof(char));

	l = 0;
	while (fgets(line, linelen, file))
	{
		while (strlen(line) == linelen - 1 &&
				line[strlen(line) - 1] != '\n')
		{
			linelen += 1024;
			line = realloc(line, linelen);
			fgets(&line[strlen(line)], 1024, file);
		}
		while (line[strlen(line) - 1] == '\r' ||
				line[strlen(line) - 1] == '\n')
			line[strlen(line) - 1] = '\0';

		if (!l && strcmp(line, "[Desktop Entry]"))
			break;

		{
			char *key, *value;

			value = strstr(line, "=");
			if (value)
			{
				*value = 0;
				value++;
			}
			key = line;

			if (key && key[0] && value)
			{
				//DBG(stderr,"Add to hash: key = %s, value =%s\n",
						//key, value);
				hash = evas_hash_add(hash, strdup(key), strdup(value));
			}
		}

		l++;
	}

	fclose(file);
	return hash;
}

/**
 * Tests if Evas_Hash *pairs values is included in fd_menu.
 */

static int e_freedesktop_menu_is_included(Evas_Hash* pairs, E_FD_Menu* fd_menu)
{
	char *categories;
	Evas_List *l;

	int rv = 0;

	categories = evas_hash_find(pairs, "Categories");
	DBG(stderr,"Categories = %s\n", categories);
	if (!categories)
		return 0;

	DBG(stderr,"fd_menu->rules count = %d\n",
			evas_list_count(fd_menu->rules));
	for (l = fd_menu->rules; l; l = evas_list_next(l))
	{
		E_FD_Rules *rule;
		rule = evas_list_data(l);

		DBG(stderr,"rule->type = %d\n", rule->type);
		DBG(stderr,"is %d, which is %d\n", rule->type & E_FD_RULE_TYPE_MASK,
				E_FD_RULE_CATEGORY);
		if ( (rule->type & E_FD_RULE_TYPE_MASK) == E_FD_RULE_CATEGORY)
		{
			Evas_List *l2;
			DBG(stderr,"rule->data count = %d\n",
					evas_list_count(rule->data));
			for (l2 = rule->data; l2; l2 = evas_list_next(l2))
			{
				char *data = l2->data;
				char buf[1024];

				snprintf(buf, 1024, "%s;", data);
				DBG(stderr,"categories = %s, data = %s\n",
						categories, buf);
				if (strstr(categories, buf))
					rv = ((rule->type & E_FD_RULE_TAG_MASK)
							== E_FD_RULE_INCLUDE)?1:0;
			}
		}
	}


	return rv;
}

static int _e_freedesktop_categories_cache_update(E_D_Entry* d_entry)
{
	char *categories;
	char *category;
	char *ptr;
	Evas_List *l;

	categories = evas_hash_find(d_entry->values, "Categories");
	if (!categories)
		return 0;

	categories = strdup(categories);

	category = strtok_r(categories, ";", &ptr);
	do
	{
		DBG(stderr,"adding %s to category %s\n", d_entry->filename, 
				category);
		l = evas_hash_find(category_list, category);
		if (!l)
		{
			l = evas_list_append(l, d_entry);
			category_list = evas_hash_add(category_list, category, l);
		}
		else
			l = evas_list_append(l, d_entry);

	}
	while(category = strtok_r(NULL,";", &ptr));

}

static void e_freedesktop_desktop_cache_update(void)
{
	/* list files in dir app */
	char *applications_dir = APP_DIRS;
	char buf[1024];
	char *name;
	Ecore_List *files;
	
	files = ecore_file_ls(applications_dir);
	
	
	while(name = ecore_list_next(files))
	{
		E_D_Entry *d_entry;

		strncpy(buf, APP_DIRS, 1024);
		strncat(buf, name, 1024);

		if (evas_hash_find(desktop_entry, buf))
		{
			/* FIXME: check time stamp */
			DBG(stderr,"%s is already in cache\n", buf);
			continue;
		}

		d_entry = calloc(1, sizeof(E_D_Entry));
		d_entry->dirname = strdup(applications_dir);
		d_entry->filename = strdup(name);
		
		d_entry->values = e_freedesktop_menu_parse_desktop_file(buf);

		DBG(stderr,"%s added to cache\n", buf);
		desktop_entry = evas_hash_add(desktop_entry, buf, d_entry);

		_e_freedesktop_categories_cache_update(d_entry);
		
	}

	if (files) ecore_list_destroy(files);
	
}

static E_D_Item* _e_freedesktop_desktop_item_from_desktop_entry
					(E_D_Entry* d_entry)
{
	E_D_Item* item;
	char lang[36];
	char buf[128];

	if (d_entry->mi)
		return d_entry->mi;

	DBG(stderr,"Ok, some item included\n");
	if (getenv("LANG"))
	{
		strncpy(lang, getenv("LANG"), 36);
		lang[2] = 0;
	}
	else
		lang[0] = 0;

	if(lang[0])
		snprintf(buf, 128, "Name[%s]", lang);
	else
		snprintf(buf, 128, "Name");

	item = calloc(1, sizeof(E_D_Item));
	item->name = evas_hash_find(d_entry->values, buf);

	if (!item->name) 
		item->name = evas_hash_find(d_entry->values, "Name");

	item->icon = evas_hash_find(d_entry->values, "Icon");
	item->app = evas_hash_find(d_entry->values, "Exec");

	item->d_entry = d_entry;
	d_entry->mi = item;

	return d_entry->mi;
}

static Evas_Bool _e_freedesktop_node_is_included(Evas_Hash* node,
		const char *key __UNUSED__, void *data, void *fdata)
{
	E_FD_Menu* fd_menu;
	fd_menu = fdata;
	E_D_Entry *d_entry;
	d_entry = data;
	
	if (e_freedesktop_menu_is_included(d_entry->values, fd_menu))
		fd_menu->items = evas_list_append(fd_menu->items,
				_e_freedesktop_desktop_item_from_desktop_entry(d_entry));

	return 1;
}

static int e_freedesktop_menu_load_desktop(E_FD_Menu* fd_menu)
{
	Evas_List *l;

// old way.. we has category cache! ypiee!
//	evas_hash_foreach(desktop_entry, _e_freedesktop_node_is_included,
//			fd_menu);

	// new wa{y,ve}
	/* lets list fd_menu categories */
	l = fd_menu->rules;
	for(; l; l = evas_list_next(l))
	{
		E_FD_Rules *rule;
		rule = l->data;

		if ( (rule->type & E_FD_RULE_TYPE_MASK) == E_FD_RULE_CATEGORY)
		{
			Evas_List *l2;
			DBG(stderr,"rule->data count = %d\n",
					evas_list_count(rule->data));
			for (l2 = rule->data; l2; l2 = evas_list_next(l2))
			{
				char *data = l2->data;

				Evas_List *d_entries;

				d_entries = evas_hash_find(category_list, data);
				if(!d_entries)
				{
					DBG(stderr,"NOT found in category cache: %s\n", data);
					continue;
				}

				DBG(stderr,"found in category cache: %s\n", data);

				for (; d_entries; d_entries = evas_list_next(d_entries))
				{
					E_D_Entry* d_entry;
					d_entry = evas_list_data(d_entries);
					fd_menu->items = evas_list_append(fd_menu->items,
							_e_freedesktop_desktop_item_from_desktop_entry(
									d_entry));
				}
			}
		}
	}

}

static int e_freedesktop_menu_load_directory(E_FD_Menu* fd_menu)
{
	char buf[1024];
	char lang[36];
	char *val;
	Evas_Hash *entry_pairs;

	if (!fd_menu->directory)
		return 0;

	snprintf(buf, 1024, "%s/%s", DESKTOP_DIRS, fd_menu->directory);

	entry_pairs = e_freedesktop_menu_parse_desktop_file(buf);

	if (getenv("LANG"))
	{
		snprintf(lang, 36, getenv("LANG"));
		lang[2] = 0;
	}
	else
		lang[0] = 0;

	if(lang[0])
		snprintf(buf, 1024, "Name[%s]", lang);
	else
		snprintf(buf, 1024, "Name");

	/* Ok, set name and icon */
	val = evas_hash_find(entry_pairs, buf);
	if (val)
		fd_menu->name = val;

	val = evas_hash_find(entry_pairs, "Icon");
	if (val)
	{
		snprintf(buf, 1024, "/usr/share/icons/hicolor/32x32/apps/%s", val);
		fd_menu->icon = strdup(buf);
	}

	evas_hash_free(entry_pairs);

	return 1;
}

static void e_freedesktop_menu_pre_act_cb(void *data __UNUSED__, E_Menu* menu);

static E_Menu* e_menu_from_freedesktop(E_FD_Menu* fd_menu, E_Menu *parent)
{
	E_Menu *menu;
	E_Menu_Item *mi;

	if (!fd_menu)
		return NULL;

	e_freedesktop_menu_load_directory(fd_menu);
	menu = e_menu_new();

	mi = e_menu_item_new(parent);
	e_menu_item_label_set(mi, fd_menu->name);
	e_menu_item_icon_file_set(mi, fd_menu->icon);
	e_menu_item_submenu_set(mi, menu);

	e_menu_pre_activate_callback_set(menu,
			e_freedesktop_menu_pre_act_cb, fd_menu);
	
	return menu;
}

static void _e_freedesktop_run_app(void *data, E_Menu* mn, E_Menu_Item* mi)
{
	E_D_Item *item;

	item = data;

	DBG(stderr,"Running %s\n", item->app);
	ecore_exe_run(item->app, NULL);
	return;
}

static E_Menu_Item* e_menu_item_from_desktop_item(E_D_Item* item,
		E_Menu *parent)
{
	E_Menu_Item *mi;

	mi = e_menu_item_new(parent);
	e_menu_item_label_set(mi, item->name);
	e_menu_item_icon_file_set(mi, item->icon);

	e_menu_item_callback_set(mi, _e_freedesktop_run_app, item);
	
	return mi;
}

static void e_fd_menu_free(E_FD_Menu* menu)
{
	Evas_List *l;

	if (!menu)
		return;
	l = menu->submenus;
	for(; l; l = evas_list_next(l))
	{
		e_fd_menu_free(evas_list_data(l));
	}


	for( l = menu->app_dirs; l; l = evas_list_next(l))
		free(l->data);
	evas_list_free(menu->app_dirs);

	for( l = menu->dir_dirs; l; l = evas_list_next(l))
		free(l->data);
	evas_list_free(menu->dir_dirs);

	for( l = menu->rules; l; l = evas_list_next(l))
		free(l->data);
	evas_list_free(menu->rules);

	if (menu->directory)
		free(menu->directory);
	if (menu->name)
		free(menu->name);
	if (menu->icon)
		free(menu->icon);

	free(menu);
}

int _global_fd_menu_free(void *data __UNUSED__)
{
	DBG(stderr,"%s\n", __FUNCTION__);
	e_fd_menu_free(global_fd_menu);
	global_fd_menu = NULL;
}

static void e_freedesktop_menu_pre_act_cb(void *data, E_Menu* menu)
{
	/* Try to load MENU_FILE */
	EXML* menu_xml;
	E_FD_Menu *fd_menu;
	E_Menu_Item* mi;

	fd_menu = (E_FD_Menu*)data;

	if (global_fd_menu == NULL && fd_menu == NULL)
	{
	//	DBG(stderr,"Load file: %s\n", MENU_FILE);
		menu_xml = exml_new();
		exml_file_read(menu_xml, MENU_FILE);


		fd_menu = e_freedesktop_menu_parse(menu_xml, NULL);
		/* cache it */
		e_freedesktop_desktop_cache_update();
		//e_freedesktop_menu_load_desktop(fd_menu);
		global_fd_menu = fd_menu;
		free_timer = ecore_timer_add(60, _global_fd_menu_free, NULL);

		exml_destroy(menu_xml);
	}
	else if (global_fd_menu && fd_menu == NULL)
	{
		fd_menu = global_fd_menu;
		if (free_timer)
			ecore_timer_del(free_timer);
		free_timer = ecore_timer_add(60, _global_fd_menu_free, NULL);
	}

	if (fd_menu->submenus)
	{
		Evas_List *l;
		l = fd_menu->submenus;
		for (; l; l = l->next)
		{
			E_FD_Menu* fd_mn;
			fd_mn = l->data;
//			DBG(stderr," adding sub menu: %s\n", fd_mn?fd_mn->name:NULL);
			e_menu_from_freedesktop(fd_mn, menu);
		}
	}

	if (!fd_menu->items_loaded)
	{
		e_freedesktop_menu_load_desktop(fd_menu);
		fd_menu->items_loaded = 1;
	}


	if (fd_menu->items)
	{
		DBG(stderr,"some items!\n");
		Evas_List *l;
		l = fd_menu->items;
		for (; l; l = l->next)
		{
			E_D_Item *item;
			item = l->data;
			DBG(stderr,"adding item = %s\n", item->name);
			e_menu_item_from_desktop_item(item, menu);
		}
	}


	e_menu_pre_activate_callback_set(menu, NULL, NULL);
}

static char* _e_fd_root_menu_name(char* file)
{
	EXML* menu_xml;
	char *name, *tag;

	menu_xml = exml_new();
	exml_file_read(menu_xml, file);

	name = NULL;

	exml_down(menu_xml);
	while (tag = exml_next(menu_xml))
	{
//		DBG(stderr,"tag = %s\n", tag);
		if (!strcasecmp(tag, "name"))
		{
			name = strdup(exml_value_get(menu_xml));
//			DBG(stderr,"set name =%s\n", name);
			break;
		}
	}

	exml_destroy(menu_xml);

	if (!name)
		name = strdup("ALT Linux");
	return name;
}

static void
_e_freedesktop_menu_del_hook(void *obj)
{
	E_Menu *m;
	Evas_List *l;

	m = obj;
	for (l = m->items; l; l = l->next)
	{
		E_Menu_Item *mi;

		mi = l->data;
		if (mi->submenu) e_object_del(E_OBJECT(mi->submenu));
	}
}
	

E_Menu_Item* e_freedesktop_menu_new(E_Menu* rmn, E_Menu** this)
{
	E_Menu *mn;
	E_Menu_Item *mi;

	mn = e_menu_new();
	*this = mn;

	mi = e_menu_item_new(rmn);
	e_menu_item_label_set(mi, _e_fd_root_menu_name(MENU_FILE));
	e_menu_item_submenu_set(mi, mn);
	
	e_menu_pre_activate_callback_set(mn, e_freedesktop_menu_pre_act_cb, NULL);
	e_object_del_attach_func_set(E_OBJECT(mn), _e_freedesktop_menu_del_hook);

	return mi;
}
