// xmlauto.hxx - a more flexible, generic way to build autopilots
//
// Written by Curtis Olson, started January 2004.
//
// Copyright (C) 2004  Curtis L. Olson  - http://www.flightgear.org/~curt
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//
// $Id: xmlauto.hxx,v 1.20 2009/02/28 16:16:14 torsten Exp $


#ifndef _XMLAUTO_HXX
#define _XMLAUTO_HXX 1

#ifndef __cplusplus
# error This library requires C++
#endif

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <simgear/compiler.h>

#include <string>
#include <vector>
#include <deque>

using std::string;
using std::vector;
using std::deque;

#include <simgear/props/props.hxx>
#include <simgear/structure/subsystem_mgr.hxx>
#include <simgear/props/condition.hxx>

#include <Main/fg_props.hxx>


/**
 * Base class for other autopilot components
 */

class FGXMLAutoComponent : public SGReferenced {

private:
    // the process input
    SGPropertyNode_ptr input_prop;
    double y_scale;             // scale process input from property system
    double y_offset;

    // the process reference where appropriate
    SGPropertyNode_ptr r_prop;
    double r_value;
    double r_scale;             // scale reference input from property system
    double r_offset;

    // post functions for output
    bool clamp;
    double u_min;               // Minimum output clamp
    double u_max;               // Maximum output clamp
    SGPropertyNode_ptr umin_prop;
    SGPropertyNode_ptr umax_prop;
    vector <SGPropertyNode_ptr> output_list;

    SGSharedPtr<const SGCondition> _condition;
    SGPropertyNode_ptr enable_prop;
    string * enable_value;

    SGPropertyNode_ptr passive_mode;
    bool honor_passive;

    string name;
protected:

    // debug flag
    bool debug;
    bool enabled;

public:

    FGXMLAutoComponent( SGPropertyNode *node);
    virtual ~FGXMLAutoComponent();

    virtual void update (double dt)=0;
    
    inline const string& get_name() { return name; }

    inline double getInputValue() {
        return input_prop != NULL ? input_prop->getDoubleValue() * y_scale + y_offset : 0.0;
    }

    inline double getReferenceValue() {
        return r_prop != NULL ? r_prop->getDoubleValue() * r_scale + r_offset : r_value;
    }

    inline double Clamp( double value ) {
        if( clamp ) {
            double d = getUMax();
            if( value > d ) value = d;
            d = getUMin();
            if( value < d ) value = d;
        }
        return value;
    }

    // Returns the minimum clamp value
    inline double getUMin() {
      return umin_prop != NULL ? umin_prop->getDoubleValue() : u_min;
    }

    // Returns the maximum clamp value
    inline double getUMax() {
      return umax_prop != NULL ? umax_prop->getDoubleValue() : u_max;
    }

    inline void setOutputValue( double value ) {
        // passive_ignore == true means that we go through all the
        // motions, but drive the outputs.  This is analogous to
        // running the autopilot with the "servos" off.  This is
        // helpful for things like flight directors which position
        // their vbars from the autopilot computations.
        if ( honor_passive && passive_mode->getBoolValue() ) return;
        for ( unsigned i = 0; i < output_list.size(); ++i ) {
            output_list[i]->setDoubleValue( Clamp(value) );
        }
    }

    inline double getOutputValue() {
      return output_list.size() == 0 ? 0.0 : Clamp(output_list[0]->getDoubleValue());
    }

    /* 
       Returns true if the enable-condition is true.

       If a <condition> is defined, this condition is evaluated, 
       <prop> and <value> tags are ignored.

       If a <prop> is defined and no <value> is defined, the property
       named in the <prop></prop> tags is evaluated as boolean.

       If a <prop> is defined a a <value> is defined, the property named
       in <prop></prop> is compared (as a string) to the value defined in
       <value></value>

       Returns true, if neither <condition> nor <prop> exists

       Examples:
       Using a <condition> tag
       <enable>
         <condition>
           <!-- any legal condition goes here and is evaluated -->
         </condition>
         <prop>This is ignored</prop>
         <value>This is also ignored</value>
       </enable>

       Using a single boolean property
       <enable>
         <prop>/some/property/that/is/evaluated/as/boolean</prop>
       </enable>

       Using <prop> == <value>
       This is the old style behaviour
       <enable>
         <prop>/only/true/if/this/equals/true</prop>
         <value>true<value>
       </enable>
    */
    bool isPropertyEnabled();
};


/**
 * Roy Ovesen's PID controller
 */

class FGPIDController : public FGXMLAutoComponent {

private:

    // Input values
    double y_n;                 // measured process value
    double r_n;                 // reference (set point) value

    // Configuration values
    double Kp;                  // proportional gain
    SGPropertyNode_ptr Kp_prop;

    double alpha;               // low pass filter weighing factor (usually 0.1)
    double beta;                // process value weighing factor for
                                // calculating proportional error
                                // (usually 1.0)
    double gamma;               // process value weighing factor for
                                // calculating derivative error
                                // (usually 0.0)

    double Ti;                  // Integrator time (sec)
    SGPropertyNode_ptr Ti_prop;
    double Td;                  // Derivator time (sec)
    SGPropertyNode_ptr Td_prop;

    // Previous state tracking values
    double ep_n_1;              // ep[n-1]  (prop error)
    double edf_n_1;             // edf[n-1] (derivative error)
    double edf_n_2;             // edf[n-2] (derivative error)
    double u_n_1;               // u[n-1]   (output)
    double desiredTs;            // desired sampling interval (sec)
    double elapsedTime;          // elapsed time (sec)
    
    
    
public:

    FGPIDController( SGPropertyNode *node );
    FGPIDController( SGPropertyNode *node, bool old );
    ~FGPIDController() {}

    void update_old( double dt );
    void update( double dt );
};


/**
 * A simplistic P [ + I ] PID controller
 */

class FGPISimpleController : public FGXMLAutoComponent {

private:

    // proportional component data
    bool proportional;
    double Kp;
    SGPropertyNode_ptr Kp_prop;
    SGPropertyNode_ptr Ki_prop;
    SGPropertyNode_ptr offset_prop;
    double offset_value;

    // integral component data
    bool integral;
    double Ki;
    double int_sum;


    // Input values
    double y_n;                 // measured process value
    double r_n;                 // reference (set point) value

public:

    FGPISimpleController( SGPropertyNode *node );
    ~FGPISimpleController() {}

    void update( double dt );
};


/**
 * Predictor - calculates value in x seconds future.
 */

class FGPredictor : public FGXMLAutoComponent {

private:

    // proportional component data
    double last_value;
    double average;
    double seconds;
    double filter_gain;

    // Input values
    double ivalue;                 // input value
    
public:

    FGPredictor( SGPropertyNode *node );
    ~FGPredictor() {}

    void update( double dt );
};


/**
 * FGDigitalFilter - a selection of digital filters
 *
 * Exponential filter
 * Double exponential filter
 * Moving average filter
 * Noise spike filter
 *
 * All these filters are low-pass filters.
 *
 */

class FGDigitalFilter : public FGXMLAutoComponent
{
private:
    double Tf;            // Filter time [s]
    unsigned int samples; // Number of input samples to average
    double rateOfChange;  // The maximum allowable rate of change [1/s]
    double gainFactor;
    SGPropertyNode_ptr gain_prop;

    deque <double> output;
    deque <double> input;
    enum filterTypes { exponential, doubleExponential, movingAverage,
                       noiseSpike, gain, reciprocal };
    filterTypes filterType;

public:
    FGDigitalFilter(SGPropertyNode *node);
    ~FGDigitalFilter() {}

    void update(double dt);
};

/**
 * Model an autopilot system.
 * 
 */

class FGXMLAutopilot : public SGSubsystem
{

public:

    FGXMLAutopilot();
    ~FGXMLAutopilot();

    void init();
    void reinit();
    void bind();
    void unbind();
    void update( double dt );

    bool build();

protected:

    typedef vector<SGSharedPtr<FGXMLAutoComponent> > comp_list;

private:

    bool serviceable;
    SGPropertyNode_ptr config_props;
    comp_list components;
};


#endif // _XMLAUTO_HXX
