/*
 *  Fl_Native_Printer.h
 *
 */
#ifdef __APPLE__
#include <Carbon/Carbon.h>
#elif defined(WI3N2)
#include <Windows.h>
#endif


#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/fl_draw.H>

/**
 * @brief Provides an OS-independent interface to printing under MSWindows and Mac OS X.
 *
 This class provides an OS-independent interface to printing under MSWindows and Mac OS X only.
 It allows to use all FLTK drawing, color, text, and clip functions, and to have them operate
 on printed page(s).
 */
class Fl_Native_Printer {
   int y_offset;
   int x_offset;
#ifdef __APPLE__
   float scale_x;
   float scale_y;
   PMPrintSession  printSession;
   PMPageFormat    pageFormat;
   PMPrintSettings printSettings;
#elif defined(WIN32)
   int   abortPrint;
   PRINTDLG      pd;
   HDC           hPr;
   int           prerr;
#endif
public:
  /** 
   @brief The constructor.
   */
  Fl_Native_Printer(void);
  
  /**
   @brief Starts a print job.
   *
   @param[in] pagecount the total number of pages of the job
   @param[out] frompage if non-null, *frompage is set to the first page the user wants printed
   @param[out] topage if non-null, *topage is set to the last page the user wants printed
   @return 0 iff OK
   */
  int startjob(int pagecount, int *frompage = NULL, int *topage = NULL);
  
  /**
   @brief Starts a new printed page
   *
   The page coordinates are initially in points, i.e., 1/72 inch, 
   and with origin at the top left of the page or of the printable page area.
   @return 0 iff OK
   */
  int startpage (void);
  
  /**
   @brief Computes the origin and size of the printable area of the page.
   *
   Values are in the same unit as that used by FLTK drawing functions,
   are unchanged by calls to setorigin(), but are changed by rescale() calls.
   @return 0 iff OK.
   */
  int getprintablerect(int *x, int *y, int *w, int *h);
  
  /**
   @brief Sets the position in page coordinates of the origin of graphics functions.
  */
  void setorigin(int x, int y);
  
  /**
   @brief Changes the scaling of page coordinates.
   *
   Use next getprintablerect() to get the new coordinates of the printable page area.
   Successive rescale() calls don't cumulate their effects.
   @param scale_x Horizontal dimensions of plot are multiplied by this quantity
   @param scale_y Vertical dimensions of plot are multiplied by this quantity
   */
  void rescale (float scale_x, float scale_y);
  
  /**
   @brief To be called at the end of each page.
   *
   @return 0 iff OK.
   */
  int endpage (void);
  
  /**
   @brief To be called at the end of a print job.
   */
  void endjob (void);
};
