# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------

import string
import os
import struct
import array
import fcntl
import sys
import time
import binascii

import dvb_ts


# Different formats depending on word size
bit32 = struct.calcsize('L') == struct.calcsize('I')
def i32(x):
    return (x&0x80000000L and -2*0x40000000 or 0) + int(x&0x7fffffff)

_IOC_NRBITS   = 8
_IOC_TYPEBITS = 8
_IOC_SIZEBITS = 14
_IOC_DIRBITS  = 2

_IOC_NRMASK   = ((1 << _IOC_NRBITS)   - 1)
_IOC_TYPEMASK = ((1 << _IOC_TYPEBITS) - 1)
_IOC_SIZEMASK = ((1 << _IOC_SIZEBITS) - 1)
_IOC_DIRMASK  = ((1 << _IOC_DIRBITS)  - 1)

_IOC_NRSHIFT   = 0
_IOC_TYPESHIFT = (_IOC_NRSHIFT   + _IOC_NRBITS)
_IOC_SIZESHIFT = (_IOC_TYPESHIFT + _IOC_TYPEBITS)
_IOC_DIRSHIFT  = (_IOC_SIZESHIFT + _IOC_SIZEBITS)

# Direction bits.
_IOC_NONE  = 0
_IOC_WRITE = 1
_IOC_READ  = 2

def _IOC(dir, type, nr, size):
    return (((dir)    << _IOC_DIRSHIFT)  | \
           (ord(type) << _IOC_TYPESHIFT) | \
           ((nr)      << _IOC_NRSHIFT)   | \
           ((size)    << _IOC_SIZESHIFT))

def _IO(type, nr):
    return _IOC(_IOC_NONE, (type), (nr), 0)

def _IOR(type, nr, size):
    return _IOC(_IOC_READ, (type), (nr), struct.calcsize(size))

def _IOW(type, nr, size):
    return _IOC(_IOC_WRITE, (type), (nr), struct.calcsize(size))

def _IOWR(type, nr, size):
    return _IOC (_IOC_READ|_IOC_WRITE, (type), (nr), struct.calcsize(size))

# used to decode ioctl numbers..
def _IOC_DIR(nr):
    return (((nr) >> _IOC_DIRSHIFT) & _IOC_DIRMASK)
def _IOC_TYPE(nr):
    return (((nr) >> _IOC_TYPESHIFT) & _IOC_TYPEMASK)
def _IOC_NR(nr):
    return (((nr) >> _IOC_NRSHIFT) & _IOC_NRMASK)
def _IOC_SIZE(nr):
    return (((nr) >> _IOC_SIZESHIFT) & _IOC_SIZEMASK)


# Following code is derived from the Linux DVB API version 3
# http://www.linuxtv.org/downloads/linux-dvb-api-1.0.0.pdf
# See /usr/include/linux/dvb/frontend.h
FE_TYPE = {
   'FE_UNKNOWN' : -1,
   'FE_QPSK'    : 0,
   'FE_QAM'     : 1,
   'FE_OFDM'    : 2,
   'FE_ATSC'    : 3,
}

# See struct dvb_frontend_info at /usr/include/linux/dvb/frontend.h
FRONTEND_INFO_ST = "128s10L"
FRONTEND_INFO_NO = _IOR('o', 61, FRONTEND_INFO_ST)

FE_SPECTRAL_INVERSION = {
   'INVERSION_OFF'  : 0,
   'INVERSION_ON'   : 1,
   'INVERSION_AUTO' : 2,
}

FE_BANDWIDTH = {
   'BANDWIDTH_8_MHZ' : 0,
   'BANDWIDTH_7_MHZ' : 1,
   'BANDWIDTH_6_MHZ' : 2,
   'BANDWIDTH_AUTO'  : 3,
}

FE_CODE_RATE = {
   'FEC_NONE' : 0,
   'FEC_1_2'  : 1,
   'FEC_2_3'  : 2,
   'FEC_3_4'  : 3,
   'FEC_4_5'  : 4,
   'FEC_5_6'  : 5,
   'FEC_6_7'  : 6,
   'FEC_7_8'  : 7,
   'FEC_8_9'  : 8,
   'FEC_AUTO' : 9,
}

FE_MODULATION = {
   'QPSK'     : 0,
   'QAM_16'   : 1,
   'QAM_32'   : 2,
   'QAM_64'   : 3,
   'QAM_128'  : 4,
   'QAM_256'  : 5,
   'QAM_AUTO' : 6,
   'VSB_8'    : 7,
   'VSB_16'   : 8,
}

FE_TRANSMISION_MODE = {
   'TRANSMISSION_MODE_2K'   : 0,
   'TRANSMISSION_MODE_8K'   : 1,
   'TRANSMISSION_MODE_AUTO' : 2,
}

FE_GUARD_INTERVAL = {
   'GUARD_INTERVAL_1_32' : 0,
   'GUARD_INTERVAL_1_16' : 1,
   'GUARD_INTERVAL_1_8'  : 2,
   'GUARD_INTERVAL_1_4'  : 3,
   'GUARD_INTERVAL_AUTO' : 4,
}

FE_HIERARCHY = {
   'HIERARCHY_NONE' : 0,
   'HIERARCHY_1'    : 1,
   'HIERARCHY_2'    : 2,
   'HIERARCHY_4'    : 3,
   'HIERARCHY_AUTO' : 4,
}

# See struct dvb_frontend_parameters at /usr/include/linux/dvb/frontend.h
FRONTEND_PARAMETERS_ST = "9L"
FRONTEND_SET_NO = _IOW('o', 76, FRONTEND_PARAMETERS_ST)
FRONTEND_GET_NO = _IOR('o', 77, FRONTEND_PARAMETERS_ST)

FE_STATUS = {
   'FE_HAS_SIGNAL'  : 0x01,
   'FE_HAS_CARRIER' : 0x02,
   'FE_HAS_VITERBI' : 0x04,
   'FE_HAS_SYNC'    : 0x08,
   'FE_HAS_LOCK'    : 0x10,
   'FE_TIMEDOUT'    : 0x20,
   'FE_REINIT'      : 0x40,
}
def fe_has_lock(status):    return (status & FE_STATUS['FE_HAS_LOCK'])
# See enum fe_status at /usr/include/linux/dvb/frontend.h
FRONTEND_READ_STATUS_ST = "L"
FRONTEND_READ_STATUS_NO = _IOR('o', 69, FRONTEND_READ_STATUS_ST)

DMX_INPUT = {
   'DMX_IN_FRONTEND' : 0,
   'DMX_IN_DVR'      : 1,
}

DMX_OUTPUT = {
   'DMX_OUT_DECODER' : 0,
   'DMX_OUT_TAP'     : 1,
   'DMX_OUT_TS_TAP'  : 2,
}

DMX_PES_TYPE = {
   'DMX_PES_AUDIO'    : 0,
   'DMX_PES_VIDEO'    : 1,
   'DMX_PES_TELETEXT' : 2,
   'DMX_PES_SUBTITLE' : 3,
   'DMX_PES_PCR'      : 4,
   'DMX_PES_OTHER'    : 20,
}

DMX_IMMEDIATE_START = 4

# See struct dmx_pes_filter_params at /usr/include/linux/dvb/dmx.h
DMX_SET_PES_FILTER_ST = "H4L"
DMX_SET_PES_FILTER_NO = _IOW('o', 44, DMX_SET_PES_FILTER_ST)
DMX_STOP_NO           = _IO('o', 42)

# Adapter #0 with Front-end #0 sourced to Demux/Dvr #0
DVB_HW_DEFAULT = [[0, 0, 0, ],]
dvb_transponders = []
itran = 0

DVB_SCANNER_RES_FREQ_DEFAULT = 1000000
DVB_DWELL_TIME_DEFAULT       = 5.0
DVB_FE_DEV_MASK_DEFAULT      = '/dev/dvb/adapter%u/frontend%u'
DVB_TS_DEV_MASK_DEFAULT      = '/dev/dvb/adapter%u/dvr%u'
DVB_DMX_DEV_MASK_DEFAULT     = '/dev/dvb/adapter%u/demux%u'

DVB_TS_STATUS = {
    'NO_DEV'  : 0,
    'IDLE'    : 1,
}

if __name__ != '__main__':
    import plugin
    import config

    if hasattr(config, 'DVB_HW'):
       DVB_HW = config.DVB_HW
    else:
       DVB_HW = DVB_HW_DEFAULT

    if hasattr(config, 'DVB_SCANNER_RES_FREQ'):
       DVB_SCANNER_RES_FREQ = config.DVB_SCANNER_RES_FREQ
    else:
       DVB_SCANNER_RES_FREQ = DVB_SCANNER_RES_FREQ_DEFAULT

    if hasattr(config, 'DVB_DWELL_TIME'):
       DVB_DWELL_TIME = config.DVB_DWELL_TIME
    else:
       DVB_DWELL_TIME = DVB_DWELL_TIME_DEFAULT

    if hasattr(config, 'DVB_FE_DEV_MASK'):
       DVB_FE_DEV_MASK  = config.DVB_FE_DEV_MASK
    else:
       DVB_FE_DEV_MASK  = DVB_FE_DEV_MASK_DEFAULT

    if hasattr(config, 'DVB_TS_DEV_MASK'):
       DVB_TS_DEV_MASK = config.DVB_TS_DEV_MASK
    else:
       DVB_TS_DEV_MASK = DVB_TS_DEV_MASK_DEFAULT

    if hasattr(config, 'DVB_DMX_DEV_MASK'):
       DVB_DMX_DEV_MASK = config.DVB_DMX_DEV_MASK
    else:
       DVB_DMX_DEV_MASK = DVB_DMX_DEV_MASK_DEFAULT

    class PluginInterface(plugin.DaemonPlugin):
        """
           DVB plug-in 
        """
        def __init__(self):
           plugin.DaemonPlugin.__init__(self)
	   self.dvb_machine = DVB_main()

	def poll(self):
           self.dvb_machine.step()
           return

        def config(self):
           """returns the config variables used by this plugin"""
	   return [
              ('DVB_HW', DVB_HW_DEFAULT,
               'Available (card,frontend,stream) tuples'),
              ('DVB_DWELL_TIME', DVB_DWELL_TIME_DEFAULT,
               'Time to remain in any valid frequency (in sec)'),
              ('DVB_SCANNER_RES_FREQ', DVB_SCANNER_RES_FREQ_DEFAULT,
               'Frequency resolution for scanner steps (in Hz)'),
              ('DVB_FE_DEV_MASK', DVB_FE_DEV_MASK_DEFAULT,
               'Mask for special device files of frontends'),
              ('DVB_TS_DEV_MASK', DVB_TS_DEV_MASK_DEFAULT,
               'Mask for special device files of TS feeds'),
              ('DVB_DMX_DEV_MASK', DVB_DMX_DEV_MASK_DEFAULT,
               'Mask for special device files of demuxes'),
	      ]

# if __name__ != '__main__':

class DVB_frontend:
    def __init__(self, transponder):

        self.type = FE_TYPE['FE_UNKNOWN']

        # Get frontend info
        if not self.open(transponder, os.O_RDONLY):
           return
	val = struct.pack(FRONTEND_INFO_ST, "", 0, 0, 0, 0, 0, 0, 0, 0, 0, 0)
        r = self.ioctl (transponder, FRONTEND_INFO_NO, val)
	if not r:
	   self.close(transponder)
           return

	results                 = struct.unpack(FRONTEND_INFO_ST, r)
	self.name               = results[0][:results[0].find('\0')]
	self.type               = results[1]
	self.frequency_min      = results[2]
	self.frequency_max      = results[3]
	self.frequency_stepsize = results[4]
	self.symbol_rate_min    = results[6]
	self.symbol_rate_max    = results[7]
	self.capabilities       = results[10]

	self.close(transponder)


    def open(self, transponder, mode=os.O_RDWR|os.O_NONBLOCK):
        hw_entry = transponder.hw_entry
        fdev     = DVB_FE_DEV_MASK % (hw_entry[0], hw_entry[1])

        self.close(transponder)
        try:
            transponder.fd_frontend = os.open(fdev, mode)
        except OSError, (errno, strerror):
            _debug_('Cannot open frontend device %r: %s' % (fdev, strerror), DCRITICAL)
            return False

        return True


    def close(self, transponder):
        if transponder.fd_frontend == -1: return
	os.close(transponder.fd_frontend)
        transponder.fd_frontend = -1


    def ioctl(self, transponder, request, val=None):
        try:
	   if val:
	      r = fcntl.ioctl(transponder.fd_frontend, i32(request), val)
	   else:
	      r = fcntl.ioctl(transponder.fd_frontend, i32(request))
        except IOError, (errno, strerror):
            hw_entry = transponder.hw_entry
            _debug_ ('Error ioctl request %i for adapter/frontend %r/%r: %s' % (request, hw_entry[0], hw_entry[1], strerror), DCRITICAL)
            return None
        return r


    def status (self, transponder):
	val = struct.pack (FRONTEND_READ_STATUS_ST, 0)
        r = self.ioctl (transponder, FRONTEND_READ_STATUS_NO, val)
        if not r: return 0

	results = struct.unpack(FRONTEND_READ_STATUS_ST, r)
        return results[0]


    def tune (self, transponder):
        if self.type == FE_TYPE['FE_OFDM']:
	   val = struct.pack(FRONTEND_PARAMETERS_ST,
              transponder.frequency,
              transponder.spectral_inversion,
              transponder.bandwidth,
              transponder.code_rate_HP,
              transponder.code_rate_LP,
              transponder.baseband_modulation,
              transponder.transmision_mode,
              transponder.guard_interval,
              transponder.hierarchy)
        elif self.type == FE_TYPE['FE_ATSC']:
	   val = struct.pack(FRONTEND_PARAMETERS_ST,
              transponder.frequency,
              transponder.spectral_inversion,
              transponder.baseband_modulation,
              0, 0, 0, 0, 0, 0)
        elif self.type == FE_TYPE['FE_QAM']:
	   val = struct.pack(FRONTEND_PARAMETERS_ST,
              transponder.frequency,
              transponder.spectral_inversion,
              transponder.symbol_rate,
              transponder.fec_inner,
              transponder.baseband_modulation,
              0, 0, 0, 0)
        elif self.type == FE_TYPE['FE_QPSK']:
	   val = struct.pack(FRONTEND_PARAMETERS_ST,
              transponder.frequency,
              transponder.spectral_inversion,
              transponder.symbol_rate,
              transponder.fec_inner,
              0, 0, 0, 0, 0)
        else:
	   self.close(transponder)
           self.type = FE_TYPE['FE_UNKNOWN']
           return False

        if not self.ioctl (transponder, FRONTEND_SET_NO, val):
	   self.close(transponder)
           return False

        return True

class DVB_stream:
    def __init__(self, transponder):
	self.status = DVB_TS_STATUS['NO_DEV']

        # Try to open dvr device
        if not self.open (transponder): return

	self.close(transponder)
	self.status = DVB_TS_STATUS['IDLE']


    def open(self, transponder, pid=-1):
        hw_entry = transponder.hw_entry
        if pid == -1:
           fdev = DVB_TS_DEV_MASK % (hw_entry[0], hw_entry[2])
           mode = os.O_RDONLY|os.O_NONBLOCK
        else:
           fdev = DVB_DMX_DEV_MASK % (hw_entry[0], hw_entry[2])
           mode = os.O_RDWR|os.O_NONBLOCK

        self.close(transponder, pid)
        try:
            transponder.fd_stream[pid] = os.open(fdev, mode)
        except OSError, (errno, strerror):
            _debug_('Cannot open stream device %r: %s' % (fdev, strerror), DCRITICAL)
            return False

        if pid != -1:
           if transponder.ts.psi.pid_carries_audio(pid):
              stream_type = DMX_PES_TYPE['DMX_PES_AUDIO']
           elif transponder.ts.psi.pid_carries_video(pid):
              stream_type = DMX_PES_TYPE['DMX_PES_VIDEO']
           else:
              stream_type = DMX_PES_TYPE['DMX_PES_OTHER']

	   val = struct.pack(DMX_SET_PES_FILTER_ST, pid,
                             DMX_INPUT['DMX_IN_FRONTEND'],
                             DMX_OUTPUT['DMX_OUT_TS_TAP'],
                             stream_type, DMX_IMMEDIATE_START)

           if not self.ioctl (transponder, DMX_SET_PES_FILTER_NO, pid, val):
	      self.close(transponder, pid)
              return False
           #_debug_ ("OPEN PES Filter 0x%x" % pid, DINFO)

        return True


    def close(self, transponder, pid=-1):
        if pid not in transponder.fd_stream: return

        if pid != -1:
           self.ioctl (transponder, DMX_STOP_NO, pid)

	os.close(transponder.fd_stream[pid])
        del transponder.fd_stream[pid]

        # When closing dvr device (pid == -1), recursively close
        # all other demux devices (pid != -1)
        # WARNING. The file descriptor of the dvr device (pid == -1)
        # to have to be closed before the other demux devices.
        # Otherwise, rare effects arise (as persistent residual data
        # in the kernel buffers)
        if pid == -1:
           for dmx_pid in transponder.fd_stream.keys():
               self.close(transponder, dmx_pid)
               #_debug_ ("CLOSE PES Filter 0x%x" % dmx_pid, DINFO)


    def ioctl(self, transponder, request, pid=-1, val=None):
        try:
	   if val:
	      r = fcntl.ioctl(transponder.fd_stream[pid], i32(request), val)
	   else:
	      r = fcntl.ioctl(transponder.fd_stream[pid], i32(request))
        except IOError, (errno, strerror):
            hw_entry = transponder.hw_entry
            _debug_('Error ioctl request %i for adapter/demux %r/%r: %s' % (request, hw_entry[0], hw_entry[2], strerror), DCRITICAL)
            return None

        return r


    def read_packet(self, transponder):
        try:
           ret = os.read (transponder.fd_stream[-1], 188)
        except OSError, (errno, strerror):
           if errno != 11: _debug_('Read error %i: %s' % (errno, strerror), DERROR)
           return ""
        if len(ret) != 188: _debug_("Read over 188 bytes, len %i" % len(ret), DWARNING)

        return ret


class DVB_transponder:
    def __init__(self, hw_entry):
        self.hw_entry      = hw_entry
        self.tunning_time  = 0
        self.is_locked     = False

        self.fd_frontend = -1
	self.frontend = DVB_frontend (self)
	if self.frontend.type == FE_TYPE['FE_UNKNOWN']:
	   del self.frontend
	   self.frontend = None

        self.fd_stream= {}
	self.stream= DVB_stream(self)
	if not self.stream.status:
	   del self.stream
	   self.stream = None

        if not self.frontend or not self.stream:
           return

        # If this hw_entry has got no frequency info, scan for
        # frequencies with lockable signal.
        if len(hw_entry) > 3:
           self.is_scanner = False
           self.frequency  = hw_entry[3]
           self.ts         = dvb_ts.DVB_TS()
        else:
           self.is_scanner = True
           self.frequency  = self.frontend.frequency_max
           

    def tune (self):
        self.tunning_time = 0
        hw_entry          = self.hw_entry

        self.spectral_inversion = FE_SPECTRAL_INVERSION['INVERSION_AUTO']
        if len(hw_entry) > 4 and hw_entry[4] in FE_SPECTRAL_INVERSION:
           self.spectral_inversion = FE_SPECTRAL_INVERSION[hw_entry[4]]

        if self.frontend.type == FE_TYPE['FE_OFDM']:
           # OFDM is typically used by DVB-T (European terrestrial digital TV)
           if self.frequency <= 300000000:
              self.bandwidth = FE_BANDWIDTH['BANDWIDTH_7_MHZ'] 
           else:
              self.bandwidth = FE_BANDWIDTH['BANDWIDTH_8_MHZ'] 
           self.code_rate_HP        = FE_CODE_RATE['FEC_AUTO']
           self.code_rate_LP        = FE_CODE_RATE['FEC_AUTO']
           self.baseband_modulation = FE_MODULATION['QAM_AUTO']
           self.transmision_mode    = FE_TRANSMISION_MODE['TRANSMISSION_MODE_AUTO']
           self.guard_interval      = FE_GUARD_INTERVAL['GUARD_INTERVAL_AUTO']
           self.hierarchy           = FE_HIERARCHY['HIERARCHY_NONE']
           if len(hw_entry) > 5 and hw_entry[5] in FE_BANDWIDTH:
              self.bandwidth = FE_BANDWIDTH[hw_entry[5]]
           if len(hw_entry) > 6 and hw_entry[6] in FE_CODE_RATE:
              self.code_rate_HP = FE_CODE_RATE[hw_entry[6]]
           if len(hw_entry) > 7 and hw_entry[7] in FE_CODE_RATE:
              self.code_rate_LP = FE_CODE_RATE[hw_entry[7]]
           if len(hw_entry) > 8 and hw_entry[8] in FE_MODULATION:
              self.baseband_modulation = FE_MODULATION[hw_entry[8]]
           if len(hw_entry) > 9 and hw_entry[9] in FE_TRANSMISION_MODE:
              self.transmision_mode = FE_TRANSMISION_MODE[hw_entry[9]]
           if len(hw_entry) > 10 and hw_entry[10] in FE_GUARD_INTERVAL:
              self.guard_interval = FE_GUARD_INTERVAL[hw_entry[10]]
           if len(hw_entry) > 11 and hw_entry[11] in FE_HIERARCHY:
              self.hierarchy = FE_HIERARCHY[hw_entry[11]]
        elif self.frontend.type == FE_TYPE['FE_ATSC']:
           # ATSC is typically used by US terrestrial digital TV
           self.baseband_modulation = FE_MODULATION['VSB_8']
           if len(hw_entry) > 5 and hw_entry[5] in FE_MODULATION:
              self.baseband_modulation = FE_MODULATION[hw_entry[5]]
        elif self.frontend.type == FE_TYPE['FE_QAM']:
           # QAM is typically used by DVB-C (cable digital TV)
           # See http://www.satellite-calculations.com/Satellite/bitrates.htm
           # for a symbol rate calculator
           if len(hw_entry) > 5:
              self.symbol_rate = hw_entry[5]
           else:
              self.symbol_rate = self.frontend.symbol_rate_min
           self.fec_inner           = FE_CODE_RATE['FEC_AUTO']
           self.baseband_modulation = FE_MODULATION['QAM_AUTO']
           if len(hw_entry) > 6 and hw_entry[6] in FE_CODE_RATE:
              self.fec_inner = FE_CODE_RATE[hw_entry[6]]
           if len(hw_entry) > 7 and hw_entry[7] in FE_MODULATION:
              self.baseband_modulation = FE_MODULATION[hw_entry[7]]
        elif self.frontend.type == FE_TYPE['FE_QPSK']:
           # QPSK is typically used by DVB-S (satellite digital TV)
           # See http://www.satellite-calculations.com/Satellite/bitrates.htm
           # for a symbol rate calculator
           if len(hw_entry) > 5:
              self.symbol_rate = hw_entry[5]
           else:
              self.symbol_rate = self.frontend.symbol_rate_min
           self.fec_inner = FE_CODE_RATE['FEC_AUTO']
           if len(hw_entry) > 6 and hw_entry[6] in FE_CODE_RATE:
              self.fec_inner = FE_CODE_RATE[hw_entry[6]]
        else:
           _debug_('DVB_transponder.tune(): Unknown frontend type', DERROR)
           return False

        if not self.frontend.tune(self):
           return False

        self.tunning_time = time.time()
        return True

    def open(self):
        if not self.frontend or not self.stream:
           return False

        if not self.frontend.open(self):
           self.close()
           return False

        if not self.is_scanner:
           # Open dvr device
           if not self.stream.open(self):
              self.close()
              return False

        return True

    def close(self):
        self.is_locked    = False
        self.tunning_time = 0
        if self.stream:   self.stream.close(self)
        if self.frontend: self.frontend.close(self)


class DVB_main:
    def __init__(self):
       for id in range(len(DVB_HW)):
          if len(DVB_HW[id]) >= 3:
             transponder = DVB_transponder(DVB_HW[id])
             dvb_transponders.append(transponder)


    # Return True if next _step should be called for the same present transponder
    def _step(self, transponder):
       global itran

       frontend = transponder.frontend
       stream   = transponder.stream
       if not frontend or not stream:
          return False

       if transponder.is_scanner:
          if transponder.frequency == transponder.frontend.frequency_min:
             if __name__ == '__main__':
                _debug_("SCAN[%i] Freq %i FIN" % (itran, transponder.frequency), DCRITICAL)
          if transponder.frequency < transponder.frontend.frequency_min:
             return False

       # Test if tunning is required. Silently return if tunning fails
       # because the frontend could be used for another transponder or application
       if not transponder.is_locked and not transponder.tunning_time:
          if not transponder.open(): return False
          transponder.tune()
          return True

       delta_time = time.time() - transponder.tunning_time
       if not transponder.is_locked:
          if delta_time > 2.0:
             if transponder.is_scanner:
                transponder.frequency -= 1000000
                if __name__ == '__main__':
                   _debug_("SCAN[%i] Next Freq %i" % (itran, transponder.frequency), DCRITICAL)
             return False
          if not fe_has_lock(frontend.status(transponder)):
             return True

       # Now, the transponder is well tuned
       if __name__ == '__main__' and not transponder.is_locked:
          if transponder.is_scanner:
             _debug_("SCAN[%i] Freq %i: LOCKED at %f !!!" % (itran, transponder.frequency, delta_time), DCRITICAL)
          else:
             _debug_("FREQ[%i] Freq %i: LOCKED at %f !!!" % (itran, transponder.frequency, delta_time), DCRITICAL)
       transponder.is_locked = True

       if transponder.is_scanner:
          # Add a new transponder for the frequency locked by the scanner
          hw_entry        = transponder.hw_entry
          new_hw_entry    = [hw_entry[0], hw_entry[1], hw_entry[2], transponder.frequency]
          new_transponder = DVB_transponder(new_hw_entry)
          dvb_transponders.append(new_transponder)

          transponder.frequency -= DVB_SCANNER_RES_FREQ
          if __name__ == '__main__':
             _debug_("SCAN[%i] Next Freq %i" % (itran, transponder.frequency), DCRITICAL)
          return False

       # Now, the transponder is well tuned and is NOT a scanner
       # Update demux filters according to the active_pids list
       active_pids = transponder.ts.get_active_pids()
       for pid in active_pids:
           if not pid in transponder.fd_stream:
              stream.open(transponder, pid)
       for pid in transponder.fd_stream.keys():
           if pid == -1: continue
           if pid not in active_pids:
              stream.close(transponder, pid)

       packet = stream.read_packet(transponder)
       ts_ret = transponder.ts.add_packet(packet, transponder.frequency)
       
       if ts_ret == -2:
             _debug_("Critical transport error. Packet: %s" % binascii.hexlify(packet), DWARNING)
       elif ts_ret == -1:
             _debug_("Transport error. Packet: %s" % binascii.hexlify(packet), DWARNING)
       elif ts_ret == 0:
             _debug_("Payload error. Packet: %s" % binascii.hexlify(packet), DWARNING)

       expired = delta_time > DVB_DWELL_TIME
       if expired:
          if __name__ == '__main__':
             _debug_("FREQ[%i] Freq %i. PSI Summary: %s" % (itran, transponder.frequency, self.summarize_psi(transponder)), DCRITICAL)

       return not expired


    def summarize_psi(self, transponder):
        ret = ""
        for (table_id, table) in transponder.ts.psi.tables.items():
          for (subtable_id, subtable) in table.subtables.items():
            if subtable.current_version == -1: continue 
            ret += "(0x%x, "            % table_id
            ret += "0x%x, 0x%x, 0x%x, " % subtable_id
            ret += "%i), "              % subtable.current_version
        return ret


    def step(self):
       global itran
       if itran >= len(dvb_transponders): itran = 0
       transponder = dvb_transponders[itran]

       # The function _step return True if next step should be
       # called for the same present transponder
       if not self._step(transponder):
          transponder.tunning_time = time.time()
          if len(dvb_transponders) > 1:
             transponder.close()
             itran += 1


# Execute python dvb3l.py
if __name__ == '__main__':
    import __builtin__
    import kaa

    DINFO     = 0
    DWARNING  = -1
    DERROR    = -2
    DCRITICAL = -3
    def _debug_function_ (str, level=DWARNING):
       actual_debug_level = DERROR
       if level <= actual_debug_level: print str
       return

    __builtin__.__dict__['_debug_']   = _debug_function_
    __builtin__.__dict__['DCRITICAL'] = DCRITICAL
    __builtin__.__dict__['DERROR']    = DERROR
    __builtin__.__dict__['DWARNING']  = DWARNING
    __builtin__.__dict__['DINFO']     = DINFO

    def poll(dvb_machine):
       dvb_machine.step()
       return

    DVB_SCANNER_RES_FREQ = DVB_SCANNER_RES_FREQ_DEFAULT
    DVB_DWELL_TIME       = DVB_DWELL_TIME_DEFAULT
    DVB_FE_DEV_MASK      = DVB_FE_DEV_MASK_DEFAULT
    DVB_TS_DEV_MASK      = DVB_TS_DEV_MASK_DEFAULT
    DVB_DMX_DEV_MASK     = DVB_DMX_DEV_MASK_DEFAULT

#    DVB_HW = [ [3, 3, 3, 700000000, ],
#               [0, 0, 0, 770000000, "INVERSION_AUTO", "BANDWIDTH_8_MHZ", "FEC_2_3", "FEC_1_2", "QAM_64", "TRANSMISSION_MODE_8K", "GUARD_INTERVAL_1_4", "HIERARCHY_NONE"] ]
#    DVB_HW = [ [0, 0, 0, 850000000, ], [0, 0, 0, 842000000, ], [0, 0, 0, 834000000, ], [0, 0, 0, 770000000, ] ]
#    DVB_HW = [ [0, 0, 0, 834000000, ] ]
    DVB_HW = [ [0, 0, 0, ] ]

    dvb_machine = DVB_main()

    kaa.Timer(poll, dvb_machine).start(0.1)

    kaa.main.run()

# if __name__ == '__main__':


