# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------
#
import struct
import h222_desc
import dvb_util

class DVB_descriptor (h222_desc.H222_descriptor):
      def __init__(self, table_id, subtable_key=(0,0,0)):
          h222_desc.H222_descriptor.__init__(self, table_id, subtable_key)
          return


      # As Annex A.2 ETSI EN.300.468
      def __text_decoding (self, text):
         if not len(text):                    return ("\x00", text)
         if text[0] not in range(0x00, 0x20): return ("\x00", text)
         if text[0] != 0x10:                  return (text[:1], text[1:])
         return (text[:3], text[3:])


      # Tag 0x40. Network name descriptor: Section 6.2.26 ETSI EN.300.468
      def __add_network_name_descriptor(self):
          format = "%is" % len(self.descriptor_data)
          (self.network_name, ) = struct.unpack_from(format, self.descriptor_data, 0)
          _debug_("DVB Network Name: %s" % self.network_name, DINFO)
          return True


      # Tag 0x41. Service list descriptor: Section 6.2.34 ETSI EN.300.468
      def __add_service_list_descriptor(self):

          # Dictionary indexed by service_id
          self.services = {}
          for idx in range(len(self.descriptor_data)/3):
              (b1, b2, b3, ) = struct.unpack_from("3B", self.descriptor_data, 3*idx)
              service_id   = b1<<8 | b2
              service_type = b3
              self.services[service_id] = service_type
          return True


      # Tag 0x42. Stuffing descriptor: Section 6.2.39 ETSI EN.300.468
      def __add_stuffing_descriptor(self):
          # No Action
          return True


      # Tag 0x43. Satellite delivery system descriptor: Section 6.2.13.2 ETSI EN.300.468
      def __add_satellite_delivery_system_descriptor(self):
          (b1, b2, b3, b4, b5, b6, b7) = struct.unpack_from("7B", self.descriptor_data, 0)
          self.frequency         = b1<<24 | b2<<16 | b3<<8 | b4
          self.orbital_position  = b5<<8 | b6
          self.west_east_flag    = ((b7 & 0x80) != 0x00)
          self.polarization      = (b7 & 0x60)>>5
          self.roll_off          = (b7 & 0x18)>>3
          self.modulation_system = ((b7 & 0x04) != 0x00)
          self.modulation_type   = b7 & 0x03

          (b1, b2, b3, b4, ) = struct.unpack_from("4B", self.descriptor_data, 7)
          self.symbol_rate = b1<<20 | b2<<12 | b3<<4 | (b4 & 0xf0)>>4
          self.FEC_inner   = b4 & 0x0f
          return True


      # Tag 0x44. Cable delivery system descriptor: Section 6.2.13.1 ETSI EN.300.468
      def __add_cable_delivery_system_descriptor(self):
          (b1, b2, b3, b4, b5, b6, b7) = struct.unpack_from("7B", self.descriptor_data, 0)
          self.frequency  = b1<<24 | b2<<16 | b3<<8 | b4
          self.FEC_outer  = b6 & 0x0f
          self.modulation = b7

          (b1, b2, b3, b4, ) = struct.unpack_from("4B", self.descriptor_data, 7)
          self.symbol_rate = b1<<20 | b2<<12 | b3<<4 | (b4 & 0xf0)>>4
          self.FEC_inner   = b4 & 0x0f
          return True


      # Tag 0x45. VBI data descriptor: Section 6.2.46 ETSI EN.300.468
      def __add_VBI_data_descriptor(self):
          pointer_data = 0
          len_data     = len(self.descriptor_data)

          # Dictionary indexed by data_service_id
          self.data_services = {}
          while pointer_data < len_data:
            (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, pointer_data)
            data_service_id                = b1
            data_service_descriptor_length = b2

            if data_service_id == 0x00 or data_service_id == 0x03 or data_service_id in range(0x08, 0x100):
               pointer_data += data_service_descriptor_length
               continue

            line_numbers = []
            for idx in range(data_service_descriptor_length):
               (b3, ) = struct.unpack_from("B", self.descriptor_data, pointer_data+idx)
               field_parity = (b3 & 0x20) != 0x00
               line_offset  = b3 & 0x1f
               line_numbers.append((field_parity, line_offset))
            self.data_services[data_service_id] = line_numbers
            pointer_data += data_service_descriptor_length

          return True


      # Tag 0x46. VBI teletext descriptor: Section 6.2.47 ETSI EN.300.468
      def __add_VBI_teletext_descriptor(self):
          return __add_teletext_descriptor()


      # Tag 0x47. Bouquet name descriptor: Section 6.2.4 ETSI EN.300.468
      def __add_bouquet_name_descriptor(self):
          format = "%is" % len(self.descriptor_data)
          (self.bouquet_name, ) = struct.unpack_from(format, self.descriptor_data, 0)
          return True


      # Tag 0x48. Service descriptor: Section 6.2.32 ETSI EN.300.468
      def __add_service_descriptor(self):
          (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, 0)
          self.service_type            = b1
          service_provider_name_length = b2
          pointer_data = 2

          format = "%isB" % service_provider_name_length
          (s1, b3,) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          self.service_provider_name = self.__text_decoding(s1)
          service_name_length = b3
          pointer_data += service_provider_name_length + 1
     
          format = "%is" % service_name_length
          (s2, ) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          self.service_name = self.__text_decoding(s2)

          _debug_("DVB Service, Provider %s, Name: %s" % (self.service_provider_name[1], self.service_name[1]), DINFO)
          return True


      # Tag 0x49. Country availability descriptor: Section 6.2.10 ETSI EN.300.468
      def __add_country_availability_descriptor(self):
          (b1, ) = struct.unpack_from("B", self.descriptor_data, 0)
          self.country_availability_flag = (b1 & 0x80) != 0x00

          self.countries=[]
          for idx in range((len(self.descriptor_data)-1)/3):
              (s1, ) = struct.unpack_from("3s", self.descriptor_data, 1+idx*3)
              self.countries.append(s1)

          return True


      # Tag 0x4a. Linkage descriptor: Section 6.2.18 ETSI EN.300.468
      def __add_linkage_descriptor(self):
          (b1, b2, b3, b4, b5, b6, b7) = struct.unpack_from("7B", self.descriptor_data, 0)
          self.transport_stream_id = ((b1 & 0x00ff) << 8) | b2
          self.original_network_id = ((b3 & 0x00ff) << 8) | b4
          self.service_id          = ((b5 & 0x00ff) << 8) | b6
          self.linkage_type        = b7
          pointer_data             = 7

          if self.linkage_type == 0x08:
             (b8,) = struct.unpack_from("B", self.descriptor_data, pointer_data)
             self.handover_type = (b8 & 0x00f0) >> 4
             self.origin_type   = ((b8 & 0x0001) != 0x00)
             pointer_data      += 1

             if self.handover_type in range(0x00, 0x04):             
                (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, pointer_data)
                self.network_id = ((b1 & 0x00ff) << 8) | b2
                pointer_data += 2

             if self.origin_type == 0x00:             
                (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, pointer_data)
                self.initial_service_id = ((b1 & 0x00ff) << 8) | b2
                pointer_data += 2

          self.private_data = self.descriptor_data[pointer_data:]
          return True


      # Tag 0x4b. NVOD reference descriptor: Section 6.2.25 ETSI EN.300.468
      def __add_NVOD_reference_descriptor(self):

          self.NVOD_references = []
          for idx in range(len(self.descriptor_data)):
              (b1, b2, b3, b4, b5, b6, ) = struct.unpack_from("6B", self.descriptor_data, 6*idx)
              transport_stream_id = ((b1 & 0x00ff) << 8) | b2
              original_network_id = ((b3 & 0x00ff) << 8) | b4
              service_id          = ((b5 & 0x00ff) << 8) | b6
              self.NVOD_references.append((transport_stream_id, original_network_id, service_id))

          return True


      # Tag 0x4c. Time shifted service descriptor: Section 6.2.44 ETSI EN.300.468
      def __add_time_shifted_service_descriptor(self):
          (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, 0)
          self.reference_service_id = ((b1 & 0x00ff) << 8) | b2
          return True


      # Tag 0x4d. Short event descriptor: Section 6.2.36 ETSI EN.300.468
      def __add_short_event_descriptor(self):
          (self.ISO_639_language_code, b1) = struct.unpack_from("3sB", self.descriptor_data, 0)
          pointer_data = 4

          format = "%isB" % b1
          (s1, b2) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          self.event_name_char = self.__text_decoding(s1)
          pointer_data += b1+1

          format = "%is" % b2
          (s2, ) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          self.text_char = self.__text_decoding(s2)

          #_debug_("DVB Event, Name %s, Text: %s" % (self.event_name_char[1], self.text_char[1]), DINFO)
          return True


      # Tag 0x4e. Extended event descriptor: Section 6.2.15 ETSI EN.300.468
      def __add_extended_event_descriptor(self):

          # Dictionary indexed by item_description
          self.items = {}

          (b1, self.ISO_639_language_code, b2) = struct.unpack_from("B3sB", self.descriptor_data, 0)
          pointer_data = 5

          self.descriptor_number      = (b1 & 0x00f0) >> 4
          self.last_descriptor_number =  b1 & 0x000f
          length_of_items             =  b2

          pointer_items = 0
          while pointer_items < length_of_items:
             (item_description_length, ) = struct.unpack_from("B", self.descriptor_data, pointer_data)
             pointer_items += 1
             pointer_data  += 1

             format = "%isB" % item_description_length
             (s1, item_length) = struct.unpack_from(format, self.descriptor_data, pointer_data)
             item_description_char = self.__text_decoding(s1)
             pointer_items += item_description_length+1
             pointer_data  += item_description_length+1

             format = "%is" % item_length
             (s2, ) = struct.unpack_from(format, self.descriptor_data, pointer_data)
             item_char = self.__text_decoding(s2)
             pointer_items += item_length
             pointer_data  += item_length

             self.items[item_description_char] = item_char

          (text_length, ) = struct.unpack_from("B", self.descriptor_data, pointer_data)
          format = "%is" % text_length
          (s3, ) = struct.unpack_from(format, self.descriptor_data, pointer_data+1)
          self.text_char = self.__text_decoding(s3)

          return True


      # Tag 0x4f. Time shifted event descriptor: Section 6.2.43 ETSI EN.300.468
      def __add_time_shifted_event_descriptor(self):
          (b1, b2, b3, b4, ) = struct.unpack_from("4B", self.descriptor_data, 0)
          self.reference_service_id = ((b1 & 0x00ff) << 8) | b2
          self.reference_event_id   = ((b3 & 0x00ff) << 8) | b4
          return True


      # Tag 0x50. Component descriptor: Section 6.2.8 ETSI EN.300.468
      def __add_component_descriptor(self):
          (b1, b2, b3, s1) = struct.unpack_from("3B3s", self.descriptor_data, 0)

          # Stream_content and component_type as ETSI EN.300.468 Table 26
          self.stream_content        = (b1 & 0x0f)
          self.component_type        = b2
          self.component_tag         = b3
          self.ISO_639_language_code = s1

          text_char_length = len(self.descriptor_data) - 6
          if text_char_length < 0: return False
          format = "%is" % text_char_length
          (s1, ) = struct.unpack_from(format, self.descriptor_data, 6)
          self.text_char = self.__text_decoding(s1)

          return True


      # Tag 0x52. Stream identifier descriptor: Section 6.2.38 ETSI EN.300.468
      def __add_stream_identifier_descriptor(self):
          (b1, ) = struct.unpack_from("B", self.descriptor_data, 0)
          self.component_tag = b1

          return True


      # Tag 0x54. Content descriptor: Section 6.2.9 ETSI EN.300.468
      def __add_content_descriptor(self):
          # List of content identifiers as ETSI EN.300.468 Table 28
          self.content_identifier_tuples = []
          self.user_identifiers          = []

          for idx in range(len(self.descriptor_data)/2):
              (b1, b2) = struct.unpack_from("2B", self.descriptor_data, 2*idx)
              content_nibble_level1 = (b1 & 0xf0) >> 4
              content_nibble_level2 =  b1 & 0x0f
              user_nibble1          = (b2 & 0xf0) >> 4
              user_nibble2          =  b2 & 0x0f

              self.content_identifier_tuples.append((content_nibble_level1,content_nibble_level2))
              self.user_identifiers.append(user_nibble1)
              self.user_identifiers.append(user_nibble2)

          return True


      # Tag 0x55. Parental rating descriptor: Section 6.2.27 ETSI EN.300.468
      def __add_parental_rating_descriptor(self):
          # Dictionary indexed by country_code
          self.ratings = {}

          for idx in range(len(self.descriptor_data)/4):
              (country_code, rating) = struct.unpack_from("3sB", self.descriptor_data, 4*idx)
              self.ratings[country_code] = rating

          return True


      # Tag 0x56. Teletext descriptor: Section 6.2.42 ETSI EN.300.468
      def __add_teletext_descriptor(self):
          self.teletext_pages = []

          for idx in range(len(self.descriptor_data)/5):
              (s1, b1, b2) = struct.unpack_from("3s2B", self.descriptor_data, 5*idx)
              ISO_639_language_code    = s1
              teletext_type            = (b1 & 0xf8) >> 3
              teletext_magazine_number = b1 & 0x07
              teletext_page_number     = b2
              self.teletext_pages.append((ISO_639_language_code, teletext_type, teletext_magazine_number, teletext_page_number))

          return True


      # Tag 0x58. Local time offset descriptor: Section 6.2.19 ETSI EN.300.468
      def __add_local_time_offset_descriptor(self):
          # Dictionary indexed by the tuple (country_code, country_region_id)
          self.time_offset_info = {}

          for idx in range(len(self.descriptor_data)/13):
              (country_code, b1) = struct.unpack_from("3sB", self.descriptor_data, 13*idx)
              country_region_id          = (b1 & 0xfc) >> 2
              local_time_offset_polarity = (b1 & 0x01) != 0x00

              (bcd1, bcd2) = struct.unpack_from("2B", self.descriptor_data, 13*idx+4)
              local_time_offset_seconds  = 3600*(10*((bcd1 & 0xf0)>>4) + (bcd1 & 0x0f))
              local_time_offset_seconds +=   60*(10*((bcd2 & 0xf0)>>4) + (bcd2 & 0x0f))
              if local_time_offset_polarity: local_time_offset_seconds *= -1

              utc_of_change = dvb_util.parse_utc_time(self.descriptor_data[13*idx+6:])
              if utc_of_change <= 0: return False

              (bcd3, bcd4) = struct.unpack_from("2B", self.descriptor_data, 13*idx+11)
              next_time_offset_seconds  = 3600*(10*((bcd3 & 0xf0)>>4) + (bcd3 & 0x0f))
              next_time_offset_seconds +=   60*(10*((bcd4 & 0xf0)>>4) + (bcd4 & 0x0f))
              if local_time_offset_polarity: next_time_offset_seconds *= -1

              self.time_offset_info[(country_code, country_region_id)] = (local_time_offset_seconds, utc_of_change, next_time_offset_seconds)

          return True


      # Tag 0x59. Subtitling descriptor: Section 6.2.40 ETSI EN.300.468
      def __add_subtitling_descriptor(self):
          self.subtitles = []
          for idx in range(len(self.descriptor_data)/8):
              (s1, b1, b2, b3, b4, b5) = struct.unpack_from("3s5B", self.descriptor_data, 8*idx)
              ISO_639_language_code = s1
              subtitling_type       = b1
              composition_page_id   = (b2 & 0x00ff) << 8 | b3
              ancillary_page_id     = (b4 & 0x00ff) << 8 | b5
              self.subtitles.append((ISO_639_language_code, subtitling_type, composition_page_id, ancillary_page_id))

          return True


      # Tag 0x5a. Terrestial delivery system descriptor: Section 6.2.13.4 ETSI EN.300.468
      def __add_terrestrial_delivery_system_descriptor(self):
          (b1, b2, b3, b4, b5, b6, b7) = struct.unpack_from("7B", self.descriptor_data, 0)
          self.center_frequency       = b1<<24 | b2<<16 | b3<<8 | b4
          self.bandwidth              =  (b5 & 0xe0)>>5
          self.priority               = ((b5 & 0x10) != 0x00)
          self.Time_Slicing_indicator = ((b5 & 0x08) != 0x00)
          self.MPE_FEC_indicator      = ((b5 & 0x04) != 0x00)
          self.constellation          =  (b6 & 0xc0)>>6
          self.hierarchy_information  =  (b6 & 0x38)>>3
          self.code_rate_HP_stream    =   b6 & 0x07
          self.code_rate_LP_stream    =  (b7 & 0xe0)>>5
          self.guard_interval         = (b7 & 0x18)>>3
          self.transmission_mode      = (b7 & 0x06)>>1
          self.other_frequency_flag   = ((b7 & 0x01) != 0x00)
          _debug_("Terrestrial frequency: %i" % self.center_frequency, DINFO)
          return True


      # Tag 0x5f. Private data specifier descriptor: Section 6.2.30 ETSI EN.300.468
      def __add_private_data_specifier_descriptor(self):
          (b1, b2, b3, b4, ) = struct.unpack_from("4B", self.descriptor_data, 0)
          self.private_data_specifier = b1<<24 | b2<<16 | b3<<8 | b4
          return True


      # Tag 0x61. Short smoothing buffer descriptor: Section 6.2.37 ETSI EN.300.468
      def __add_short_smoothing_buffer_descriptor(self):
          (b1, ) = struct.unpack_from("B", self.descriptor_data, 0)
          self.sb_size      = (b1 & 0xc0) >> 6
          self.sb_leak_rate =  b1 & 0x3f
          return True

      
      # Tag 0x64. Data broadcast descriptor: Section 6.2.11 ETSI EN.300.468
      def __add_data_broadcast_descriptor(self):
          (b1, b2, b3, b4) = struct.unpack_from("4B", self.descriptor_data, 0)
          pointer_data = 4
          self.data_broadcast_id = ((b1 & 0x00ff) << 8) | b2
          self.component_tag     = b3
          selector_length        = b4

          format = "%is" % selector_length
          (self.selector, ) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          pointer_data += selector_length

          (s1, b5) = struct.unpack_from("3sB", self.descriptor_data, pointer_data)
          pointer_data += 4
          self.ISO_639_language_code = s1
          text_length                = b5

          format = "%is" % text_length
          (s2, ) = struct.unpack_from(format, self.descriptor_data, pointer_data)
          self.text = self.__text_decoding(s2)

          return True


      # Tag 0x66. Data broadcast id descriptor: Section 6.2.12 ETSI EN.300.468
      # Section 10.7.2 ETSI TS.102.812 (http://www.mhp.org)
      def __add_data_broadcast_id_descriptor(self):
          (b1, b2, ) = struct.unpack_from("2B", self.descriptor_data, 0)
          self.data_broadcast_id = ((b1 & 0x00ff) << 8) | b2
          self.id_selector       = self.descriptor_data[2:]
          return True


      # Tag 0x69. PDC descriptor 
      def __add_PDC_descriptor(self):
          (b1, b2, b3, ) = struct.unpack_from("3B", self.descriptor_data, 0)
          self.PIL    = (b1 & 0x0f) << 16 | b2 << 8 | b3
          self.minute =  self.PIL & 0x00003f
          self.hour   = (self.PIL & 0x0007c0) >> 6
          self.month  = (self.PIL & 0x007800) >> 11
          self.day    = (self.PIL & 0x0f8000) >> 15

          PIL_str = binascii.hexlify(self.descriptor_data[:3])
          _debug_("PDC descriptor: PIL %s. Month %i, Day %i, Hour %i, Minute %i" % (PIL_str, self.month, self.day, self.hour, self.minute), DINFO)
          return True


      # Tag 0x6f. Application signalling descriptor 
      # Section 10.7.1 ETSI TS.102.812 (http://www.mhp.org)
      def __add_application_signalling_descriptor(self):

          # Dictionary indexed by application_type
          self.AIT_subtables = {}
          for idx in range(len(self.descriptor_data)/3):
              (b1, b2, b3, ) = struct.unpack_from("3B", self.descriptor_data, 3*idx)
              application_type   = (b1 & 0x00ff)<<8 | b2
              AIT_version_number = b3 & 0x1f
              self.AIT_subtables[application_type] = AIT_version_number

          return True

      # Tag 0x79. S2 Satellite delivery system descriptor: Section 6.2.13.3 ETSI EN.300.468
      def __add_S2_satellite_delivery_system_descriptor(self):
          (b1, ) = struct.unpack_from("B", self.descriptor_data, 0)
          self.scrambling_sequence_selector      = ((b1 & 0x80) != 0x00)
          self.multiple_input_stream_flag        = ((b1 & 0x40) != 0x00)
          self.backwards_compatibility_indicator = ((b1 & 0x20) != 0x00)
          pointer_data = 1

          if self.scrambling_sequence_selector:
             (b1, b2, b3, ) = struct.unpack_from("3B", self.descriptor_data, pointer_data)
             self.scrambling_sequence_index  = (b1 & 0x03)<<16 | b2<<8 | b3
             pointer_data += 3

          if self.multiple_input_stream_flag:
             (b1, ) = struct.unpack_from("B", self.descriptor_data, pointer_data)
             self.input_stream_identifier = b1


      def add_data(self, bindata):
          if len(bindata) < 2: return False

          # DVB descriptors are user-defined H222 descriptors (see h222_desc.py)
          if not h222_desc.H222_descriptor.add_data(self, bindata):
             return False

          # Detect descriptor type as ETSI EN.300.468 Table 12
          if self.descriptor_tag == 0x40:
             return self.__add_network_name_descriptor()
          elif self.descriptor_tag == 0x41:
             return self.__add_service_list_descriptor()
          elif self.descriptor_tag == 0x42:
             return self.__add_stuffing_descriptor()
          elif self.descriptor_tag == 0x43:
             return self.__add_satellite_delivery_system_descriptor()
          elif self.descriptor_tag == 0x44:
             return self.__add_cable_delivery_system_descriptor()
          elif self.descriptor_tag == 0x45:
             return self.__add_VBI_data_descriptor()
          elif self.descriptor_tag == 0x46:
             return self.__add_VBI_teletext_descriptor()
          elif self.descriptor_tag == 0x47:
             return self.__add_bouquet_name_descriptor()
          elif self.descriptor_tag == 0x48:
             return self.__add_service_descriptor()
          elif self.descriptor_tag == 0x49:
             return self.__add_country_availability_descriptor()
          elif self.descriptor_tag == 0x4a:
             return self.__add_linkage_descriptor()
          elif self.descriptor_tag == 0x4b:
             return self.__add_NVOD_reference_descriptor()
          elif self.descriptor_tag == 0x4c:
             return self.__add_time_shifted_service_descriptor()
          elif self.descriptor_tag == 0x4d:
             return self.__add_short_event_descriptor()
          elif self.descriptor_tag == 0x4e:
             return self.__add_extended_event_descriptor()
          elif self.descriptor_tag == 0x4f:
             return self.__add_time_shifted_event_descriptor()
          elif self.descriptor_tag == 0x50:
             return self.__add_component_descriptor()
          elif self.descriptor_tag == 0x52:
             return self.__add_stream_identifier_descriptor()
          elif self.descriptor_tag == 0x54:
             return self.__add_content_descriptor()
          elif self.descriptor_tag == 0x55:
             return self.__add_parental_rating_descriptor()
          elif self.descriptor_tag == 0x56:
             return self.__add_teletext_descriptor()
          elif self.descriptor_tag == 0x58:
             return self.__add_local_time_offset_descriptor()
          elif self.descriptor_tag == 0x59:
             return self.__add_subtitling_descriptor()
          elif self.descriptor_tag == 0x5a:
             return self.__add_terrestrial_delivery_system_descriptor()
          elif self.descriptor_tag == 0x5f:
             return self.__add_private_data_specifier_descriptor()
          elif self.descriptor_tag == 0x61:
             return self.__add_short_smoothing_buffer_descriptor()
          elif self.descriptor_tag == 0x64:
             return self.__add_data_broadcast_descriptor()
          elif self.descriptor_tag == 0x66:
             return self.__add_data_broadcast_id_descriptor()
          elif self.descriptor_tag == 0x69:
             return self.__add_PDC_descriptor()
          elif self.descriptor_tag == 0x6f:
             return self.__add_application_signalling_descriptor()
          elif self.descriptor_tag == 0x79:
             return self.__add_S2_satellite_delivery_system_descriptor()
          elif self.descriptor_tag in range(0x80, 0xff):
             # DVB user-defined descriptor
             return True
          elif self.descriptor_tag in [0x7d, 0x7e, 0xff, ]:
             subtable_str = "(0x%x, 0x%x, 0x%x)" % self.subtable_key
             _debug_("Invalid DVB descriptor tag. Tag 0x%x. Table ID 0x%x Subtable key %s" % (self.descriptor_tag, self.table_id, subtable_str), DWARNING)
             return False
          elif self.descriptor_tag in range(0x40, 0x100):
             subtable_str = "(0x%x, 0x%x, 0x%x)" % self.subtable_key
             _debug_("DVB descriptor not implemented yet. Tag 0x%x. Table ID 0x%x Subtable key %s" % (self.descriptor_tag, self.table_id, subtable_str), DWARNING)

          return True
