# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------
#
import struct
import time
import binascii
import h222_ts
import h222_psi
import dvb_tva
import dvb_desc
import dvb_util


# Packet id as ETSI EN.300.468 Section 5.1.3
NIT_PACKET_ID  = 0x10
ST_PACKET_ID_0 = 0x10
BAT_PACKET_ID  = 0x11
SDT_PACKET_ID  = 0x11
EIT_PACKET_ID  = 0x12
RST_PACKET_ID  = 0x13
TDT_PACKET_ID  = 0x14
TOT_PACKET_ID  = 0x14

# Table id as ETSI EN.300.468 Section 5.1.3
ACTUAL_TS_NIT_TABLE_ID            = 0x40
OTHER_TS_NIT_TABLE_ID             = 0x41
ACTUAL_TS_SDT_TABLE_ID            = 0x42
OTHER_TS_SDT_TABLE_ID             = 0x46
BAT_TABLE_ID                      = 0x4a
ACTUAL_TS_PF_EIT_TABLE_ID         = 0x4e
OTHER_TS_PF_EIT_TABLE_ID          = 0x4f
ACTUAL_TS_SCHEDULE_EIT_TABLE_ID_0 = 0x50
OTHER_TS_SCHEDULE_EIT_TABLE_ID_0  = 0x60
TDT_TABLE_ID                      = 0x70
RST_TABLE_ID                      = 0x71
ST_TABLE_ID                       = 0x72
TOT_TABLE_ID                      = 0x73


# Syntax NIT/BAT as ETSI EN.300.468 Sections 5.2.1 and 5.2.2
class DVB_TS_NIT_BAT_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)

          self.descriptors = []

          # Dictionary indexed by transport_stream_id
          self.other_descriptors = {}
          return

      def __process_NIT_BAT_section (self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]
          section_len  = len (section_data)
          pointer_data = 0

          if self.table_id == BAT_TABLE_ID:
             self.bouquet_id = self.table_id_extension
          else:
             self.network_id = self.table_id_extension

          if section_len < 2: return False
          (b1, b2) = struct.unpack_from("2B", section_data, 0)
          nit_bat_descriptors_length = ((b1 & 0x000f) << 8) | b2
          pointer_data += 2

          # Extract network/bouquet descriptors
          pointer_desc=0
          while pointer_desc < nit_bat_descriptors_length:
            descriptor = dvb_desc.DVB_descriptor(self.table_id)
            if not descriptor.add_data(section_data[pointer_data:]):
               _debug_("__process_NIT_BAT_section: Invalid descriptor", DERROR)
               return False
            self.descriptors.append(descriptor)
            pointer_desc += 2+descriptor.descriptor_length
            pointer_data += 2+descriptor.descriptor_length

          if section_len < pointer_data+2: return False
          (b3, b4) = struct.unpack_from("2B", section_data, pointer_data)
          transport_stream_loop_length = ((b3 & 0x000f) << 8) | b4
          pointer_data += 2
          if section_len < pointer_data + transport_stream_loop_length:
             return False

          while pointer_data < section_len:
            (b5, b6, b7, b8, b9, b10) = struct.unpack_from("6B", section_data, pointer_data)
            transport_stream_id          = ((b5 & 0x00ff) << 8) | b6
            original_network_id          = ((b7 & 0x00ff) << 8) | b8
            transport_descriptors_length = ((b9 & 0x000f) << 8) | b10
            pointer_data += 6

            # Extract transport stream descriptors
            descriptors  = []
            pointer_desc = 0
            while pointer_desc < transport_descriptors_length:
              descriptor = dvb_desc.DVB_descriptor(self.table_id)
              if not descriptor.add_data(section_data[pointer_data:]):
                 _debug_("__process_NIT_BAT_section: Invalid descriptor", DERROR)
                 return False
              descriptors.append(descriptor)
              pointer_desc += 2+descriptor.descriptor_length
              pointer_data += 2+descriptor.descriptor_length
            self.other_descriptors[transport_stream_id]= (original_network_id, descriptors)

          return True

      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable NIT-BAT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              if isec not in self.section_data: return False
              if not self.__process_NIT_BAT_section(self.section_data[isec]):
                 return False
          return True


class DVB_TS_service():
      def __init__(self, service_id):
          self.service_id  = service_id
          self.descriptors = []
          # See DVB_TS_SDT_version.__process_SDT_section() for
          # other members of this class
          return

# Syntax SDT as ETSI EN.300.468 Sections 5.2.3
class DVB_TS_SDT_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)

          # Dictionary indexed by service_id
          self.services = {}
          return

      def __process_SDT_section (self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]
          section_len  = len (section_data)
          pointer_data = 0

          self.transport_stream_id = self.table_id_extension

          if section_len < 3: return False
          (b1, b2, b3) = struct.unpack_from("3B", section_data, 0)
          pointer_data += 3
          self.original_network_id = ((b1 & 0x00ff) << 8) | b2

          while pointer_data < section_len:
            (b4, b5, b6, b7, b8, ) = struct.unpack_from("5B", section_data, pointer_data)
            pointer_data += 5
            service_id = ((b4 & 0x00ff) << 8) | b5
            service    = DVB_TS_service(service_id)

            # Semantics of "running_status": ETSI EN.300.468 Table 6
            service.EIT_schedule_flag          = (b6 & 0x02) != 0x00
            service.EIT_present_following_flag = (b6 & 0x01) != 0x00
            service.running_status             = (b7 & 0xe0) >> 5 
            service.free_CA_mode               = (b7 & 0x10) != 0x00
            descriptors_loop_length            = ((b7 & 0x000f) << 8) | b8

            pointer_desc = 0
            while pointer_desc < descriptors_loop_length:
              descriptor = dvb_desc.DVB_descriptor(self.table_id)
              if not descriptor.add_data(section_data[pointer_data:]):
                 _debug_("__process_SDT_section: Invalid descriptor", DERROR)
                 return False
              service.descriptors.append(descriptor)
              pointer_desc += 2+descriptor.descriptor_length
              pointer_data += 2+descriptor.descriptor_length

            self.services[service_id] = service

          return True

      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable SDT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              if isec not in self.section_data: return False
              if not self.__process_SDT_section(self.section_data[isec]):
                 return False
          return True


class DVB_TS_event():
      def __init__(self, event_id):
          self.event_id    = event_id
          self.descriptors = []
          # See DVB_TS_EIT_version.__process_SDT_section() for
          # other members of this class
          return

# Syntax EIT as ETSI EN.300.468 Sections 5.2.3
class DVB_TS_EIT_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)

          # Dictionary indexed by service_id
          self.events = {}
          return


      def __process_EIT_section (self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]
          section_len  = len(section_data)

          self.service_id = self.table_id_extension

          (b1, b2, b3, b4, b5, b6) = struct.unpack_from("6B", section_data, 0)
          pointer_data = 6
          self.transport_stream_id    = ((b1 & 0x00ff) << 8) | b2
          self.original_network_id    = ((b3 & 0x00ff) << 8) | b4
          segment_last_section_number = b5
          last_table_id               = b6

          while pointer_data < section_len:
            (b7, b8, ) = struct.unpack_from("2B", section_data, pointer_data)
            pointer_data += 2
            event_id = ((b7 & 0x00ff) << 8) | b8
            event    = DVB_TS_event(event_id)

            # Start UTC Time for the event. If NVOD, Start UTC Time shall be -1
            event.start_utc_time = dvb_util.parse_utc_time(section_data[pointer_data:])
            pointer_data += 5
            if not event.start_utc_time: return False

            (bcd1, bcd2, bcd3, ) = struct.unpack_from("3B", section_data, pointer_data)
            pointer_data += 3
            event.duration_seconds  =       10*((bcd3 & 0xf0)>>4) + (bcd3 & 0x0f)
            event.duration_seconds +=   60*(10*((bcd2 & 0xf0)>>4) + (bcd2 & 0x0f))
            event.duration_seconds += 3600*(10*((bcd1 & 0xf0)>>4) + (bcd1 & 0x0f))

            (b9, b10) = struct.unpack_from("2B", section_data, pointer_data)
            pointer_data += 2
            # Semantics of "running_status": ETSI EN.300.468 Table 6
            event.running_status    =  (b9 & 0xe0) >> 5
            event.free_CA_mode      =  (b9 & 0x10) != 0x00
            descriptors_loop_length = ((b9 & 0x000f) << 8) | b10

            pointer_desc = 0
            while pointer_desc < descriptors_loop_length:
              descriptor = dvb_desc.DVB_descriptor(self.table_id, self.subtable_key)
              if not descriptor.add_data(section_data[pointer_data:]):
                 _debug_("__process_EIT_section: Invalid descriptor", DERROR)
                 return False
              event.descriptors.append(descriptor)
              pointer_desc += 2+descriptor.descriptor_length
              pointer_data += 2+descriptor.descriptor_length

            self.events[event_id] = event

          # while pointer_data < section_len:

          return True


      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable EIT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              # Section numbering gaps are allowed in this table
              if isec not in self.section_data: continue
              if not self.__process_EIT_section(self.section_data[isec]):
                 return False

          return True


# Syntax TDT as ETSI EN.300.468 Sections 5.2.5
# This table is setup using a single section without syntax. Therefore
# it has not got versions and it will be rewritten (updated) for any
# new incoming data
class DVB_TS_TDT_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)
          return

      def __process_TDT_section (self, __section_data):
          # This section has no syntax. Remove header
          section_data = __section_data[3:]

          self.utc_time = dvb_util.parse_utc_time(section_data)
          if not self.utc_time: return False

          _debug_("TDT, UTC time %s" % time.strftime("%a, %d %b %Y %H:%M:%S", time.gmtime(self.utc_time)), DINFO)
          return True

      def process_subtable(self):
          # This table allows only one section
          if self.last_section_number != 0: return False
          if 0 not in self.section_data:    return False
          return self.__process_TDT_section(self.section_data[0])


# Syntax TOT as ETSI EN.300.468 Sections 5.2.6
# This table is setup using a single section without syntax. Therefore
# it has not got versions and it will be rewritten (updated) for any
# new incoming data
class DVB_TS_TOT_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)

          self.descriptors = []
          return

      def __process_TOT_section (self, __section_data):
          # This section has no syntax. Remove header
          section_data = __section_data[3:]

          self.utc_time = dvb_util.parse_utc_time(section_data)
          pointer_data = 5
          if not self.utc_time: return False

          (b1, b2, ) = struct.unpack_from("2B", section_data, pointer_data)
          pointer_data += 2
          descriptors_loop_length = ((b1 & 0x000f) << 8) | b2

          self.descriptors = []
          pointer_desc     = 0
          while pointer_desc < descriptors_loop_length:
            descriptor = dvb_desc.DVB_descriptor(self.table_id)
            if not descriptor.add_data(section_data[pointer_data:]):
                 _debug_("__process_TOT_section: Invalid descriptor", DERROR)
                 return False
            self.descriptors.append(descriptor)
            pointer_desc += 2+descriptor.descriptor_length
            pointer_data += 2+descriptor.descriptor_length

          _debug_("TOT, UTC time %s" % time.strftime("%a, %d %b %Y %H:%M:%S", time.gmtime(self.utc_time)), DINFO)
          return True

      def process_subtable(self):
          # This table allows only one section
          if self.last_section_number != 0: return False
          if 0 not in self.section_data:    return False
          if h222_psi.crc32_be(self.section_data[0]):
             _debug_("TOT CRC error", DWARNING)
             return False
          return self.__process_TOT_section(self.section_data[0])


# Syntax RST as ETSI EN.300.468 Sections 5.2.7
# This table is setup using a single section without syntax. Therefore
# it has not got versions and it will be rewritten (updated) for any
# new incoming data
class DVB_TS_RST_version (h222_psi.H222_TS_PSI_version):
      def __init__(self, table_id):
          h222_psi.H222_TS_PSI_version.__init__(self, table_id)
          
          # Dictionary indexed by tuples
          # (original_network_id, transport_stream_id, service_id, event_id)
          self.event_status = {}
          return

      def __process_RST_section (self, __section_data):
          # This section has no syntax. Remove header
          section_data = __section_data[3:]

          self.event_status = {}
          for idx in range(len(section_data)/9):
            (b1, b2, ) = struct.unpack_from("2B", section_data, 9*idx)
            transport_stream_id = ((b1 & 0x00ff) << 8) | b2
            (b3, b4, ) = struct.unpack_from("2B", section_data, 9*idx+2)
            original_network_id = ((b3 & 0x00ff) << 8) | b4
            (b5, b6, ) = struct.unpack_from("2B", section_data, 9*idx+4)
            service_id = ((b5 & 0x00ff) << 8) | b6
            (b7, b8, ) = struct.unpack_from("2B", section_data, 9*idx+6)
            event_id = ((b7 & 0x00ff) << 8) | b8

            # Semantics of running_status as ETSI EN.300.468 Table 6
            (b9, ) = struct.unpack_from("B", section_data, 9*idx+8)
            running_status = b9 & 0x07

            self.event_status[(original_network_id, transport_stream_id, service_id, event_id)] = running_status
 
          return True

      def process_subtable(self):
          # This table allows only one section
          if self.last_section_number != 0: return False
          if 0 not in self.section_data:    return False
          return self.__process_RST_section(self.section_data[0])


class DVB_TS_NIT ():
      def __init__(self, psi):
          array_tids = [ACTUAL_TS_NIT_TABLE_ID, OTHER_TS_NIT_TABLE_ID,]
          array_pids = [NIT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_NIT_BAT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          network_id = section.table_id_extension
          return (network_id, 0, 0)


class DVB_TS_BAT ():
      def __init__(self, psi):
          array_tids = [BAT_TABLE_ID,]
          array_pids = [BAT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_NIT_BAT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          bouquet_id = section.table_id_extension
          return (bouquet_id, 0, 0)


class DVB_TS_SDT ():
      def __init__(self, psi):
          array_tids = [ACTUAL_TS_SDT_TABLE_ID, OTHER_TS_SDT_TABLE_ID,]
          array_pids = [SDT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_SDT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          if len(section.data) < 8+2: return None

          transport_stream_id = section.table_id_extension
          (b1, b2) = struct.unpack_from("2B", section.data, 8)
          original_network_id = ((b1 & 0x00ff) << 8) | b2

          return (original_network_id, transport_stream_id, 0)


class DVB_TS_EIT ():
      def __init__(self, psi):
          array_tids  = [ACTUAL_TS_PF_EIT_TABLE_ID, OTHER_TS_PF_EIT_TABLE_ID,]
          array_tids += range(ACTUAL_TS_SCHEDULE_EIT_TABLE_ID_0, 0x10+ACTUAL_TS_SCHEDULE_EIT_TABLE_ID_0)
          array_tids += range( OTHER_TS_SCHEDULE_EIT_TABLE_ID_0, 0x10+ OTHER_TS_SCHEDULE_EIT_TABLE_ID_0)
          array_pids  = [EIT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_EIT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          if len(section.data) < 8+4: return None

          service_id = section.table_id_extension
          (b1, b2, b3, b4) = struct.unpack_from("4B", section.data, 8)
          transport_stream_id = ((b1 & 0x00ff) << 8) | b2
          original_network_id = ((b3 & 0x00ff) << 8) | b4

          return (original_network_id, transport_stream_id, service_id)


class DVB_TS_TDT ():
      def __init__(self, psi):
          array_tids = [TDT_TABLE_ID, ]
          array_pids = [TDT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_TDT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          return h222_psi.DEFAULT_SUBTABLE_KEY


class DVB_TS_TOT ():
      def __init__(self, psi):
          array_tids = [TOT_TABLE_ID, ]
          array_pids = [TOT_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_TOT_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          return h222_psi.DEFAULT_SUBTABLE_KEY


class DVB_TS_RST ():
      def __init__(self, psi):
          array_tids = [RST_TABLE_ID, ]
          array_pids = [RST_PACKET_ID,]
          psi.define_psi_tables (array_tids, array_pids, DVB_TS_RST_version, self.get_subtable_key)
          return

      def get_subtable_key(table, section):
          return h222_psi.DEFAULT_SUBTABLE_KEY


class DVB_TS_ST ():
      def __init__(self, psi):
          array_tids = [ST_TABLE_ID, ]
          array_pids = range(ST_PACKET_ID_0, ST_PACKET_ID_0+5)
          psi.define_psi_tables (array_tids, array_pids, None, None)
          return


class DVB_TS (h222_ts.H222_TS):
      def __init__(self):

          h222_ts.H222_TS.__init__(self, dvb_desc.DVB_descriptor)

          self.nit = DVB_TS_NIT (self.psi)
          self.bat = DVB_TS_BAT (self.psi)
          self.sdt = DVB_TS_SDT (self.psi)
          self.eit = DVB_TS_EIT (self.psi)
          self.tdt = DVB_TS_TDT (self.psi)
          self.tot = DVB_TS_TOT (self.psi)
          self.rst = DVB_TS_RST (self.psi)
          self.st  = DVB_TS_ST  (self.psi)
          self.tva = dvb_tva.DVB_TVA (self.psi)

          # ToDo. SMI related tables: DIT, SIT
          return

