# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------
#
import struct

class H222_descriptor ():
      def __init__(self, table_id, subtable_key=(0,0,0)):
          self.table_id     = table_id
          self.subtable_key = subtable_key
          return

      # Tag #2. Video descriptor: Section 2.6.3 ITU-T H.222.0
      def __add_video_descriptor(self, bindata):

          (b1, ) = struct.unpack_from("B", bindata, 2)
          self.multiple_frame_rate_flag   = ((b1 & 0x80) != 0x00)
          self.frame_rate_code            = (b1 & 0x78)>>3
          self.MPEG_1_only_flag           = ((b1 & 0x04) != 0x00)
          self.constrained_parameter_flag = ((b1 & 0x02) != 0x00)
          self.still_picture_flag         = ((b1 & 0x01) != 0x00)

          if not self.MPEG_1_only_flag:
             if len(bindata) < 5: return False
             (b2, b3, ) = struct.unpack_from("2B", bindata, 3)
             self.profile_and_level_indication = b2
             self.chroma_format                = (b3 & 0xc0)>>6
             self.frame_rate_extension_flag    = ((b3 & 0x20) != 0x00)

          return True


      # Tag #3. Audio descriptor: Section 2.6.4 ITU-T H.222.0
      def __add_audio_descriptor(self, bindata):

          (b1, ) = struct.unpack_from("B", bindata, 2)
          self.free_format_flag              = ((b1 & 0x80) != 0x00)
          self.ID                            = ((b1 & 0x40) != 0x00)
          self.layer                         = (b1 & 0x30)>>4
          self.variable_rate_audio_indicator = ((b1 & 0x08) != 0x00)

          return True


      # Tag #6. Data stream alignment descriptor: Section 2.6.10 ITU-T H.222.0
      def __add_data_stream_alignment_descriptor(self, bindata):

          (b1, ) = struct.unpack_from("B", bindata, 2)
          self.alignment_type = b1
          return True


      # Tag #9. CA descriptor: Section 2.6.16 ITU-T H.222.0
      def __add_CA_descriptor(self, bindata):

          (b1, b2, b3, b4) = struct.unpack_from("4B", bindata, 2)
          self.CA_system_ID = (b1 & 0x00ff)<<8 | b2
          self.CA_PID       = (b3 & 0x001f)<<8 | b4

          self.private_data = bindata[6:6+self.descriptor_length]
          return True


      # Tag #10. ISO639 language descriptor: Section 2.6.17 ITU-T H.222.0
      def __add_ISO_639_language_descriptor(self, bindata):

          self.languages = []
          pointer_data   = 0
          while pointer_data < self.descriptor_length:
            available_bytes = self.descriptor_length - pointer_data
            if available_bytes < 4: return False

            (ISO_639_language_code, audio_type) = struct.unpack_from("3sB", bindata, 2+pointer_data)
            self.languages.append((ISO_639_language_code, audio_type))
            pointer_data += 4

          return True


      # Tag #12. Multiplex buffer utilization descriptor: Section 2.6.22 ITU-T H.222.0
      def __add_multiplex_buffer_utilization_descriptor(self, bindata):

          (b1, b2, b3, b4, ) = struct.unpack_from("4B", bindata, 2)
          self.bound_valid_flag = (b1 & 0x80) != 0x00
          if self.bound_valid_flag:
             self.LTW_offset_lower_bound = (b1 & 0x7f) << 8 | b2
             self.LTW_offset_upper_bound = (b3 & 0x7f) << 8 | b4

          return True


      # Tag #14. Maximum bitrate descriptor: Section 2.6.26 ITU-T H.222.0
      def __add_maximum_bitrate_descriptor(self, bindata):

          (b1, b2, b3, ) = struct.unpack_from("3B", bindata, 2)
          self.maximum_bitrate = (b1 & 0x3f) << 16 | b2 << 8 | b3
          return True


      # Tag #17. STD descriptor: Section 2.6.33 ITU-T H.222.0
      def __add_STD_descriptor(self, bindata):

          (b1, ) = struct.unpack_from("B", bindata, 2)
          self.leak_valid_flag = ((b1 & 0x01) != 0x00)
          return True


      # ISO 13818-6 descriptors
      def __add_ISO_13818_6_descriptor(self, bindata):
          # ToDo, anybody has this standard?
          return True


      # Tag #32. External ES ID descriptor: Section 2.6.46 ITU-T H.222.0
      def __add_external_ES_ID_descriptor(self, bindata):

          (b1, b2, ) = struct.unpack_from("2B", bindata, 2)
          self.external_ES_ID = (b1 & 0xff) << 8 | b2
          return True


      def add_data(self, bindata):
          if len(bindata) < 2:
             _debug_("H222_descriptor. Table ID 0x%x. Len %i < 2" % (self.table_id, len(bindata)), DERROR)
             return False

          (b1, b2, ) = struct.unpack_from("2B", bindata, 0)
          self.descriptor_tag    = b1
          self.descriptor_length = b2
          if 2+self.descriptor_length > len(bindata):
             _debug_("H222_descriptor 0x%x. Table ID 0x%x. No enough data %i" % (self.descriptor_tag, self.table_id, 2+self.descriptor_length-len(bindata)), DERROR)
             return False

          # Detect descriptor type as ITU-T H.222.0, ISO 13818-1, Table 2-39
          if self.descriptor_tag == 2:
             return self.__add_video_descriptor(bindata)
          elif self.descriptor_tag == 3:
             return self.__add_audio_descriptor(bindata)
          elif self.descriptor_tag == 6:
             return self.__add_data_stream_alignment_descriptor(bindata)
          elif self.descriptor_tag == 9:
             return self.__add_CA_descriptor(bindata)
          elif self.descriptor_tag == 10:
             return self.__add_ISO_639_language_descriptor(bindata)
          elif self.descriptor_tag == 12:
             return self.__add_multiplex_buffer_utilization_descriptor(bindata)
          elif self.descriptor_tag == 14:
             return self.__add_maximum_bitrate_descriptor(bindata)
          elif self.descriptor_tag == 17:
             return self.__add_STD_descriptor(bindata)
          elif self.descriptor_tag in range (19, 27):
             return self.__add_ISO_13818_6_descriptor(bindata)
          elif self.descriptor_tag == 32:
             return self.__add_external_ES_ID_descriptor(bindata)
          elif self.descriptor_tag in range (2, 36):
             subtable_str = "(0x%x, 0x%x, 0x%x)" % self.subtable_key
             _debug_ ("H222_descriptor. Tag %i not implemented yet. Table ID 0x%x, Subtable key %s" % (self.descriptor_tag, self.table_id, subtable_str), DWARNING)
             return True
          elif self.descriptor_tag in range (64, 256):
             # User private descriptor
             self.descriptor_data = bindata[2:2+self.descriptor_length]
             return True

          subtable_str = "(0x%x, 0x%x, 0x%x)" % self.subtable_key
          _debug_("H222_descriptor. Invalid descriptor tag %i. Table ID 0x%x Subtable key %s" % (self.descriptor_tag, self.table_id, subtable_str), DWARNING)
          return False

