# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------
#
import binascii
import struct
import time

import h222_desc

bit32 = struct.calcsize('L') == struct.calcsize('I')

# Packet id as ITU-T H.222.0 Table 2-23
PAT_PACKET_ID  = 0x00
CAT_PACKET_ID  = 0x01
TSDT_PACKET_ID = 0x02

# Table id as ITU-T H.222.0 Table 2-26
PAT_TABLE_ID  = 0x00
CAT_TABLE_ID  = 0x01
PMT_TABLE_ID  = 0x02
TSDT_TABLE_ID = 0x03

DEFAULT_SUBTABLE_KEY = (0, 0, 0)

# Coded as /usr/src/linux-2.6.x/lib/crc32.c
def crc32_be(buf):
    crc = 0xffffffff
    for i in range(len(buf)):
        (b,) = struct.unpack_from("B", buf, i)
        crc ^= int(b)<<24
        for j in range(8):
            mask = ((crc & 0x80000000) != 0) and 0x04c11db7 or 0x00L
            crc = ((crc << 1) & 0xffffffff) ^ mask
    return crc


class H222_TS_PSI_version ():
      def __init__(self, table_id):
          self.table_id = table_id

          # Dictionary indexed by section_number
          self.section_data = {}

          # Meanwhile all sections of a table version have not been
          # received, last_seen shall be 0.
          self.last_seen = 0.0
          return


# Indeed, "subtable" is not an H222 object, it is rather an
# object used by upper layers as DVB. A pure-H222 table has only one
# subtable at the DEFAULT_SUBTABLE_KEY. However the program maps in the
# PMT table are well managed as subtables
class H222_TS_PSI_subtable ():
      def __init__(self, table_id, subtable_key, table_version_class):
          self.table_id            = table_id
          self.subtable_key        = subtable_key
          self.table_version_class = table_version_class

          # Dictionary indexed by version_number
          self.versions            = {}
          self.current_version     = -1
          self.next_version        = -1
          return

      # Return True only if this section completes a new and valid subtable
      def add_section(self, section):

          # If no "table_version_class" is specified, return True. Typically
          # stuffing subtables has no "table_version_class" as they do not
          # require action
          if not self.table_version_class: return True

          # If needed, create a new version according to the
          # "table_version_class"
          if section.version_number not in self.versions:
             self.versions[section.version_number] = self.table_version_class(self.table_id)
          version                  = self.versions[section.version_number]
          version.descriptor_class = self.descriptor_class
          version.subtable_key     = self.subtable_key

          # If this version has been already parsed, return (if this
          # table allows version, i.e. it has got syntax)
          if version.last_seen > 0:
             version.last_seen = time.time()
             if section.section_syntax_indicator: return False

          # This table version is not complete yet, copy section data
          version.section_data[section.section_number] = section.data
              
          version.section_syntax_indicator = section.section_syntax_indicator
          version.table_id_extension       = section.table_id_extension
          version.current_next_indicator   = section.current_next_indicator
          version.last_section_number      = section.last_section_number

          # Check if this section is the last one.
          # Note, some table types allow gaps in numbering section (for
          # instance, DVB EIT tables)
          if section.section_number == section.last_section_number:
             version.last_seen = time.time()
          else:
             return False

          # Now, we know this table version is complete. Parse for validity
          if not version.process_subtable():
             subtable_key_str = "(0x%x, 0x%x, 0x%x)" % self.subtable_key
             _debug_("Error processing subtable. Discarding table version #%i. Table id 0x%x, Subtable key %s" % (section.version_number, section.table_id, subtable_key_str), DWARNING)
             del self.versions[section.version_number]
             return False

          # Version tracking and clean-up. Note the wrap-around from 31 to 0
          for (iver, lver) in self.versions.items():
             if lver.last_seen == 0.0: continue
             if not lver.current_next_indicator:
                if (iver > self.next_version) or ((iver == 0) and (self.next_version == 31)):
                   self.next_version = iver
             else:
                if (iver > self.current_version) or ((iver == 0) and (self.current_version == 31)):
                   self.current_version = iver
                if (iver < self.current_version) or ((iver == 31) and (self.current_version == 0)):
                   del self.versions[iver]


          return True


class H222_TS_PSI_table ():
      def __init__(self, table_id):
          self.table_id            = table_id
          self.pids                = []
          self.table_version_class = None

          # Dictionary indexed by subtable_key tuples
          self.subtables           = {}
          self.get_subtable_key    = None
          return


      def add_section(self, section):
          # If no "get_subtable_key" is specified, return True. Typically
          # stuffing tables has no "get_subtable_key" as no requiring
          # action
          if not self.get_subtable_key: return True

          subtable_key = self.get_subtable_key(section)
          if not subtable_key:
             _debug_("Table id 0x%x. Unable to get subtable key. Discarding section" % section.table_id, DWARNING)
             return False

          # If needed, create a new subtable
          if subtable_key not in self.subtables:
             self.subtables[subtable_key] = H222_TS_PSI_subtable(self.table_id, subtable_key, self.table_version_class)
          subtable                  = self.subtables[subtable_key]
          subtable.descriptor_class = self.descriptor_class

          return subtable.add_section (section)


class H222_TS_PAT_version (H222_TS_PSI_version):
      def __init__(self, table_id):
          H222_TS_PSI_version.__init__(self, table_id)

          # Dictionary of Program Maps' PIDs indexed by Program Numbers
          self.program_map_pids = {}
          return

      def __process_PAT_section(self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]

          self.transport_stream_id = self.table_id_extension

          for idx in range(len(section_data)/4):
              (b1, b2, b3, b4) = struct.unpack_from("4B", section_data, 4*idx)
              program_number = ((b1 & 0x00ff) << 8) | b2
              program_pid    = ((b3 & 0x001f) << 8) | b4
              self.program_map_pids[program_number] = program_pid

          return True

      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable PAT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              if isec not in self.section_data: return False
              if not self.__process_PAT_section(self.section_data[isec]):
                 return False
          return True


class H222_TS_CAT_version (H222_TS_PSI_version):
      def __init__(self, table_id):
          H222_TS_PSI_version.__init__(self, table_id)

          self.descriptors  = []
          return


      def __process_CAT_section(self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]

          # Extract descriptors
          pointer_desc=0
          while pointer_desc < len(section_data):
            descriptor = self.descriptor_class(self.table_id)
            if not descriptor.add_data(section_data[pointer_desc:]):
               _debug_("CAT process_CAT_section: Invalid descriptor", DERROR)
               return False
            self.descriptors.append(descriptor)
            pointer_desc += 2+descriptor.descriptor_length

          return True


      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable CAT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              if isec not in self.section_data: return False
              if not self.__process_CAT_section(self.section_data[isec]):
                 return False
          return True


class H222_TS_PMT_version (H222_TS_PSI_version):
      def __init__(self, table_id):
          H222_TS_PSI_version.__init__(self, table_id)

	  # Dictionary of elementary stream info, indexed by its PID
          self.elements    = {}
          self.descriptors = []
          return

      # PMT syntax, ITU-T H.222.0 Table 2-28
      def __process_PMT_section(self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]
          section_len  = len(section_data)

          (b1, b2, b3, b4) = struct.unpack_from("4B", section_data, 0)
          self.PCR_pid = ((b1 & 0x001f) << 8) | b2 
          program_info_length = ((b3 & 0x000f) << 8) | b4
          total_elements_info_length = section_len - program_info_length

          # Extract program descriptors
          pointer_data    = 4+program_info_length
          descriptor_data = section_data[4:pointer_data] 
          pointer_desc = 0
          while pointer_desc < len(descriptor_data):
            descriptor = self.descriptor_class(self.table_id)
            if not descriptor.add_data(descriptor_data[pointer_desc:]):
               _debug_("__process_PMT_section: Invalid program descriptor", DERROR)
               return False
            self.descriptors.append(descriptor)
            pointer_desc += 2+descriptor.descriptor_length

          while section_len-pointer_data >= 5:
              (b1, b2, b3, b4, b5) = struct.unpack_from("5B", section_data, pointer_data)
              stream_type         = b1
              elementary_pid      = ((b2 & 0x001f) << 8) | b3
              element_info_length = ((b4 & 0x000f) << 8) | b5
              pointer_data       += 5

              if section_len <= pointer_data+element_info_length: break 

              # Extract program element descriptors
              element_descriptors = []
              descriptor_data     = section_data[pointer_data:pointer_data+element_info_length]
              pointer_desc        = 0
              while pointer_desc < len(descriptor_data):
                descriptor = self.descriptor_class(self.table_id)
                if not descriptor.add_data(descriptor_data[pointer_desc:]):
                   _debug_("__process_PMT_section: Invalid program element descriptor", DERROR)
                   return False
                element_descriptors.append(descriptor)
                pointer_desc += 2+descriptor.descriptor_length
              self.elements[elementary_pid] = (stream_type, element_descriptors)

              pointer_data += element_info_length

          # while section_len-pointer_data >= 5:
          return True

      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable PMT should have syntax. Discard", DWARNING)
             return False

          if self.last_section_number != 0: return False
          if 0 not in self.section_data:    return False
          return self.__process_PMT_section(self.section_data[0])



class H222_TS_TSDT_version (H222_TS_PSI_version):
      def __init__(self, table_id):
          H222_TS_PSI_version.__init__(self, table_id)

          self.descriptors = []
          return


      def __process_TSDT_section(self, __section_data):
          # This section has syntax. Remove header and CRC
          section_data = __section_data[8:len(__section_data)-4]

          pointer_desc = 0
          while pointer_desc < len(section_data):
              descriptor = self.descriptor_class(self.table_id)
              if not descriptor.add_data(section_data[pointer_desc:]):
                 _debug_("__process_TSDT_section: Invalid program element descriptor", DERROR)
                 return False
              self.descriptors.append(descriptor)
              pointer_desc += 2+descriptor.descriptor_length

          return True


      def process_subtable(self):
          if not self.section_syntax_indicator:
             _debug_("This subtable TSDT should have syntax. Discard", DWARNING)
             return False

          for isec in range(self.last_section_number+1):
              if isec not in self.section_data: return False
              if not self.__process_TSDT_section(self.section_data[isec]):
                 return False
          return True


# As defined at ITU-T H.222.0 section 2.4.4: Tables 2-25, 2-27, 2-28, 2-30
class H222_TS_section:
      def __init__ (self, table_id):
          self.table_id = table_id

          # Eventually, member "data" shall host the complete
          # binary section
          self.data = ""

class H222_TS_PSI ():
      def __init__(self, descriptor_class=h222_desc.H222_descriptor):

          self.descriptor_class = descriptor_class

          # ITU-T H.222.0 Annex C.7: "[sections are allowed] to be mixed
          # in the TS packets of the same PID value or even in the same
          # TS packet. Note, however, that within packets of the same PID,
          # a complete section must be transmitted before the next
          # section can be started"
          #
          # Dictionary "incoming_sections" indexed by PID has
          # "H222_TS_section" objects
          self.incoming_sections = {}

          # Dictionary "tables" indexed by table_id as ITU-T H.222.0
          # Table 2-26
          self.tables = {}
          self.__define_psi_table (PAT_TABLE_ID,  [PAT_PACKET_ID,],  H222_TS_PAT_version,  self.__get_subtable_key)
          self.__define_psi_table (CAT_TABLE_ID,  [CAT_PACKET_ID,],  H222_TS_CAT_version,  self.__get_subtable_key)
          self.__define_psi_table (TSDT_TABLE_ID, [TSDT_PACKET_ID,], H222_TS_TSDT_version, self.__get_subtable_key)

          return


      def __define_psi_table (self, table_id, array_pids, table_version_class, get_subtable_key):
          if table_id in range(0x04, 0x40) or table_id == 0xff:
             return False

          if not table_id in self.tables:
             self.tables[table_id] = H222_TS_PSI_table (table_id)

          table = self.tables[table_id]
          table.pids                = array_pids
          table.table_version_class = table_version_class
          table.get_subtable_key    = get_subtable_key
          table.descriptor_class    = self.descriptor_class

          self.active_pids = self.__get_active_pids()
          return True


      def define_psi_tables (self, array_tids, array_pids, table_version_class, get_subtable_key):
          for tid in array_tids:
              if not self.__define_psi_table (tid, array_pids, table_version_class, get_subtable_key):
                 return False
          return True


      # Only manage packets included to the list of active pids.
      # Another packets are discarded.
      # See ITU-T H.222.0 Table 2-23 for PID
      def __get_active_pids (self):
          active_pids = []
 
          # Walk through all tables to gather associated packet ids
          for (table_id, table) in self.tables.items():
              for pid in table.pids:
                  active_pids.append(pid)

          return active_pids


      def pid_carries_pes (self, PID, current_next_indicator=True):
          if PMT_TABLE_ID not in self.tables: return False

          # Look PMT table for elementary streams carrying private data
          # See ITU-T H.222.0 Table 2-29
          for (subtable_key, subtable) in self.tables[PMT_TABLE_ID].subtables.items():
              if current_next_indicator:
                version_number = subtable.current_version
              else:
                version_number = subtable.next_version
              if version_number == -1: continue
              version = subtable.versions[version_number]
              if PID in version.elements:
                 if version.elements[PID].stream_type != 0x05:
                    return True

          return False


      def pid_carries_video (self, PID, current_next_indicator=True):
          if PMT_TABLE_ID not in self.tables: return False

          # Look PMT table for elementary streams carrying video
          # See ITU-T H.222.0 Table 2-29
          for (subtable_key, subtable) in self.tables[PMT_TABLE_ID].subtables.items():
              if current_next_indicator:
                version_number = subtable.current_version
              else:
                version_number = subtable.next_version
              if version_number == -1: continue
              version = subtable.versions[version_number]
              if PID in version.elements:
                 if version.elements[PID].stream_type == 0x01: return True
                 if version.elements[PID].stream_type == 0x02: return True

          return False


      def pid_carries_audio (self, PID, current_next_indicator=True):
          if PMT_TABLE_ID not in self.tables: return False

          # Look PMT table for elementary streams carrying audio
          # See ITU-T H.222.0 Table 2-29
          for (subtable_key, subtable) in self.tables[PMT_TABLE_ID].subtables.items():
              if current_next_indicator:
                version_number = subtable.current_version
              else:
                version_number = subtable.next_version
              if version_number == -1: continue
              version = subtable.versions[version_number]
              if PID in version.elements:
                 if version.elements[PID].stream_type == 0x03: return True
                 if version.elements[PID].stream_type == 0x04: return True

          return False

      def __get_subtable_key(self, section):
          if section.table_id == PMT_TABLE_ID:
             program_number = section.table_id_extension
             return (0, 0, program_number)
          return DEFAULT_SUBTABLE_KEY

      def __process_incoming_sections(self, context_packet):

          for (pid, section) in self.incoming_sections.items():
             len_data = len(section.data)

             # If this section has been not fulfilled yet, continue
             # and wait for more data for this section
             if len_data < 3: continue
             if (3 + section.section_length) > len_data: continue
          
             if section.section_syntax_indicator:
                (b4, b5, b6, b7, b8) = struct.unpack_from("5B", section.data, 3)
                section.table_id_extension     = ((b4 & 0x00ff)<<8) | b5
                section.version_number         = (b6 & 0x3e)>>1
                section.current_next_indicator = ((b6 & 0x01) != 0x00)
                section.section_number         = b7
                section.last_section_number    = b8
             else:
                section.table_id_extension     = 0
                section.version_number         = 0
                section.current_next_indicator = True
                section.section_number         = 0
                section.last_section_number    = 0

             # Table type is detected using table_id as ITU-T H.222.0 Table 2-26
             table_id = section.table_id
             if (table_id in range(0x06, 0x38)) or (table_id == 0xff):
                _debug_("PID 0x%x, Invalid table_id 0x%x, discarding section" % (context_packet.PID, table_id), DWARNING)
                del self.incoming_sections[pid]
                continue

             # Check CRC, discard section if corrupted
             if section.section_syntax_indicator:
                if crc32_be(section.data):
                   _debug_("PID 0x%x, CRC error. Table id 0x%x, Section length %i" % (context_packet.PID, table_id, len_data), DWARNING)
                   del self.incoming_sections[pid]
                   continue

             # Add this new and valid section to its corresponding table.
             if table_id in self.tables:
                table = self.tables[table_id]
                if table.add_section(section):
                   # PAT table carries the table definition for the PMT table
                   if table_id == PAT_TABLE_ID:
                      PAT_subtable        = table.subtables[DEFAULT_SUBTABLE_KEY]
                      PAT_version_number  = PAT_subtable.current_version
                      if PAT_version_number != -1:
                         PAT_version         = PAT_subtable.versions[PAT_version_number]
                         PMT_pids            = PAT_version.program_map_pids.itervalues()
                         self.__define_psi_table (PMT_TABLE_ID, PMT_pids, H222_TS_PMT_version, self.__get_subtable_key)
             else:
                _debug_("PID 0x%x, Undefined table_id 0x%x, discarding section" % (context_packet.PID, table_id), DWARNING)

             # Remove this section from the dictionary "incoming_sections"
             del self.incoming_sections[pid]


      def add_packetized_sections(self, payload_packet, context_packet):

          pid = context_packet.PID
          if pid not in self.active_pids: return True
          
          payload_len   = len(payload_packet)
          pointer_field = 0

          # Semantics of "pointer_field" are in section 2.4.4.2 ITU-T
          # H.222.0
          if context_packet.payload_unit_start_indicator:
             (pointer_field,) = struct.unpack_from("B", payload_packet, 0)
             pointer_field += 1

             # Remove garbage data, which should NOT be there; likely, due to
             # missing or discarded packets by lower layers
             if pid in self.incoming_sections:
                del self.incoming_sections[pid]
          else:
             # If this packet does not start a section, there should be
             # an "incoming_section" for this PID. Else, ignore this packet
             if not pid in self.incoming_sections: return True

          # Go through the payload to extract one or more sections
          # Add data to an existing incoming_sections or create new ones
          # Keep in mind, within TS packets, sections are fragmented at
          # any point
          while pointer_field < payload_len:
            available_bytes = payload_len - pointer_field

            # Starting a new section
            if pid not in self.incoming_sections:
               (table_id, ) = struct.unpack_from("B", payload_packet, pointer_field)

               # Check for stuffing bytes (i.e. 0xff). If detected, give
               # up any more processing because all following bytes until the
               # end of the packet shall be stuffing bytes
               if table_id == 0xff:
                  pointer_field = payload_len 
                  continue

               section = H222_TS_section(table_id)
               self.incoming_sections[pid] = section

               section.data  += payload_packet[pointer_field]
               pointer_field += 1
               continue

            # Adding data to an existing, but incomplete, section 
            section = self.incoming_sections[pid]

            if not hasattr (section, "reserved_psi"):
               (b2, ) = struct.unpack_from("B", payload_packet, pointer_field)
               section.section_syntax_indicator = ((b2 & 0x80) != 0x00)
               section.private_indicator        = ((b2 & 0x40) != 0x00)
               section.reserved_psi             = ((b2 & 0x30) >> 4)
               section.data  += payload_packet[pointer_field]
               pointer_field += 1
               continue

            if not hasattr (section, "section_length"):
               (b2, ) = struct.unpack_from("B", section.data, 1)
               (b3, ) = struct.unpack_from("B", payload_packet, pointer_field)
               section.section_length = (((b2 & 0x000f) << 8) | b3)
               section.data  += payload_packet[pointer_field]
               pointer_field += 1
               continue

            # Available section's bytes in this packet
            rest_section_bytes = section.section_length - (len(section.data)-3)
            if rest_section_bytes > available_bytes:
               rest_section_bytes = available_bytes

            section.data  += payload_packet[pointer_field:pointer_field+rest_section_bytes]
            pointer_field += rest_section_bytes

            # When return, all complete sections will have been removed from the
            # dictionary "incoming_sections"
            self.__process_incoming_sections(context_packet)

          # while pointer_field < payload_len:

          return True

