# -*- coding: iso-8859-1 -*-
# -----------------------------------------------------------------------
#
# ISO/IEC 13818-1 is identical as ITU Rec.H.222.0 (2000E)

import struct
import time
import binascii

import h222_psi
import h222_pes

bit32 = struct.calcsize('L') == struct.calcsize('I')

class H222_TS_packet_context ():
      def __init__(self, frequency=0):
          self.frequency = frequency
          self.toa       = time.time()

          # Default value, see ITU-T H.222.0 section 2.4.3.5 
          self.discontinuity_indicator = False
          return

class H222_TS ():
      def __init__(self, descriptor_class=None):
          # Program Specific Information of the TS stream
          # See ITU-T H.222.0 section 2.4.4
          self.psi = h222_psi.H222_TS_PSI(descriptor_class)

          # Packetized Elemtary Streams of the TS stream
          # See ITU-T H.222.0 section 2.4.3.6
          self.pes = h222_pes.H222_TS_PES()

          # Dictionary indexed by PID
          self.last_valid_counter = {}

          self.descriptor_class = descriptor_class
          return

      def get_active_pids(self):
          active_pids = self.psi.active_pids
          for pid in self.pes.get_active_pids():
              active_pids.append(pid)
          return active_pids

      # As defined at ITU-T H.222.0 section 2.4.3.5
      def process_adaptation_field_extension (self, binfield, context):
          pointer_data = 0

          (b1,) = struct.unpack_from("B", binfield, 0)
          ltw_flag             = ((b1 & 0x80) != 0x00)
          piecewise_rate_flag  = ((b1 & 0x40) != 0x00)
          seamless_splice_flag = ((b1 & 0x20) != 0x00)
          pointer_data += 1

          if ltw_flag:
             (b1,b2) = struct.unpack_from("2B", binfield, pointer_data)
             pointer_data += 2
             ltw_valid_flag = ((b1 & 0x80) != 0x00)
             if ltw_valid_flag:
                context.ltw_offset = ((b1 & 0x007f) << 8) | b2

          if piecewise_rate_flag:
             (b1,b2,b3) = struct.unpack_from("3B", binfield, pointer_data)
             pointer_data += 3
             context.piecewise_rate = ((b1 & 0x00003f) << 16) | ((b2 & 0x0000ff) << 8) | b3

          if seamless_splice_flag:
             (b1,b2,b3,b4,b5) = struct.unpack_from("5B", binfield, pointer_data)
             pointer_data += 5
             context.splice_type  = (b1 & 0xf0) >> 4
             context.DTS_next_AU  = (b1 & 0x00000000e) << 29
             context.DTS_next_AU |= (b2 & 0x0000000ff) << 22
             context.DTS_next_AU |= (b3 & 0x0000000fe) << 14
             context.DTS_next_AU |= (b4 & 0x0000000ff) << 7
             context.DTS_next_AU |= (b5 & 0x0000000fe) >> 1

      # As defined at ITU-T H.222.0 section 2.4.3.5
      def process_adaptation_field (self, binfield, context):
          pointer_data = 0
          
          (b1,) = struct.unpack_from("B", binfield, 0)
          context.discontinuity_indicator              = ((b1 & 0x80) != 0x00)
          context.random_access_indicator              = ((b1 & 0x40) != 0x00)
          context.elementary_stream_priority_indicator = ((b1 & 0x20) != 0x00)
          context.PCR_flag                             = ((b1 & 0x10) != 0x00)
          context.OPCR_flag                            = ((b1 & 0x08) != 0x00)
          context.splicing_point_flag                  = ((b1 & 0x04) != 0x00)
          context.transport_private_flag               = ((b1 & 0x02) != 0x00)
          context.adaptation_field_extension_flag      = ((b1 & 0x01) != 0x00)
          pointer_data += 1

          if context.PCR_flag:
             (i1, i2) = struct.unpack_from (bit32 and "2I" or "2Q", binfield, pointer_data)
             pointer_data += 8
             context.program_clock_reference_base      = ((i1 & 0x00ffffffff) << 1) | ((i2 & 0x80000000) >> 31)
             context.program_clock_reference_extension = i2 & 0x000001ff
             if context.OPCR_flag:
                (i1, i2) = struct.unpack_from (bit32 and "2I" or "2Q", binfield, pointer_data)
                pointer_data += 8
                context.original_program_clock_reference_base      = ((i1 & 0x00ffffffff) << 1) | ((i2 & 0x80000000) >> 31)
                context.original_program_clock_reference_extension = i2 & 0x000001ff

          if context.splicing_point_flag:
             (context.splice_countdown,) = struct.unpack_from("b", binfield, pointer_data)
             pointer_data += 1

          if context.transport_private_flag:
             (private_data_length,) = struct.unpack_from("B", binfield, pointer_data)
             pointer_data += 1
             if private_data_length:
                context.transport_private_data = binfield[pointer_data:pointer_data+private_data_length]
                pointer_data += private_data_length

          if context.adaptation_field_extension_flag:
             (adaptation_field_extension_length,) = struct.unpack_from("B", binfield, pointer_data)
             pointer_data += 1
             if adaptation_field_extension_length:
                adaptation_field_extension = binfield[pointer_data:pointer_data+adaptation_field_extension_length]
                process_adaptation_field_extension(adaptation_field_extension, context)
          

      # As defined at ITU-T H.222.0 section 2.4.3.1
      # Return -2: Critical transport error
      # Return -1: Transport error
      # Return  0: Payload error
      # Return >0: Success
      def add_packet (self, binbuf, frequency=0):
          if len(binbuf) != 188: return -2

          context = H222_TS_packet_context(frequency)

          (b1, b2, b3, b4) = struct.unpack_from("4B", binbuf, 0)
          sync_byte                            = b1
          transport_error_indicator            = ((b2 & 0x80) != 0x00)
          context.payload_unit_start_indicator = ((b2 & 0x40) != 0x00)
          context.transport_priority           = ((b2 & 0x20) != 0x00)
          context.PID                          = (((b2 & 0x001f) << 8) | b3)
          context.transport_scrambling_control = (b4 & 0xc0) >> 6
          adaptation_field_present             = ((b4 & 0x20) != 0x00)
          payload_present                      = ((b4 & 0x10) != 0x00)
          continuity_counter                   = b4 & 0x0f
          pointer_data                         = 4
          
          if sync_byte != 0x47                 : return -2
          if transport_error_indicator         : return -2
          if context.PID in range(0x0002,00010): return -1

          # If this packet is a duplicated packet, discard it if the
          # previous packet was succesfully processed
          if payload_present:
             if context.PID in self.last_valid_counter:
                if self.last_valid_counter[context.PID] == continuity_counter: return 0
          
          if adaptation_field_present:
             (adaptation_field_length,) = struct.unpack_from("B", binbuf, pointer_data)
             pointer_data +=1
             if adaptation_field_length:
                adaptation_field = binbuf[pointer_data:pointer_data+adaptation_field_length]
                pointer_data += adaptation_field_length
                self.process_adaptation_field (adaptation_field, context)

          if payload_present:
             if pointer_data >= 188: return -1
             payload = binbuf[pointer_data:188]
             if context.PID in self.psi.active_pids:
                if not self.psi.add_packetized_sections(payload, context):
                   return 0 
             elif self.psi.pid_carries_pes(context.PID):
                if not self.pes.add_data(payload, context):
                   return 0
             else:
                _debug_("Discarding packet, ID 0x%x. Unknown stream type" % context.PID, DWARNING)
                return -1

          # Packet with payload succesfully processed
          if payload_present:
             self.last_valid_counter[context.PID] = continuity_counter

          return 1

