----------------------------------------------------------------------------
-- Clausify.hs

module Clausify
  ( M          -- :: * -> *; Monad
  , run        -- :: M a    -> a
  , clausify   -- :: Form   -> M [[Form]]
  , clauses    -- :: Form   -> [[Form]]
  , showClause -- :: [Form] -> String
  )
 where

{- This module implements a naive clausification algorithm. It uses
   a monad to keep track of new variable names, and which variables
   have been quantified (for Skolemnization).
-}

import Form

import List
  ( intersperse
  )

----------------------------------------------------------------------------
-- clausification monad

newtype M a
  = M ([Term] -> Int -> (a, Int))

instance Monad M where
  return x =
    M (\univs n -> (x, n))

  M f >>= k =
    M (\univs n0 ->
        let (a, n1) = f univs n0
            M g     = k a
            (b, n2) = g univs n1
         in (b, n2)
      )

new :: M Int
new = M (\univs n -> let n' = n+1 in (n',n'))

universal :: Term -> M a -> M a
universal x (M f) = M (\univs n -> f (x:univs) n)

universals :: M [Term]
universals = M (\univs n -> (univs, n))

run :: M a -> a
run (M f) = let (a, _) = f [] 0 in a

----------------------------------------------------------------------------
-- clausification algorithm

clausify :: Form -> M [[Form]]
clausify (And gs) =
  do css <- sequence [ clausify g | g <- gs ]
     return (concat css)

clausify (Or gs) =
  do css <- sequence [ clausify g | g <- gs ]
     return (map concat (cross css))

clausify (Equiv g h) =
  do clausify (And [Or [g, nt h], Or [nt g, h]])

clausify (All g) =
  do n <- new
     let x = Var ("All" ++ show n)
     universal x (clausify (g x))

clausify (Exi g) =
  do n  <- new
     xs <- universals
     let t = Fun ("exi" ++ show n) xs
     clausify (g t)

clausify (Not (Not g))     = do clausify g
clausify (Not (And gs))    = do clausify (Or (map Not gs))
clausify (Not (Or gs))     = do clausify (And (map Not gs))
clausify (Not (Equiv g h)) = do clausify (Equiv (Not g) h)
clausify (Not (All g))     = do clausify (Exi (Not . g))
clausify (Not (Exi g))     = do clausify (All (Not . g))

clausify f {- literal -}   = do return [[f]]

clauses :: Form -> [[Form]]
clauses f = run (clausify f)

----------------------------------------------------------------------------
-- show functions
        
showClause :: [Form] -> String
showClause [] = "p(a). -p(a). % empty clause"
showClause fs = concat (intersperse " | " (map show fs)) ++ "."

----------------------------------------------------------------------------
-- cross product
        
cross :: [[a]] -> [[a]]
cross []       = [[]]
cross (xs:xss) = [ x:ys | x <- xs, ys <- yss ]
  where
    yss = cross xss

----------------------------------------------------------------------------
-- the end.


