----------------------------------------------------------------------------
-- Form.hs

module Form
  ( Name                       -- :: *
  , Form(..)                   -- :: *; Show
  , Term(..)                   -- :: *; Show
  , nt                         -- :: Form -> Form
  , (<&>), (<|>), (==>), (<=>) -- :: Form -> Form -> Form
  , forall, exists             -- :: (Term -> Form) -> Form
  , (<==>)                     -- :: Term -> Term -> Form
  )
 where

{- This module implements a simple formula datatype, using higher-order
   syntax to implement the quantifiers. This makes it easy to generate
   formulas in this datatype, but less suitable for formula manipulation.
-}

import List
  ( intersperse
  )

----------------------------------------------------------------------------
-- Name, Form, Term

type Name
  = String

data Form
  = Not   Form
  | And   [Form]
  | Or    [Form]
  | Equiv Form Form
  | All (Term -> Form)
  | Exi (Term -> Form)
  | Pred Name [Term]

data Term
  = Fun Name [Term]
  | Var Name

----------------------------------------------------------------------------
-- constructor functions

nt :: Form -> Form
nt f = Not f

(<&>), (<|>), (==>), (<=>) :: Form -> Form -> Form
f <&> g = And [f, g]
f <|> g = Or [f, g]
f ==> g = nt f <|> g
f <=> g = Equiv f g

forall, exists :: (Term -> Form) -> Form
forall f = All f
exists f = Exi f

(<==>) :: Term -> Term -> Form
a <==> b = Pred "equal" [a, b]

----------------------------------------------------------------------------
-- show functions

instance Show Term where
  showsPrec n (Var x)     = showString x
  showsPrec n (Fun f ts)  = showString f . showTs ts
    where
      showTs [] = id
      showTs ts = showsBrack n ", " ts

instance Show Form where
  showsPrec n (Not f)     = showString "-" . showsPrec n f

  showsPrec n (And [])    = showString "$T"
  showsPrec n (And [f])   = showsPrec  n f
  showsPrec n (And fs)    = showsBrack n " & " fs

  showsPrec n (Or  [])    = showString "$F"
  showsPrec n (Or  [f])   = showsPrec  n f
  showsPrec n (Or  fs)    = showsBrack n " | " fs

  showsPrec n (Equiv f g) = showsPrec n f
                          . showString " <-> "
                          . showsPrec n g

  showsPrec n (All f)     = showsQuant n "all" f
  showsPrec n (Exi f)     = showsQuant n "exi" f
  
  showsPrec n (Pred p ts) = showsPrec n (Fun p ts)

showsBrack n op fs = showString "("
                   . foldr (.) id
                     ( intersperse (showString op)
                     $ map (showsPrec n) fs
                     )
                   . showString ")"

showsQuant n quant f = showString "("
                     . showString quant
                     . showString " "
                     . showString x
                     . showString " . "
                     . showsPrec n' (f (Var x))
                     . showString ")"
  where
    x  = "v" ++ show n
    n' = n+1

----------------------------------------------------------------------------
-- the end.


