-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   Chan Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent.Chan
  (
    CodePosition(..),
    Chan(..),
    newChan,         -- :: IO (Chan a)
    newChanLine,     -- :: CodePosition -> IO (Chan a)
    writeChan,       -- :: Chan a -> a -> IO ()
    writeChanLine,   --	:: CodePosition -> Chan a -> a -> IO ()
    writeChanLabel,  -- :: Chan a -> a -> String -> IO ()
    readChan,        -- :: Chan a -> IO a
    readChanLine,    -- :: CodePosition -> Chan a -> IO a
    dupChan,         -- :: Chan a -> IO (Chan a)
    unGetChan,       -- :: Chan a -> a -> IO ()
    getChanContents, -- :: Chan a -> IO [a]
    writeList2Chan,  -- :: Chan a -> [a] -> IO ()
    isEmptyChan,     --	:: Chan a -> Bool
    isEmptyChanLine  --	:: CodePosition -> Chan a -> Bool
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.Mem.Weak

import CHD.DebugInterface
import CHD.DebugMsgChan
import CHD.BaseTypes


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

-- the following is as described in the diploma thesis

data Chan a = Chan ChanNo 
		   (C.Chan a)
instance Show (Chan a) where
  showsPrec p (Chan number _) = shows number


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

newChan :: IO (Chan a)
newChan = newChanLine NoPosition


newChanLine :: CodePosition -> IO (Chan a)
newChanLine pos = do
  returnNewNoVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg (ChanNewSuspend (CHD returnNewNoVar) pos)
  C.takeMVar debugStop1
  chan <- C.newChan
  chanNo <- C.readMVar returnNewNoVar
  debugStop2 <- sendDebugMsg (ChanNew chanNo pos)
  C.takeMVar debugStop2
  addFinalizer chan (do {sendDebugMsg (ChanDied chanNo) ; return ()})
  return (Chan chanNo chan)


writeChan :: Chan a -> a -> IO ()
writeChan = writeChanLineLabel NoPosition ""


writeChanLabel :: String -> Chan a -> a -> IO ()
writeChanLabel = writeChanLineLabel NoPosition


writeChanLine :: CodePosition -> Chan a -> a -> IO ()
writeChanLine pos = writeChanLineLabel pos ""


writeChanLineLabel :: CodePosition -> String -> Chan a -> a -> IO ()
writeChanLineLabel pos label (Chan chanNo chan) val = do
  debugStop1 <- sendDebugMsg (ChanWriteSuspend chanNo pos)
  C.takeMVar debugStop1
  C.writeChan chan val
  debugStop2 <- sendDebugMsg (ChanWrite chanNo label pos)
  C.takeMVar debugStop2


-- just check if atomar holds a value. if then, no one writes to the chan.
-- this is just to take care of the order of events

readChan :: Chan a -> IO a
readChan = readChanLine NoPosition


readChanLine :: CodePosition -> Chan a -> IO a
readChanLine pos (Chan chanNo chan) = do
  debugStop1 <- sendDebugMsg (ChanReadSuspend chanNo pos)
  C.takeMVar debugStop1
  val <- C.readChan chan
  debugStop2 <- sendDebugMsg (ChanRead chanNo pos)
  C.takeMVar debugStop2
  return val



------------------------------------------------------------
-- not checked yet.
-- these simply base on readChan and writeChan
-- just sequencing...

getChanContents :: Chan a -> IO [a]
getChanContents chan = do
  value <- readChan chan
  list <- getChanContents chan
  return (value:list)


writeList2Chan :: Chan a -> [a] -> IO ()
writeList2Chan chan [] = return ()
writeList2Chan chan (value:list) = do
  writeChan chan value
  writeList2Chan chan list


-------------------------------------------------------------------------------
-- maybe this one could send a message too?

isEmptyChan :: Chan a -> IO Bool
isEmptyChan = isEmptyChanLine NoPosition


isEmptyChanLine :: CodePosition -> Chan a -> IO Bool
isEmptyChanLine pos (Chan chanNo chan) = C.isEmptyChan chan



-- is that right????
-- this is wrong at the time 
-- must figure out better concept for this. 
-- maybe one writes to more channels 
-- display on the canvas will be a small problem.
dupChan :: Chan a -> IO (Chan a)
dupChan (Chan chanNo chan) = do
  newChan <- C.dupChan chan
  return (Chan chanNo newChan)



-- this should be quite easy
-- simply add some new messages to debugMsgChan

unGetChan :: Chan a -> a -> IO ()
unGetChan (Chan chanNo chan) element =
  C.unGetChan chan element

