-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   ConcurrentMVar Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent.MVar
  (
    CodePosition(..),
    MVar(..),
    newEmptyMVar,	-- :: IO (MVar a)
    newEmptyMVarLine,	 -- :: CodePosition -> IO (MVar a)
    newMVar,		 -- :: a -> IO (MVar a)
    newMVarLine,	 -- :: CodePosition -> a -> IO (MVar a)
    newMVarLabel,	 -- :: String -> a -> IO (MVar a)
    newMVarLabelLine,	 -- :: CodePosition -> String -> a -> IO (MVar a)
    takeMVar,		 -- :: MVar a -> IO a
    takeMVarLine,	 -- :: CodePosition -> MVar a -> IO a
    putMVar,		 -- :: MVar a -> a -> IO ()
    putMVarLine,	 -- :: CodePosition -> MVar a -> a -> IO () 
    putMVarLabel,	 -- :: String -> MVar a -> a -> IO ()
    putMVarLabelLine,	 -- :: CodePosition -> String -> MVar a -> a -> IO ()
    readMVar,		 -- :: MVar a -> IO a
    readMVarLine,	 -- :: CodePosition -> MVar a -> IO a 
    swapMVar,		 -- :: MVar a -> a -> IO a
    swapMVarLine,	 -- :: CodePosition -> MVar a -> a -> IO a
    swapMVarLabel,	 -- :: MVar a -> a -> String -> IO a
    swapMVarLabelLine,	 -- :: CodePosition -> String -> MVar a -> a -> IO a
    tryTakeMVar,	 -- :: MVar a -> IO (Maybe a)
    tryTakeMVarLine,	 -- :: CodePosition -> MVar a -> IO (Maybe a)
    tryPutMVar,		 -- :: MVar a -> IO (Maybe a)
    tryPutMVarLine,	 -- :: CodePosition -> MVar a -> a -> IO Bool
    tryPutMVarLabel,	 -- :: String -> MVar a -> a -> IO Bool
    tryPutMVarLabelLine, -- :: CodePosition -> String -> MVar a -> a -> IO Bool
    isEmptyMVar,	 -- :: MVar a -> IO Bool
    isEmptyMVarLine,	 -- :: CodePosition -> MVar a -> IO Bool
    labelMVar		 -- :: MVar a -> String -> IO String
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.Mem.Weak

import CHD.DebugInterface
import CHD.BaseTypes


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

-- the following is as described in the diploma thesis

data MVar a = MVar MVarNo 
		   (C.MVar a)
instance Show (MVar a) where
  showsPrec p (MVar number _) = shows number
--  deriving Eq


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

-- the atomar feature is just to take care of the order of events. 
-- better to understand in channel.

newEmptyMVar :: IO (MVar a)
newEmptyMVar = newEmptyMVarLine NoPosition


newEmptyMVarLine :: CodePosition -> IO (MVar a)
newEmptyMVarLine pos = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg (MVarNewEmptySuspend (CHD returnNewNoMVar) pos)
  C.takeMVar debugStop1
  mvar <- C.newEmptyMVar
  mvarNo <- C.readMVar returnNewNoMVar
  debugStop2 <- sendDebugMsg (MVarNewEmpty mvarNo pos)
  C.takeMVar debugStop2
  addFinalizer mvar (do {sendDebugMsg (MVarDied mvarNo) ; return ()})
  return (MVar mvarNo mvar)


newMVar :: a -> IO (MVar a)
newMVar = newMVarLabelLine NoPosition ""


newMVarLabel :: String -> a -> IO (MVar a)
newMVarLabel = newMVarLabelLine NoPosition


newMVarLine :: CodePosition -> a -> IO (MVar a)
newMVarLine pos = newMVarLabelLine pos ""


newMVarLabelLine :: CodePosition -> String -> a -> IO (MVar a)
newMVarLabelLine pos label x = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg (MVarNewSuspend (CHD returnNewNoMVar) label pos)
  C.takeMVar debugStop1
  mvarNo <- C.readMVar returnNewNoMVar
  mvar <- C.newMVar x
  debugStop2 <- sendDebugMsg (MVarNew mvarNo label pos)
  C.takeMVar debugStop2
  addFinalizer mvar (do {sendDebugMsg (MVarDied mvarNo) ; return ()})
  return (MVar mvarNo mvar)


takeMVar :: MVar a -> IO a
takeMVar = takeMVarLine NoPosition


takeMVarLine :: CodePosition -> MVar a -> IO a
takeMVarLine pos (MVar mvarNo mvar) = do
  debugStop1 <- sendDebugMsg (MVarTakeSuspend mvarNo pos)
  C.takeMVar debugStop1
  x <- C.takeMVar mvar
  debugStop2 <- sendDebugMsg (MVarTake mvarNo pos)
  C.takeMVar debugStop2
  return x


putMVar :: MVar a -> a -> IO ()
putMVar = putMVarLabelLine NoPosition ""


putMVarLabel :: String -> MVar a -> a -> IO ()
putMVarLabel = putMVarLabelLine NoPosition


putMVarLine :: CodePosition -> MVar a -> a -> IO ()
putMVarLine pos = putMVarLabelLine pos ""


putMVarLabelLine :: CodePosition -> String -> MVar a -> a -> IO ()
putMVarLabelLine pos label (MVar mvarNo mvar) x = do
  debugStop1 <- sendDebugMsg (MVarPutSuspend mvarNo label pos)
  C.takeMVar debugStop1
  C.putMVar mvar x
  debugStop2 <- sendDebugMsg (MVarPut mvarNo label pos)
  C.takeMVar debugStop2


readMVar :: MVar a -> IO a
readMVar = readMVarLine NoPosition


readMVarLine :: CodePosition -> MVar a -> IO a
readMVarLine pos (MVar mvarNo mvar) = do
  debugStop1 <- sendDebugMsg (MVarReadSuspend mvarNo pos)
  C.takeMVar debugStop1
  x <- C.readMVar mvar
  debugStop2 <- sendDebugMsg (MVarRead mvarNo pos)
  C.takeMVar debugStop2
  return x


-- not really atomar implemented in Concurrent !?!?!?
swapMVar :: MVar a -> a -> IO a
swapMVar = swapMVarLabelLine NoPosition ""


swapMVarLabel :: String -> MVar a -> a -> IO a
swapMVarLabel = swapMVarLabelLine NoPosition


swapMVarLine :: CodePosition -> MVar a -> a -> IO a
swapMVarLine pos = swapMVarLabelLine pos ""


swapMVarLabelLine :: CodePosition -> String -> MVar a -> a -> IO a
swapMVarLabelLine pos newLabel (MVar mvarNo mvar) new  = do
  debugStop1 <- sendDebugMsg (MVarSwapSuspend mvarNo pos)
  C.takeMVar debugStop1
  old <- C.swapMVar mvar new
  debugStop2 <- sendDebugMsg (MVarSwap mvarNo newLabel pos)
  C.takeMVar debugStop2
  return old


tryTakeMVar :: MVar a -> IO (Maybe a)
tryTakeMVar = tryTakeMVarLine NoPosition


tryTakeMVarLine :: CodePosition -> MVar a -> IO (Maybe a)
tryTakeMVarLine pos (MVar mvarNo mvar) = do
  debugStop1 <- sendDebugMsg (MVarTryTakeSuspend mvarNo pos)
  C.takeMVar debugStop1
  erg <- C.tryTakeMVar mvar
  debugStop2 <- case erg of
    Nothing -> sendDebugMsg (MVarTry mvarNo pos)
    Just _ -> sendDebugMsg (MVarTake mvarNo pos)
  C.takeMVar debugStop2
  return erg


tryPutMVar :: MVar a -> a -> IO Bool
tryPutMVar = tryPutMVarLabelLine NoPosition ""


tryPutMVarLabel :: String -> MVar a -> a -> IO Bool
tryPutMVarLabel = tryPutMVarLabelLine NoPosition


tryPutMVarLine :: CodePosition -> MVar a -> a -> IO Bool
tryPutMVarLine pos = tryPutMVarLabelLine pos ""


tryPutMVarLabelLine :: CodePosition -> String -> MVar a -> a -> IO Bool
tryPutMVarLabelLine pos label (MVar mvarNo mvar) value = do
  debugStop1 <- sendDebugMsg (MVarTryPutSuspend mvarNo pos)
  C.takeMVar debugStop1
  b <- C.tryPutMVar mvar value
  debugStop2 <- case b of
    False -> sendDebugMsg (MVarTry mvarNo pos)
    True -> sendDebugMsg (MVarPut mvarNo label pos)
  C.takeMVar debugStop2
  return b


isEmptyMVar :: MVar a -> IO Bool
isEmptyMVar = isEmptyMVarLine NoPosition


isEmptyMVarLine :: CodePosition -> MVar a -> IO Bool
isEmptyMVarLine pos (MVar mvarNo mvar) = C.isEmptyMVar mvar


labelMVar :: MVar a -> String -> IO String
labelMVar (MVar mvarNo mvar) name = do
  oldName <- C.newEmptyMVar
  debugStop <- sendDebugMsg (MVarLabel mvarNo name oldName)
  C.takeMVar debugStop
  C.takeMVar oldName
