-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   QSemN Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent.QSemN
  (
    CodePosition(..),
    QSemN(..),
    newQSemN,		-- :: Int -> IO QSemN
    newQSemNLine,	-- :: CodePosition -> Int -> IO QSemN
    signalQSemN,	-- :: QSemN -> Int -> IO ()
    signalQSemNLine,	-- :: CodePosition -> QSemN -> Int -> IO ()
    waitQSemN,		-- :: QSemN -> Int -> IO ()
    waitQSemNLine,	-- :: CodePosition -> QSemN -> Int -> IO ()
    labelQSemN		-- :: QSemN -> String -> IO String
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.Mem.Weak

import CHD.DebugInterface
import CHD.DebugMsgChan
import CHD.BaseTypes


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

data QSemN = QSemN QSemNNo 
		   C.QSemN      
instance Show QSemN where
  showsPrec p (QSemN number _) = shows number
--  deriving Eq


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

newQSemN :: Int -> IO QSemN
newQSemN = newQSemNLine NoPosition


newQSemNLine :: CodePosition -> Int -> IO QSemN
newQSemNLine pos x = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop <- sendDebugMsg (QSemNNewSuspend (CHD returnNewNoMVar) x pos)
  C.takeMVar debugStop
  qsemnNo <- C.readMVar returnNewNoMVar
  qsemn <- C.newQSemN x
  debugStop2 <- sendDebugMsg (QSemNNew qsemnNo x pos)
  C.takeMVar debugStop2
  addFinalizer qsemn (do {sendDebugMsg (QSemNDied qsemnNo) ; return ()})
  return (QSemN qsemnNo qsemn)


waitQSemN :: QSemN -> Int -> IO ()
waitQSemN = waitQSemNLine NoPosition


waitQSemNLine :: CodePosition -> QSemN -> Int -> IO ()
waitQSemNLine pos (QSemN qsemnNo qsemn) x = do
  debugStop1 <- sendDebugMsg (QSemNWaitSuspend qsemnNo x pos) 
  C.takeMVar debugStop1
  C.waitQSemN qsemn x
  debugStop2 <- sendDebugMsg (QSemNWait qsemnNo x pos)
  C.takeMVar debugStop2


signalQSemN :: QSemN -> Int -> IO ()
signalQSemN = signalQSemNLine NoPosition


signalQSemNLine :: CodePosition -> QSemN -> Int -> IO ()
signalQSemNLine pos (QSemN qsemnNo qsemn) x = do
  debugStop1 <- sendDebugMsg (QSemNSignalSuspend qsemnNo x pos)
  C.takeMVar debugStop1
  C.signalQSemN qsemn x
  debugStop2 <- sendDebugMsg (QSemNSignal qsemnNo x pos)
  C.takeMVar debugStop2


labelQSemN :: QSemN -> String -> IO String
labelQSemN (QSemN qsemnNo qsemn) name = do
  oldName <- C.newEmptyMVar
  debugStop <- sendDebugMsg (QSemNLabel qsemnNo name oldName)
  C.takeMVar debugStop
  C.takeMVar oldName
