-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   SampleVar Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.Control.Concurrent.SampleVar
  (
    CodePosition(..),
    SampleVar(..),
    newEmptySampleVar,	    -- :: IO (SampleVar a)
    newEmptySampleVarLine,  -- :: CodePosition -> IO (SampleVar a)
    newSampleVar,	    -- :: IO (SampleVar a)
    newSampleVarLabel,	    -- :: String -> IO (SampleVar a)
    newSampleVarLine,	    -- :: CodePosition -> IO (SampleVar a)
    newSampleVarLabelLine,  -- :: CodePosition -> String -> IO (SampleVar a
    emptySampleVar,	    -- :: SampleVar a -> IO ()
    emptySampleVarLine,	    -- :: CodePosition -> SampleVar a -> IO ()
    readSampleVar,	    -- :: SampleVar a -> IO a
    readSampleVarLine,	    -- :: CodePosition -> SampleVar a -> IO a
    writeSampleVar,	    -- :: SampleVar a -> a -> IO ()
    writeSampleVarLabel,    -- :: String -> SampleVar a -> a -> IO ()
    writeSampleVarLine,	    -- :: CodePosition -> SampleVar a -> a -> IO ()
    writeSampleVarLabelLine -- :: CodePosition -> String -> SampleVar a -> a 
			    --		       -> IO ()
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.Mem.Weak

import CHD.DebugInterface
import CHD.DebugMsgChan
import CHD.BaseTypes


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

data SampleVar a = SampleVar SampleVarNo 
		   (C.SampleVar a)
instance Show (SampleVar a) where
  showsPrec p (SampleVar number _) = shows number
--  deriving Eq


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

-- the atomar feature is just to take care of the order of events. 
-- better to understand in channel.

newEmptySampleVar :: IO (SampleVar a)
newEmptySampleVar = newEmptySampleVarLine NoPosition


newEmptySampleVarLine :: CodePosition -> IO (SampleVar a)
newEmptySampleVarLine pos = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg
		(SampleVarNewEmptySuspend (CHD returnNewNoMVar) pos)
  C.takeMVar debugStop1
  samplevarNo <- C.readMVar returnNewNoMVar
  samplevar <- C.newEmptySampleVar
  debugStop2 <- sendDebugMsg (SampleVarNewEmpty samplevarNo pos)
  C.takeMVar debugStop2
  addFinalizer samplevar ( do 
    sendDebugMsg (SampleVarDied samplevarNo)
    return ()
    )
  return (SampleVar samplevarNo samplevar)


newSampleVar :: a -> IO (SampleVar a)
newSampleVar = newSampleVarLabelLine NoPosition ""


newSampleVarLabel :: String -> a -> IO (SampleVar a)
newSampleVarLabel = newSampleVarLabelLine NoPosition 


newSampleVarLine :: CodePosition -> a -> IO (SampleVar a)
newSampleVarLine pos = newSampleVarLabelLine pos ""


newSampleVarLabelLine :: CodePosition -> String -> a -> IO (SampleVar a)
newSampleVarLabelLine pos label x = do
  returnNewNoMVar <- C.newEmptyMVar
  debugStop1 <- sendDebugMsg (SampleVarNewSuspend (CHD returnNewNoMVar) pos)
  C.takeMVar debugStop1
  samplevar <- C.newSampleVar x
  samplevarNo <- C.readMVar returnNewNoMVar
  debugStop2 <- sendDebugMsg (SampleVarNew samplevarNo label pos)
  C.takeMVar debugStop2
  addFinalizer samplevar ( do
    sendDebugMsg (SampleVarDied samplevarNo)
    return ()
    )
  return (SampleVar samplevarNo samplevar)


emptySampleVar :: SampleVar a -> IO ()
emptySampleVar = emptySampleVarLine NoPosition


emptySampleVarLine :: CodePosition -> SampleVar a -> IO ()
emptySampleVarLine pos (SampleVar samplevarNo samplevar) = do
  debugStop1 <- sendDebugMsg (SampleVarEmptySuspend samplevarNo pos)
  C.takeMVar debugStop1
  myEmptySampleVar samplevar
  debugStop2 <- sendDebugMsg (SampleVarEmpty samplevarNo pos)
  C.takeMVar debugStop2
  

readSampleVar :: SampleVar a -> IO a
readSampleVar = readSampleVarLine NoPosition


readSampleVarLine :: CodePosition -> SampleVar a -> IO a
readSampleVarLine pos (SampleVar samplevarNo samplevar) = do
  debugStop1 <- sendDebugMsg (SampleVarReadSuspend samplevarNo pos)
  C.takeMVar debugStop1
  x <- C.readSampleVar samplevar
  debugStop2 <- sendDebugMsg (SampleVarRead samplevarNo pos)
  C.takeMVar debugStop2
  return x


writeSampleVar :: SampleVar a -> a -> IO ()
writeSampleVar = writeSampleVarLabelLine NoPosition ""


writeSampleVarLabel :: String -> SampleVar a -> a -> IO ()
writeSampleVarLabel = writeSampleVarLabelLine NoPosition


writeSampleVarLine :: CodePosition -> SampleVar a -> a -> IO ()
writeSampleVarLine pos = writeSampleVarLabelLine pos ""


writeSampleVarLabelLine :: CodePosition -> String -> SampleVar a -> a -> IO ()
writeSampleVarLabelLine pos label (SampleVar samplevarNo samplevar) x = do
  debugStop1 <- sendDebugMsg (SampleVarWriteSuspend samplevarNo pos)
  C.takeMVar debugStop1
  C.writeSampleVar samplevar x
  debugStop2 <- sendDebugMsg (SampleVarWrite samplevarNo label pos)
  C.takeMVar debugStop2


myEmptySampleVar :: C.SampleVar a -> IO ()
myEmptySampleVar v = do
   (readers, var) <- C.takeMVar v
   if readers > 0 then
     C.takeMVar var >>
     C.putMVar v (0,var)
    else
     C.putMVar v (readers,var)