-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   DebugState Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

-- short Description: DS = DebuggerState

-- this is an easy record and function to change the state of it
--   or to lookup some data.
-- needed by debugmain

module CHD.DebugState
  (
    ThreadStatus(..),
    ThreadAction(..),
    ThreadState(..),
    ComObjState(..),
    DebuggerState(..),

    initDebuggerState,		-- :: C.ThreadId -> DS

    lookupThreadNo,		-- :: DS -> C.ThreadId -> ThreadNo
    lookupThread,		-- :: DS -> ThreadNo -> ThreadState

    stateAddThread,		-- :: DS -> C.ThreadId -> (DS, ThreadNo)
    stateSetThread,		-- :: DS -> ThreadNo -> ThreadState -> DS
    stateRemoveThread,		-- :: DS -> C.ThreadId -> (DS, ThreadNo)
    stateSetThreadStatus,	-- :: DS -> ThreadNo -> ThreadStatus -> DS
    stateSetThreadParent,	-- :: DS -> ThreadNo -> ThreadNo -> DS
    stateAddThreadStopper,	-- :: DS -> ThreadNo -> C.MVar() -> DS
    stateRemoveThreadStopper,	-- :: DS -> ThreadNo -> DS

    lookupVisualize,		-- :: DS -> Object -> Bool
    lookupBreak,		-- :: DS -> Object -> BreakCounter
    lookupLabel,		-- :: DS -> Object -> String
    lookupObjectList,		-- :: DS -> [Object]
    lookupThreadObjectList,	-- :: DS -> [Object]

    stateSetVisualize,		-- :: DS -> Object -> Bool -> DS
    stateSetBreak,		-- :: DS -> Object -> BreakCounter -> DS
    stateSetLabel,		-- :: DS -> Object -> String -> DS
    stateSetSuspend,		-- :: DS -> ThreadNo -> Object -> ThreadAction
				--	 -> DS
    stateRemoveSuspend,		-- :: DS -> ThreadNo -> DS

    lookupComObj,		-- :: DS -> Object -> ComObjState
    lookupActors,		-- :: DS -> Object -> [ThreadNo]
    lookupContents,		-- :: DS -> Object -> [ObjContent]

    stateSetComObj,		-- :: DS -> Object -> ComObjState -> DS
    stateRemoveComObj,		-- :: DS -> Object -> DS
        
    stateAddChannel,		-- :: DS -> (DS, ChanNo)
    stateChannelRemoveMessage,	-- :: DS -> ChanNo -> DS
    stateChannelAddMessage,	-- :: DS -> ChanNo -> ThreadNo -> [String] 
				--	 -> DS

    stateAddMVar,		-- :: DS -> (DS, MVarNo)
    stateMVarRemoveContents,	-- :: DS -> MVarNo -> DS
    stateMVarSetContents,	-- :: DS -> MVarNo -> ThreadNo -> String -> DS

    stateAddQSem,		-- :: DS -> (DS, QSemNo)
    stateAddQSemN,		-- :: DS -> (DS, QSemNNo)
    stateSetQuantity,		-- :: DS -> Object -> Int -> DS
    stateIncQuantity,		-- :: DS -> Object -> Int -> DS
    stateDecQuantity,		-- :: DS -> Object -> Int -> DS   

    stateAddSampleVar,		-- :: DS -> (DS, SampleVar)
    stateSampleVarRemoveContents,  -- :: DS -> SampleVarNo -> DS
    stateSampleVarSetContents,	-- :: DS -> SampleVarNo -> ThreadNo -> String
				--	 -> DS

    lookupActionBreak,		-- :: DS -> DebugMsgAction -> BreakCounter
    stateSetActionBreak,	-- :: DS -> DebugMsgAction -> BreakCounter 
				--	 -> DS
    lookupSteps,		-- :: DS -> Object -> [Step]
    stateAddStep		-- :: DS -> Object -> Step -> DS
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import Data.FiniteMap

import CHD.DebugMsgChan
import CHD.BaseTypes
import CHD.Environment


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

data ThreadState = ThreadState {
  threadStopper	   :: Maybe (Int, C.MVar ()),
  threadBreak	   :: BreakCounter,
  threadStatus	   :: ThreadStatus,
  threadVisualize  :: Bool,
  threadLabel	   :: String,
  threadParent	   :: ThreadNo,
  threadSuspend	   :: Maybe (Object, ThreadAction)
  }


data ComObjState = ComObjState {
  comObjVisualize :: Bool,
  comObjActors    :: [ThreadNo],
  comObjContents  :: [ObjContent],
  comObjLabel	  :: String,
  comObjBreak	  :: BreakCounter,
  comObjSuspend	  :: [ThreadNo]
  }
  


data DebuggerState = DebuggerState {
  stopperIdCounter   :: Int,
  threadCount	     :: ThreadNo,
  threadMap	     :: FiniteMap C.ThreadId ThreadNo,
  threadStates	     :: FiniteMap ThreadNo ThreadState,

--  autoContinueTime   :: Int,
  actionBreak	     :: FiniteMap DebugMsgAction BreakCounter,

  progress	     :: FiniteMap ThreadNo [Step],

  mvarCount	     :: MVarNo,
  channelCount	     :: ChanNo,
  qsemCount	     :: QSemNo,
  qsemnCount	     :: QSemNNo,
  samplevarCount     :: SampleVarNo,
  comObjStates	     :: FiniteMap Object ComObjState
  }


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- init
-------------------------------------------------------------------------------

initThreadState :: String -> ThreadState 
initThreadState label = ThreadState {
  threadStopper	  = Nothing,
  threadBreak	  = NoTime,
  threadStatus	  = TSDebug,
  threadVisualize = True,
  threadLabel	  = label,
  threadParent	  = UnknownThread,
  threadSuspend	  = Nothing
  }


initComObjState :: [ObjContent] -> String -> ComObjState
initComObjState contents label = ComObjState {
  comObjVisualize = True,
  comObjActors  = [],
  comObjContents   = contents,
  comObjLabel     = label,
  comObjBreak     = NoTime,
  comObjSuspend	  = []
  }


initActionBreak :: FiniteMap DebugMsgAction BreakCounter
initActionBreak = 
  listToFM
    (map (\elem -> (elem, EveryTime))
      [ ThreadStartA,
	ThreadForkSuspendA,
	ThreadKillSuspendA,
	MVarNewEmptySuspendA,
	MVarNewSuspendA,
	MVarTakeSuspendA,
	MVarReadSuspendA,
	MVarPutSuspendA,
	MVarSwapSuspendA,
	MVarTryTakeSuspendA,
	MVarTryPutSuspendA,
	ChanNewSuspendA,
	ChanWriteSuspendA,
	ChanReadSuspendA,
	ChanUnGetSuspendA,
	QSemNewSuspendA,
	QSemSignalSuspendA,
	QSemWaitSuspendA,
	QSemNNewSuspendA,
	QSemNSignalSuspendA,
	QSemNWaitSuspendA,
	SampleVarNewEmptySuspendA,
	SampleVarNewSuspendA,
	SampleVarEmptySuspendA,
	SampleVarWriteSuspendA,
	SampleVarReadSuspendA
	])
	

initDebuggerState :: C.ThreadId -> DebuggerState
initDebuggerState myId = DebuggerState {
  stopperIdCounter     = 0,
  threadCount	       = ThreadNo 0,
  threadMap	       = emptyFM,
  threadStates	       = emptyFM,

--  autoContinueTime     = autoContinueTimeoutOption,
  actionBreak	       = addToFM (if (autoContinueOption == 1) 
				    then emptyFM 
				    else initActionBreak)
			 ThreadBreakA (if (activateBreakThread == 1) 
					 then EveryTime 
					 else NoTime),
					      
  progress	       = emptyFM,
 
  mvarCount	       = MVarNo 0,
  channelCount	       = ChanNo 0,
  qsemCount            = QSemNo 0,
  qsemnCount	       = QSemNNo 0,
  samplevarCount       = SampleVarNo 0,
  comObjStates	       = emptyFM
  }


-------------------------------------------------------------------------------
-- statistics
--    not needed yet! - implement a statistic-window!
-------------------------------------------------------------------------------

activeThreads :: DebuggerState -> Int
activeThreads state = 
  length [ x | x <- map threadStatus (eltsFM (threadStates state)),
	       (elem x [TSRunning, TSSuspend, TSDelay]) ]


activeChannels :: DebuggerState -> Int
activeChannels state = 
  length [ x | x <- (keysFM (comObjStates state)),
	       filterChannel x ]
  where
  filterChannel (ChanObj _) = True
  filterChannel _ = False


activeMVars :: DebuggerState -> Int
activeMVars state = 
  length [ x | x <- (keysFM (comObjStates state)),
	       filterMVar x ]
  where
  filterMVar (MVarObj _) = True
  filterMVar _ = False


-------------------------------------------------------------------------------
-- lookup THREAD-section
-------------------------------------------------------------------------------

lookupThreadNo :: DebuggerState -> C.ThreadId -> ThreadNo
lookupThreadNo state threadId = 
  lookupWithDefaultFM (threadMap state) UnknownThread threadId 
				  

lookupThread :: DebuggerState -> ThreadNo -> ThreadState
lookupThread state threadNo =
  lookupWithDefaultFM (threadStates state) (initThreadState "") threadNo


-------------------------------------------------------------------------------
-- modify THREAD-section
-------------------------------------------------------------------------------

newStopperIdCounter :: DebuggerState -> (DebuggerState, Int)
newStopperIdCounter state =
  let counter = succ (stopperIdCounter state) in
  if (counter == 0)
    then (state { stopperIdCounter = 1 }, 1)
    else (state { stopperIdCounter = counter }, counter)


stateAddThread :: DebuggerState -> C.ThreadId -> (DebuggerState, ThreadNo)
stateAddThread state threadId = 
  let tNo = (lookupThreadNo state threadId) in
  case tNo of
    ThreadNo _ -> (state, tNo)
    _ -> 
      let threadNo = (threadCount state)
	  newState = state { threadMap = 
			       addToFM (threadMap state) threadId threadNo,
			     threadCount = succ threadNo }
	  newState2 = stateSetThread newState threadNo 
		        (initThreadState $ show threadNo) in
      (newState2, threadNo)


stateSetThread :: DebuggerState -> ThreadNo -> ThreadState -> DebuggerState
stateSetThread state threadNo threadState =
  state { threadStates = addToFM (threadStates state) threadNo threadState }


stateRemoveThread :: DebuggerState -> C.ThreadId -> (DebuggerState, ThreadNo)
stateRemoveThread state threadId =
  let tNo = (lookupThreadNo state threadId) in
  case tNo of
    ThreadNo _ -> 
      let newState = state { threadMap = delFromFM (threadMap state) threadId }
	  in
      (newState, tNo)
    _ -> (state, UnknownThread)


stateSetThreadStatus :: DebuggerState -> ThreadNo -> ThreadStatus 
			-> DebuggerState
stateSetThreadStatus state threadNo status =
  let threadState = lookupThread state threadNo
      newThreadState = threadState { threadStatus = status } in
  stateSetThread state threadNo newThreadState
  

stateSetThreadParent :: DebuggerState -> ThreadNo -> ThreadNo -> DebuggerState
stateSetThreadParent state threadNo parent =
  let threadState = lookupThread state threadNo
      newThreadState = threadState { threadParent = parent } in
  stateSetThread state threadNo newThreadState


stateAddThreadStopper :: DebuggerState -> ThreadNo -> C.MVar() -> DebuggerState
stateAddThreadStopper state threadNo stopper =
  let threadState = lookupThread state threadNo
      (newState, counter) = (newStopperIdCounter state)
      newThreadState = threadState { threadStopper = Just (counter, stopper) }
      in
  stateSetThread newState threadNo newThreadState  


stateRemoveThreadStopper :: DebuggerState -> ThreadNo -> DebuggerState
stateRemoveThreadStopper state threadNo =
  let threadState = lookupThread state threadNo
      newThreadState = threadState { threadStopper = Nothing } in
  stateSetThread state threadNo newThreadState  


-------------------------------------------------------------------------------
-- lookup OBJECT-section
-------------------------------------------------------------------------------

lookupVisualize :: DebuggerState -> Object -> Bool
lookupVisualize state object =
  case object of
    ThreadObj threadNo -> threadVisualize (lookupThread state threadNo)
    _ -> comObjVisualize (lookupComObj state object)


lookupBreak :: DebuggerState -> Object -> BreakCounter
lookupBreak state object =
  case object of
    ThreadObj threadNo -> threadBreak (lookupThread state threadNo)
    _ -> comObjBreak (lookupComObj state object)


lookupLabel :: DebuggerState -> Object -> String
lookupLabel state object =
  case object of
    ThreadObj threadNo -> threadLabel (lookupThread state threadNo)
    _ -> comObjLabel (lookupComObj state object)


lookupObjectList :: DebuggerState -> [Object]
lookupObjectList state =
  let comObjs = keysFM (comObjStates state) in
  (lookupThreadObjectList state) ++ comObjs


lookupThreadObjectList :: DebuggerState -> [Object]
lookupThreadObjectList state =
  map (\tNo -> ThreadObj tNo) 
		       (filter (\tNo -> 
			         elem (threadStatus (lookupThread state tNo))
				      [ TSRunning, TSSuspend, TSDelay ])
			       (eltsFM (threadMap state)))


-------------------------------------------------------------------------------
-- modify OBJECT-section
-------------------------------------------------------------------------------

stateSetVisualize :: DebuggerState -> Object -> Bool -> DebuggerState
stateSetVisualize state object visualize =
  case object of
    ThreadObj threadNo -> 
      let threadState = lookupThread state threadNo
          newThreadState = threadState { threadVisualize = visualize } in
      stateSetThread state threadNo newThreadState
    _ -> 
      let comObjState = lookupComObj state object
	  newComObjState = comObjState { comObjVisualize = visualize } in
      stateSetComObj state object newComObjState


stateSetBreak :: DebuggerState -> Object -> BreakCounter -> DebuggerState
stateSetBreak state object break =
  case object of
    ThreadObj threadNo -> 
      let threadState = lookupThread state threadNo
          newThreadState = threadState { threadBreak = break } in
      stateSetThread state threadNo newThreadState
    _ -> 
      let comObjState = lookupComObj state object
	  newComObjState = comObjState { comObjBreak = break } in
      stateSetComObj state object newComObjState


stateSetLabel :: DebuggerState -> Object -> String -> DebuggerState
stateSetLabel state object label =
  case object of
    ThreadObj threadNo -> 
      let threadState = lookupThread state threadNo
          newThreadState = threadState { threadLabel = label } in
      stateSetThread state threadNo newThreadState
    _ -> 
      let comObjState = lookupComObj state object
	  newComObjState = comObjState { comObjLabel = label } in
      stateSetComObj state object newComObjState


stateSetSuspend :: DebuggerState -> ThreadNo -> Object -> ThreadAction 
				 -> DebuggerState
stateSetSuspend state threadNo comObj action = 
  let newState = stateRemoveSuspend state threadNo
      newThreadState = (lookupThread newState threadNo)
        { threadSuspend = Just (comObj, action) }	
      comObjState = lookupComObj newState comObj
      newComObjState = comObjState 
	{ comObjSuspend = threadNo:(comObjSuspend comObjState) } in
  stateSetComObj
    (stateSetThread newState threadNo newThreadState)
    comObj newComObjState


stateRemoveSuspend :: DebuggerState -> ThreadNo -> DebuggerState
stateRemoveSuspend state threadNo =
  let threadState = lookupThread state threadNo
      newThreadState = threadState { threadSuspend = Nothing }
      newState = stateSetThread state threadNo newThreadState in
  case threadSuspend threadState of
    Nothing -> newState
    Just (comObj, _) -> 
      let comObjState = lookupComObj state comObj
	  suspend = comObjSuspend comObjState
	  newComObjState = comObjState
	    { comObjSuspend = filter (/= threadNo) suspend } in
      stateSetComObj newState comObj newComObjState
    

-------------------------------------------------------------------------------
-- lookup COMOBJ-section
-------------------------------------------------------------------------------

lookupComObj :: DebuggerState -> Object -> ComObjState
lookupComObj state object =
  lookupWithDefaultFM (comObjStates state) 
		      (initComObjState [] "") object


lookupActors :: DebuggerState -> Object -> [ThreadNo]
lookupActors state object =
  comObjActors (lookupComObj state object)


lookupContents :: DebuggerState -> Object -> [ObjContent]
lookupContents state object =
  comObjContents (lookupComObj state object)


-------------------------------------------------------------------------------
-- modify COMOBJ-section
-------------------------------------------------------------------------------

stateSetComObj :: DebuggerState -> Object -> ComObjState -> DebuggerState
stateSetComObj state object comObjState =
  state { comObjStates = addToFM (comObjStates state) object comObjState }


stateSetActors :: DebuggerState -> Object -> [ThreadNo] -> DebuggerState
stateSetActors state object actors =
  let comObjState = lookupComObj state object
      newComObjState = comObjState { comObjActors = actors } in
  stateSetComObj state object newComObjState


stateSetContents :: DebuggerState -> Object -> [ObjContent] -> DebuggerState
stateSetContents state object content =
  let comObjState = lookupComObj state object
      newComObjState = comObjState { comObjContents = content } in
  stateSetComObj state object newComObjState
    

stateRemoveComObj :: DebuggerState -> Object -> DebuggerState
stateRemoveComObj state object =
  state { comObjStates = delFromFM (comObjStates state) object }


-------------------------------------------------------------------------------
-- modify CHANNEL-section
-------------------------------------------------------------------------------

stateAddChannel :: DebuggerState -> (DebuggerState, ChanNo)
stateAddChannel state = 
  let channelNo = channelCount state
      comObj = ChanObj channelNo
      newState = state { channelCount = succ channelNo } in
  (stateSetComObj newState comObj (initComObjState [] $ show channelNo),
   channelNo)


stateChannelAddMessage :: DebuggerState -> ChanNo -> ThreadNo -> String 
					-> DebuggerState
stateChannelAddMessage state channelNo msg label =
  let comObj = ChanObj channelNo
      messages = lookupActors state comObj
      labels = lookupContents state comObj
      newState = stateSetActors state comObj (messages ++ [msg]) in
  stateSetContents state comObj (labels ++ [Label label])


stateChannelRemoveMessage :: DebuggerState -> ChanNo -> DebuggerState
stateChannelRemoveMessage state channelNo =
  let comObj = ChanObj channelNo
      messages = lookupActors state comObj 
      labels = lookupContents state comObj 
      newLabels = case labels of
	   [] -> []
	   _  -> tail labels in
  if null messages
    then error ("CHD-Error: Try To Remove From (empty-Channel-List) " 
    ++ (show channelNo))
    else let newState = stateSetActors state comObj (tail messages) in
	 stateSetContents newState comObj newLabels


-------------------------------------------------------------------------------
-- modify MVAR-section
-------------------------------------------------------------------------------

stateAddMVar :: DebuggerState -> (DebuggerState, MVarNo)
stateAddMVar state = 
  let mvarNo = mvarCount state
      comObj = MVarObj mvarNo
      newState = state { mvarCount = succ mvarNo } in
  (stateSetComObj newState comObj (initComObjState [] $ show mvarNo),
   mvarNo)


stateMVarSetContents :: DebuggerState -> MVarNo -> ThreadNo -> String
				      -> DebuggerState
stateMVarSetContents state mvarNo msg "" =
  let comObj = MVarObj mvarNo
      newState = stateSetActors state comObj [msg] in
  stateSetContents newState comObj []	     

stateMVarSetContents state mvarNo msg label =
  let comObj = MVarObj mvarNo
      newState = stateSetActors state comObj [msg] in
  stateSetContents newState comObj [Label label]


stateMVarRemoveContents :: DebuggerState -> MVarNo -> DebuggerState
stateMVarRemoveContents state mvarNo =
  let comObj = MVarObj mvarNo
      newState = stateSetActors state comObj [] in
  stateSetContents newState comObj []


-------------------------------------------------------------------------------
-- modify QSem-section
-------------------------------------------------------------------------------

stateAddQSem :: DebuggerState -> Int -> (DebuggerState, QSemNo)
stateAddQSem state quantity = 
  let qsemNo = qsemCount state
      comObj = QSemObj qsemNo
      newState = state { qsemCount = succ qsemNo } in
  (stateSetComObj newState comObj (initComObjState [Quantity quantity]
						   (show qsemNo)), qsemNo)


-------------------------------------------------------------------------------
-- modify QSemN-section
-------------------------------------------------------------------------------
 
stateAddQSemN :: DebuggerState -> Int -> (DebuggerState, QSemNNo)
stateAddQSemN state quantity = 
  let qsemnNo = qsemnCount state
      comObj = QSemNObj qsemnNo
      newState = state { qsemnCount = succ qsemnNo } in
  (stateSetComObj newState comObj (initComObjState [Quantity quantity] 
						   (show qsemnNo)), qsemnNo)


stateSetQuantity :: DebuggerState -> Object -> Int -> DebuggerState
stateSetQuantity state object quantity =
  let comObjState = lookupComObj state object
      newComObjState = comObjState { comObjContents = [Quantity quantity] } in
  stateSetComObj state object newComObjState


stateIncQuantity :: DebuggerState -> Object -> Int -> DebuggerState
stateIncQuantity state object incQ = 
  let comObjState = lookupComObj state object
      inc (Quantity q) = Quantity (q+incQ)
      oldQuantitys = comObjContents comObjState
      newQuantitys = map inc oldQuantitys
      newComObjState = comObjState { comObjContents = newQuantitys } in
  stateSetComObj state object newComObjState


stateDecQuantity :: DebuggerState -> Object -> Int -> DebuggerState
stateDecQuantity state object decQ = 
  let comObjState = lookupComObj state object
      dec (Quantity q) = Quantity (q-decQ)
      oldQuantitys = comObjContents comObjState
      newQuantitys = map dec oldQuantitys 
      newComObjState = comObjState { comObjContents = newQuantitys } in
  stateSetComObj state object newComObjState


-------------------------------------------------------------------------------
-- modify SAMPLEVAR-section
-------------------------------------------------------------------------------

stateAddSampleVar :: DebuggerState -> (DebuggerState, SampleVarNo)
stateAddSampleVar state = 
  let samplevarNo = (samplevarCount state)
      comObj = SampleVarObj samplevarNo
      newState = state { samplevarCount = succ samplevarNo } in
  (stateSetComObj newState comObj (initComObjState [] $ show samplevarNo), 
   samplevarNo)


stateSampleVarSetContents :: DebuggerState -> SampleVarNo -> ThreadNo -> String
					   -> DebuggerState
stateSampleVarSetContents state samplevarNo msg "" =
  let comObj = SampleVarObj samplevarNo
      newState = stateSetActors state comObj [msg] in
  stateSetContents newState comObj []		     

stateSampleVarSetContents state samplevarNo msg label =
  let comObj = SampleVarObj samplevarNo
      newState = stateSetActors state comObj [msg] in
  stateSetContents newState comObj [Label label]


stateSampleVarRemoveContents :: DebuggerState -> SampleVarNo -> DebuggerState
stateSampleVarRemoveContents state samplevarNo =
  let comObj = SampleVarObj samplevarNo
      newState = stateSetActors state comObj [] in
  stateSetContents newState comObj []


-------------------------------------------------------------------------------
-- lookup ACTION-section
-------------------------------------------------------------------------------

lookupActionBreak :: DebuggerState -> DebugMsgAction -> BreakCounter
lookupActionBreak state action =
  lookupWithDefaultFM (actionBreak state) NoTime action


-------------------------------------------------------------------------------
-- modify ACTION-section
-------------------------------------------------------------------------------

stateSetActionBreak :: DebuggerState -> DebugMsgAction -> BreakCounter
		       -> DebuggerState
stateSetActionBreak state action counter =
  state { actionBreak = addToFM (actionBreak state) action counter }


-------------------------------------------------------------------------------
-- lookup PROGRESS-section
-------------------------------------------------------------------------------

lookupSteps :: DebuggerState -> Object -> [Step]
lookupSteps state (ThreadObj threadNo) =
  lookupWithDefaultFM (progress state) [] threadNo


-------------------------------------------------------------------------------
-- modify PROGRESS-section
-------------------------------------------------------------------------------

stateAddStep :: DebuggerState -> Object -> Step -> DebuggerState
stateAddStep state threadObj@(ThreadObj threadNo) step = 
  let oldProgress = take 9 (lookupSteps state threadObj) in
  state { progress = addToFM (progress state) threadNo (step:oldProgress) }
