-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiGraphicWindow Module
--
-------------------------------------------------------------------------------

-- this implements the double-listbox-window

-- needed: an init-function
--         an operate-function

-- programming TclHaskell:
--   init composes all elements together
--   operate reads and writes messages to the CHD

module CHD.GuiGraphicWindow
  (
    initDisplayWindow,
    operateDisplayWindow
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import CHD.GuiMsgChan
import CHD.DebugMsgChan
import CHD.Environment
import CHD.GuiWindow
import CHD.BaseTypes
import CHD.BaseFunctions

import Tcl


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

initDisplayWindow :: GuiWindow -> GUI () -> GUI ()
initDisplayWindow viewWin windowFinalizer = do
-- new Window
  myWindow <- window []
  title myWindow ("CHD - " ++ (show viewWin))
  bind myWindow "<Destroy>" (do interactGui (CloseGuiWin viewWin)
				windowFinalizer)

-- new Grid
  windowGrid <- frame myWindow []

-- create Elements
  selectLabel <- case viewWin of
    SourceCodeViewWindow -> label myWindow [text "SourceCodeFiles:"]
    _ -> label myWindow [text "Selection:"]
  displayLabel <- case viewWin of
    SourceCodeViewWindow -> label myWindow [text "SourceCode:"]
    _ -> label myWindow [text "Display:"]
  selectListbox <- listbox myWindow [takefocus True, 
				     background "LightGoldenRod"]
  displayEdit <- edit myWindow [active_state Disabled,
    font "-adobe-helvetica-medium-r-normal--10-100-75-75-p-56-iso8859-1",
    background "white"]
  selectScroll <- vscroll selectListbox []
  displayScroll <- vscroll displayEdit []
  let widgets = (selectListbox, displayEdit)
  displayButton <- button myWindow [text "->", 
			  command (displayAndMark widgets True)]

-- pack elements in grid
  gridAdd selectLabel (0,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd selectListbox (0,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd selectScroll (1,1) [heightY 5, gfillY, ginFrame windowGrid]
  gridAdd displayButton (2,3) [ginFrame windowGrid]
  gridAdd displayLabel (4,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd displayEdit (4,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd displayScroll (5,1) [heightY 5, gfillY, ginFrame windowGrid]

-- pack grid in window
  packAdd windowGrid [packV, fillXY]

-- get defaultColor
  defaultColor <- cget displayEdit background

-- focus on enableListbox
  focus selectListbox

-- key-bindings: arrows left/right moves element
  bind selectListbox "<Double-1>" (displayAndMark widgets True)
  bind myWindow "<Key-Right>" (displayAndMark widgets True)
  
-- telling Debugger & Gui
  interactGui (OpenGuiWin viewWin 
			  (operateSelectDisplayWindow widgets))
  proc $ interactCHD (GetInfoGuiWin viewWin)

  return ()


displayAndMark :: (Listbox, Edit) -> Bool -> GUI ()
displayAndMark (selectLB,edit) reload =
  display (selectLB,edit) (\name -> interactCHD $ PositionTag name reload)


display :: (Listbox, Edit) -> (String -> IO()) -> GUI ()
display (selectLB,edit) messageConstructor = do
  items <- getListboxSelection selectLB
  list <- foldM
    (\result num -> do
      let pos = LIndex num
      elems <- getListboxEntries selectLB pos pos
      return (elems ++ result)) 
    [] 
    items
  if list==[]
     then return ()
     else proc $ messageConstructor (last list)


markLine :: Edit -> Int -> String -> String -> GUI ()
markLine edit line bgcolor fgcolor = do
  tag edit [TIndex line 0,TIndex (line+1) 0] 
	   [background bgcolor,foreground fgcolor] 
  return ()


unMarkLine :: Edit -> Int -> GUI ()
unMarkLine edit line = do
  let tindex = [TIndex line 0,TIndex (line+1) 0]
  oldTag <- tag edit tindex [] 
  tagRemove oldTag tindex
  return ()


operateDisplayWindow :: (Listbox, Edit) -> GuiWinMsg -> GUI ()
operateDisplayWindow (selectLB,edit) msg = 
  case msg of
    InsertListbox SelectListbox list -> do
	    mapM_ (insertListboxElement (selectLB,edit)) list
    LoadEdit DisplayEdit filePath ->
	    loadEdit edit filePath
    MarkEdit DisplayEdit line bgcolor fgcolor -> 
	    markLine edit line bgcolor fgcolor
    UnmarkEdit DisplayEdit line -> 
	    after 200 (markLine edit line "white" "black") >>
	    return ()
	    -- editMoveToSee edit line
    _ -> return ()


insertListboxElement :: (Listbox, Edit) -> String -> GUI ()
insertListboxElement (selectLB,edit) elem = do
  list <- getListboxEntries selectLB (LIndex 0) LIndexEnd
  selected <- getListboxSelection selectLB
  let breaked = break (>= elem) list
      first = fst breaked
      second = snd breaked
      pos = LIndex (length first)
  if (second /= []) && ((head second) == elem)
     then displayAndMark (selectLB,edit) False
     else insertListbox selectLB pos [elem] >>
	  displayAndMark (selectLB,edit) False
  if (second /= []) && ((head second) == elem) && 
     (selected /= []) &&  ((head selected) == (length first))
     then displayAndMark (selectLB,edit) False
     else clearListboxSelection selectLB (LIndex 0) LIndexEnd >>
	  addListboxSelection selectLB pos pos >>
	  displayAndMark (selectLB,edit) True
